package org.lsst.ccs.subsystem.refrig;

/**
 ******************************************************************************
 **
 **  Simulates the refrigeration hardware.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class SimDevice extends Device {

   /**
    ***************************************************************************
    **
    **  Constants.
    **
    ***************************************************************************
    */
    final static int
        MAIN_POWER_LINE = 0,
        LOAD_POWER_LINE = 1,
        N_DEV_CHANS = 24,
        N_DEV_LINES = 2;

   /**
    ***************************************************************************
    **
    **  Data fields.
    **
    ***************************************************************************
    */
    final static double LOAD_POWER_FCTR = 0.0006667;
    ChannelSim[] chanData = new ChannelSim[N_DEV_CHANS];
    ChannelSim loadPwrChan, loadTmpChan;
    double loadFract = 0.6;
    boolean mainPowerOn, loadPowerOn;
    int lineMask; // Mask of output line numbers on this device


   /**
    ***************************************************************************
    **
    **  Inner container class for channel simulation data.
    **
    ***************************************************************************
    */
    class ChannelSim {

       /**
        ***********************************************************************
        **
        **  Data fields
        **
        ***********************************************************************
        */
        int      hwChan;      // HW channel number
        double   onValue;     // On (operating) value
        double   onTime;      // Time (secs) to reach on value
        double   offValue;    // Off (ambient) value
        double   offTime;     // Time (secs) to reach off value

        double   endValue;    // Ending value
        double   rate;        // Rate of change
        boolean  negative;    // True if nominal change is negative
        double   value;       // Current value
        long     currTime;    // Current system time


       /**
        ***********************************************************************
        **
        **  Constructor
        **
        ***********************************************************************
        */
        public ChannelSim(int hwChan, double onValue, double onTime,
                          double offValue, double offTime)
        {
            this.hwChan   = hwChan;
            this.onValue  = onValue;
            this.onTime   = onTime;
            this.offValue = offValue;
            this.offTime  = offTime;
            currTime      = System.currentTimeMillis();
        }


       /**
        ***********************************************************************
        **
        **  Initializes the channel.
        **
        ***********************************************************************
        */
        void initialize()
        {
            if (this == loadPwrChan) {
                onTime = 0;
                offValue = 0;
                offTime = 0;
            }
            endValue = offValue;
            value = offValue;
        }


       /**
        ***********************************************************************
        **
        **  Reads the channel.
        **
        ***********************************************************************
        */
        void read()
        {
            long thisTime = System.currentTimeMillis();
            double intvl = (thisTime - currTime) / 1000.0;
            currTime = thisTime;
            value += rate * intvl;
            if (negative) {
                if (value < endValue) {
                    value = endValue;
                }
            }
            else {
                if (value > endValue) {
                    value = endValue;
                }
            }
        }


       /**
        ***********************************************************************
        **
        **  Sets channel to power-on value.
        **
        ***********************************************************************
        */
        void powerOn()
        {
            currTime = System.currentTimeMillis();
            rate = 0;
            if (mainPowerOn) {
                endValue = onValue;
                if (onTime != 0) {
                    rate = (onValue - offValue) / onTime;
                }
            }
            else {
                endValue = offValue;
                if (offTime != 0) {
                    rate = (offValue - onValue) / offTime;
                }
            }
            if (rate == 0) {
                value = endValue;
            }
            negative = rate < 0;
        }

    }


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    ***************************************************************************
    */
    public SimDevice(String devName, String serialNo)
    {
    }


   /**
    ***************************************************************************
    **
    **  Checks a channel's parameters for validity.
    **
    ***************************************************************************
    */
    @Override
    int[] checkChannel(String name, int hwChan, String type, String subtype)
        throws Exception
    {
        if (hwChan < 0 || hwChan >= N_DEV_CHANS) {
            refg.reportError(name, "hw channel number", hwChan);
        }
        String flds[] = subtype.split(":");
        try {
            double onValue = Double.valueOf(flds[0]);
            double onTime = Double.valueOf(flds[1]);
            double offValue = Double.valueOf(flds[2]);
            double offTime = Double.valueOf(flds[3]);
            ChannelSim chan = new ChannelSim(hwChan, onValue, onTime,
                                             offValue, offTime);
            chanData[hwChan] = chan;
            if (name.equals("LoadPower")) {
                loadPwrChan = chan;
            }
            else if (name.equals("LoadTmp")) {
                loadTmpChan = chan;
            }
        }
        catch (Exception e) {
            refg.reportError(name, "subtype", subtype);
        }
                
        return new int[]{0, 0};
    }


   /**
    ***************************************************************************
    **
    **  Performs basic initialization.
    **
    ***************************************************************************
    */
    @Override
    void initialize()
    {
        if (loadPwrChan == null) {
            log.warn("No load power channel defined");
        }
        if (loadTmpChan == null) {
            log.warn("No load temperature channel defined");
        }
        setOnline(true);
        initSensors();
    }


   /**
    ***************************************************************************
    **
    **  Initializes a channel.
    **
    ***************************************************************************
    */
    @Override
    void initChannel(int hwChan, int type, int subtype)
    {
        chanData[hwChan].initialize();
    }


   /**
    ***************************************************************************
    **
    **  Reads a channel.
    **
    ***************************************************************************
    */
    @Override
    double readChannel(int hwChan)
    {
        ChannelSim chan = chanData[hwChan];
        chan.read();

        return chan.value;
    }


   /**
    ***************************************************************************
    **
    **  Checks an output line number.
    **
    ***************************************************************************
    */
    @Override
    void checkLine(String name, int line) throws Exception
    {
        if (line < 0 || line >= N_DEV_LINES) {
            refg.reportError(name, "line number", line);
        }
    }


   /**
    ***************************************************************************
    **
    **  Adds an output line.
    **
    ***************************************************************************
    */
    @Override
    void addLine(int line)
    {
        lineMask |= 1 << line;
    }


   /**
    ***************************************************************************
    **
    **  Sets an output line on or off.
    **
    ***************************************************************************
    */
    @Override
    void setLine(int line, boolean on)
    {
        if (line == MAIN_POWER_LINE) {
            setMainPower(on);
        }
        else if (line == LOAD_POWER_LINE) {
            setLoadPower(on);
        }
    }


   /**
    ***************************************************************************
    **
    **  Gets the set state of an output line.
    **
    ***************************************************************************
    */
    @Override
    boolean isLineSet(int line)
    {
        if (line == MAIN_POWER_LINE) return mainPowerOn;
        else if (line == LOAD_POWER_LINE) return loadPowerOn;
        else return false;
    }


   /**
    ***************************************************************************
    **
    **  Closes the connection.
    **
    ***************************************************************************
    */
    @Override
    void close()
    {
    }


   /**
    ***************************************************************************
    **
    **  Sets the load power fraction
    **
    ***************************************************************************
    */
    public void setLoadFract(double fract)
    {
        loadFract = fract;
        setLoadPower();
    }


   /**
    ***************************************************************************
    **
    **  Turns the main power on or off.
    **
    ***************************************************************************
    */
    void setMainPower(boolean on)
    {
        mainPowerOn = on;
        for (ChannelSim chan : chanData) {
            if (chan == null || chan == loadPwrChan) continue;
            chan.powerOn();
        }
        setLoadPower();
    }
            

   /**
    ***************************************************************************
    **
    **  Turns the load power on or off, setting the correct level.
    **
    ***************************************************************************
    */
    void setLoadPower(boolean on)
    {
        loadPowerOn = on;
        setLoadPower();
    }

    void setLoadPower()
    {
        if (loadPwrChan == null) return;
        double endValue = 0;
        if (mainPowerOn) {
            if (loadPowerOn) {
                endValue = loadFract * loadPwrChan.onValue;
            }
            ChannelSim tc = loadTmpChan;
            if (tc != null && tc.onTime != 0) {
                tc.rate = (tc.onValue - tc.offValue) / tc.onTime
                            + LOAD_POWER_FCTR * endValue;
            }
        }
        loadPwrChan.endValue = endValue;
        loadPwrChan.value = endValue;
    }

}
