package org.lsst.ccs.subsystem.refrig;

import java.io.IOException;
import java.util.ArrayList;
import java.util.BitSet;
import java.util.List;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;
import org.lsst.ccs.bus.ValueNotification;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.subsystem.refrig.data.RefrigChannel;
import org.lsst.ccs.subsystem.refrig.data.RefrigFullState;
import org.lsst.ccs.subsystem.refrig.data.RefrigState;
import org.lsst.ccs.subsystem.refrig.status.RefrigStateStatus;
import org.lsst.ccs.utilities.logging.Logger;

/**
 ******************************************************************************
 **
 **  Implements the refrigeration long lines test modular subsystem.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class RefrigTest extends Module {

   /**
    ***************************************************************************
    **
    **  Constants.
    **
    ***************************************************************************
    */
    final static long
        UPDATE_PERIOD = 1000,
        CHECK_PERIOD = 10000;

   /**
    ***************************************************************************
    **
    **  Data fields.
    **
    ***************************************************************************
    */
    int numChans;
    MonChannel[] chanData;
    String configName = "test";
    boolean coldStart, running;
    BitSet loLimChange = new BitSet(), hiLimChange = new BitSet(),
           goodChans = new BitSet(), onlineChans = new BitSet();
    int state;
    Map<String, Device> devcMap;
    List<Device> devcList = new ArrayList<>();
    OutputLine mainPowerLine, loadPowerLine;
    List<OutputLine> alarmLine = new ArrayList<>();


   /**
    ***************************************************************************
    **
    **  Inner class to implement a timer task to update the channel state
    **  in a timely manner.
    **
    ***************************************************************************
    */
    private class UpdateState extends TimerTask {

        @Override
        public void run()
        {
            /*
            **  Read the sensor values
            */
            readSensors();

            /*
            **  Check the limits and set the channel state
            */
            checkLimits();
        }

    }

                
   /**
    ***************************************************************************
    **
    **  Inner class to implement a timer task to check hardware devices state.
    **
    ***************************************************************************
    */
    private class CheckDevices extends TimerTask {

        @Override
        public void run()
        {
            for (Device devc : devcList) {
                devc.checkOnline();
            }
        }

    }


   /**
    ***************************************************************************
    **
    **  Main constructor.
    **
    ***************************************************************************
    */
    public RefrigTest(String name, int tickMillis, String configName)
    {
        super(name, tickMillis);
        this.configName = configName;
        String cold = System.getProperty("lsst.ccs.refrig.coldstart", "");
        coldStart = cold.equals("true");
    }


   /**
    ***************************************************************************
    **
    **  Initializes the refrigeration subsystem.
    **
    ***************************************************************************
    */
    @Override
    public void initModule()
    {
        /*
        **  Initialize all configuration data
        */
        initConfiguration();

        /*
        **  Initialize the hardware
        */
        initSensors();

        /*
        **  Set output lines
        */
        setOutputLines();
        coldStart = false;

        /*
        **  Start the timer task which updates the channel state periodically
        */
        (new Timer(true)).schedule(new UpdateState(), 0L, UPDATE_PERIOD);
    }


   /**
    ***************************************************************************
    **
    **  Performs periodic trending data broadcast.
    **
    ***************************************************************************
    */
    @Override
    public void tick()
    {
        /*
        **  Broadcast the state if first time
        */
        if (!running) {
            System.out.println("Refrigeration test system started");
            publishState();    // For any GUIs
            publishLimits();   // For GUIs and the trending database
            running = true;
        }

        /*
        **  Broadcast the trending data
        */
        long timeStamp = System.currentTimeMillis();
        List<ValueNotification> trendingValues = new ArrayList<>();
        for (MonChannel ch : chanData) {
            trendingValues.add(new ValueNotification(ch.getName(), ch.value,
                                                     timeStamp));
        }
        trendingValues.add(new ValueNotification("state", getState(),
                                                 timeStamp));
        publishData(trendingValues);
    }


   /**
    ***************************************************************************
    **
    **  Sets the tick period.
    **
    **  Overrides the method in Module in order to publish a status update.
    **
    **  @param  value  The tick period (milliseconds) to set.
    **
    ***************************************************************************
    */
    @Override
    public void setTickMillis(int value)
    {
        super.setTickMillis(value);
        if (running) {
            publishState();
        }
    }


   /**
    ***************************************************************************
    **
    **  Sets the main power enabled state on or off.
    **
    **  @param  value  The enabled state value to set: 0 = off, ~0 = on.
    **
    ***************************************************************************
    */
    //@BusCommand
    public void setMainPowerEnable(int value)
    {
        setMainPowerEnable(value, true);
    }


   /**
    ***************************************************************************
    **
    **  Sets the main power enabled state on or off.
    **
    **  @param  value    The enabled state value to set: 0 = off, ~0 = on.
    **
    **  @param  publish  If true, publish the new state on the status bus.
    **
    ***************************************************************************
    */
    public void setMainPowerEnable(int value, boolean publish)
    {
        boolean on = false;
        if (value != 0) {
            if ((state & RefrigState.MAIN_TRIPPED_STATE) == 0) {
                state |= RefrigState.MAIN_POWER_STATE;
                on = true;
            }
        }
        else {
            state &= ~RefrigState.MAIN_POWER_STATE;
        }
        setMainPower(on);
        if (publish) {
            publishState();
        }
    }


   /**
    ***************************************************************************
    **
    **  Sets the load power trip enabled state on or off.
    **
    **  @param  value  The load trip enabled state value to set: 0 = off;
    **                 ~0 = on.
    **
    ***************************************************************************
    */
    //@BusCommand
    public void setLoadTripEnable(int value)
    {
        if (value != 0) {
            state |= RefrigState.LOAD_TRIP_ENAB_STATE;
        }
        else {
            state &= ~RefrigState.LOAD_TRIP_ENAB_STATE;
        }
        publishState();
    }


   /**
    ***************************************************************************
    **
    **  Saves the configuration data.
    **
    ***************************************************************************
    */
    //@BusCommand
    public void saveConfiguration() throws IOException
    {
        register(configName, "");
        loLimChange.clear();
        hiLimChange.clear();
        publishState();         // Must always do this
    }


   /**
    ***************************************************************************
    **
    **  Gets the full state of the refrigeration module.
    **
    **  This is intended to be called by GUIs during initialization
    **
    ***************************************************************************
    */
    //@BusCommand
    public RefrigFullState getFullState()
    {
        RefrigFullState status = new RefrigFullState();
        for (MonChannel ch : chanData) {
            status.addChannel(new RefrigChannel(ch.getName(), ch.description,
                                                ch.units, ch.limitLo,
                                                ch.limitHi, ch.value));
        }
        status.setRefrigState(new RefrigState(state, goodChans, onlineChans,
                                              loLimChange, hiLimChange,
                                              getTickMillis()));
        return status;
    }    


   /**
    ***************************************************************************
    **
    **  Sets the output lines from the current state.
    **
    ***************************************************************************
    */
    void setOutputLines()
    {
        /*
        **  Initialize main power state from/to hardware
        */
        setMainPowerEnable((!coldStart && isMainPowerOn()) ? 1 : 0, running);

        /*
        **  Set the load power state
        */
        setLoadPower((state & RefrigState.LOAD_POWER_STATE) != 0);

        /*
        **  Set the display
        */
        setDisplay(null, goodChans);
    }


   /**
    ***************************************************************************
    **
    **  Gets the operating state word.
    **
    **  @return  The current value of the operating state
    **
    ***************************************************************************
    */
    int getState()
    {
        return state;
    }


   /**
    ***************************************************************************
    **
    **  Publishes the state of the refrigeration module.
    **
    **  This is intended to be called whenever any element of the state is
    **  changed.
    **
    ***************************************************************************
    */
    void publishState()
    {
        RefrigState refrigState = new RefrigState(state, goodChans, onlineChans,
                                                  loLimChange, hiLimChange,
                                                  getTickMillis());
        sendToStatus(new RefrigStateStatus(refrigState));
    }    


   /**
    ***************************************************************************
    **
    **  Publishes all the limit values of the refrigeration module.
    **
    **  This is intended to be called at startup time.
    **
    ***************************************************************************
    */
    void publishLimits()
    {
        for (MonChannel ch : chanData) {
            getSubsystem().publishMetaData(ch.getName(), "alarmLow",
                                           String.valueOf(ch.limitLo));
            getSubsystem().publishMetaData(ch.getName(), "alarmHigh",
                                           String.valueOf(ch.limitHi));
        }
    }    


   /**
    ***************************************************************************
    **
    **  Checks whether all current channel values are within limits.
    **
    ***************************************************************************
    */
    void checkLimits()
    {
        /*
        **  Generate the current channel state
        */
        int sState = (state | RefrigState.LOAD_POWER_STATE)
                       & ~RefrigState.MAIN_TRIPPED_STATE;
        BitSet chanState = new BitSet(numChans);
        for (MonChannel ch : chanData) {
            sState = ch.checkLimits(sState, chanState);
        }
                
        /*
        **  If state changed
        **    Turn main power off if it was tripped off
        **    Turn load power off or on appropriately
        **    Set alarms to new values
        **    If running
        **      Publish state
        **      Log channel data
        */
        boolean update = false;
        sState ^= state;
        state ^= sState;
        if (sState != 0) {
            if ((sState & RefrigState.MAIN_TRIPPED_STATE) != 0) {
                if ((state & RefrigState.MAIN_TRIPPED_STATE) != 0) {
                    setMainPowerEnable(0, false);
                }
            }
            if ((sState & RefrigState.LOAD_POWER_STATE) != 0) {
                setLoadPower((state & RefrigState.LOAD_POWER_STATE) != 0);
            }
            update = true;
        }
        chanState.xor(goodChans);
        goodChans.xor(chanState);
        if (!chanState.isEmpty()) {
            setDisplay(chanState, goodChans);
            update = true;
        }
        if (update && running) {
            publishState();
            tick();
        }
    }


   /**
    ***************************************************************************
    **
    **  Finds the ID of a channel given its name.
    **
    ***************************************************************************
    */
    int findChannelId(String name)
    {
        for (MonChannel ch : chanData) {
            if (ch.getName().equals(name)) return ch.id;
        }

        return -1;
    }


   /**
    ***************************************************************************
    **
    **  Gets a channel given its ID.
    **
    ***************************************************************************
    */
    MonChannel getChannel(int id)
    {
        return chanData[id];
    }


   /**
    ***************************************************************************
    **
    **  Sets channels on- or off-line.
    **
    ***************************************************************************
    */
    void setOnline(BitSet mask, boolean online)
    {
        BitSet prevOnline = (BitSet)onlineChans.clone();
        if (online) {
            onlineChans.or(mask);
        }
        else {
            onlineChans.andNot(mask);
        }
        if (!onlineChans.equals(prevOnline)) {
            publishState();
        }
    }


   /**
    ***************************************************************************
    **
    **  Gets the number of channels.
    **
    ***************************************************************************
    */
    int getNumChans()
    {
        return numChans;
    }


   /**
    ***************************************************************************
    **
    **  Gets the logger.
    **
    ***************************************************************************
    */
    Logger getLogger()
    {
        return log;
    }


   /**
    ***************************************************************************
    **
    **  Reports parameter errors.
    **
    ***************************************************************************
    */
    void reportError(String cName, String pName, Object pValue)
        throws Exception
    {
        log.error("Invalid " + pName + " (" + pValue + ") for " + cName);
        throw new Exception();
    }


   /**
    ***************************************************************************
    **
    **  Configures the devices and channels with description data.
    **
    ***************************************************************************
    */
    void initConfiguration()
    {
        devcMap = getChildren(Device.class);
        for (Device devc : devcMap.values()) {
            devcList.add(devc);
            devc.configure(this);
        }

        int id = 0;
        for (OutputLine line : getChildren(OutputLine.class).values()) {
            line.configure(this, id++);
            if (line.type == OutputLine.TYPE_MAIN_POWER) {
                mainPowerLine = line;
            }
            else if (line.type == OutputLine.TYPE_LOAD_POWER) {
                loadPowerLine = line;
            }
            else if (line.type == OutputLine.TYPE_ALARM) {
                alarmLine.add(line);
            }
        }

        Map<String, MonChannel> chanMap = getChildren(MonChannel.class);
        chanData = new MonChannel[chanMap.size()];
        id = 0;
        for (MonChannel ch : chanMap.values()) {
            ch.configure(this, id);
            chanData[id++] = ch;
        }
        for (MonChannel ch : chanData) {
            ch.configDerived();
        }
    }


   /**
    ***************************************************************************
    **
    **  Initializes the refrigeration sensors.
    **
    ***************************************************************************
    */
    void initSensors()
    {
        /*
        **  Perform full initialization of all devices
        */
        for (Device devc : devcList) {
            devc.initialize();
        }

        /*
        **  Start the timer task which checks the device state periodically
        */
        (new Timer(true)).schedule(new CheckDevices(), 0L, CHECK_PERIOD);
    }


   /**
    ***************************************************************************
    **
    **  Reads the sensor data.
    **
    ***************************************************************************
    */
    void readSensors()
    {
        /*
        **  Read all the hardware channels
        */
        for (MonChannel ch : chanData) {
            ch.readSensor();
        }

        /*
        **  Compute derived values
        */
        for (MonChannel ch : chanData) {
            ch.calcDerived();
        }
    }


   /**
    ***************************************************************************
    **
    **  Turns the main power on or off.
    **
    ***************************************************************************
    */
    void setMainPower(boolean on)
    {
        if (mainPowerLine == null) return;
        mainPowerLine.set(on);
    }
            

   /**
    ***************************************************************************
    **
    **  Turns the load power on or off.
    **
    ***************************************************************************
    */
    void setLoadPower(boolean on)
    {
        if (loadPowerLine == null) return;
        loadPowerLine.set(on);
    }


   /**
    ***************************************************************************
    **
    **  Gets the hardware main power state
    **
    ***************************************************************************
    */
    boolean isMainPowerOn()
    {
        boolean on = false;
        if (mainPowerLine != null) {
            on = mainPowerLine.isSet();
        }

        return on;
    }
            

   /**
    ***************************************************************************
    **
    **  Gets the hardware load power state
    **
    ***************************************************************************
    */
    boolean isLoadPowerOn()
    {
        boolean on = false;
        if (loadPowerLine != null) {
            on = loadPowerLine.isSet();
        }

        return on;
    }
            

   /**
    ***************************************************************************
    **
    **  Sets the front-panel channel state lights.
    **
    ***************************************************************************
    */
    void setDisplay(BitSet mask, BitSet state)
    {
        for (MonChannel ch : chanData) {
            if (ch.alarm < 0 || (mask != null && !mask.get(ch.id))) continue;
            alarmLine.get(ch.alarm).set(state.get(ch.id));
        }
    }


   /**
    ***************************************************************************
    **
    **  Turns an alarm on or off.
    **
    ***************************************************************************
    */
    void setAlarm(int alarm, boolean on)
    {
        if (alarm < 0 || alarm >= alarmLine.size()) return;
        alarmLine.get(alarm).set(on);
    }


   /**
    ***************************************************************************
    **
    **  Checks an alarm number for validity
    **
    ***************************************************************************
    */
    boolean checkAlarm(int alarm)
    {
        return alarm < alarmLine.size();
    }


   /**
    ***************************************************************************
    **
    **  Gets the device with the given name.
    **
    ***************************************************************************
    */
    Device getDevice(String name)
    {
        return devcMap.get(name);
    }
    
}
