package org.lsst.ccs.subsystem.refrig;

import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.framework.ConfigurableComponent;
import org.lsst.ccs.utilities.logging.Logger;

/**
 ***************************************************************************
 **
 **  Implements hardware output lines.
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class OutputLine extends ConfigurableComponent {

   /**
    ***********************************************************************
    **
    **  Constants.
    **
    ***********************************************************************
    */
    final static int
        TYPE_UNKNOWN    = -1,
        TYPE_MAIN_POWER = 0,
        TYPE_LOAD_POWER = 1,
        TYPE_ALARM      = 2;

   /**
    ***********************************************************************
    **
    **  Data fields.
    **
    ***********************************************************************
    */
    // Supplied immutable fields
    String    typeS;       // Line type string (MAIN, LOAD, ALARM)
    String    devcName;    // Device name
    int       chan;        // HW channel number
    String    devcNameC;   // Complementary device name
    int       chanC;       // Complementary HW channel number

    // Derived immutable fields
    int       type;        // Encoded type
    Device    device;      // Device for this line
    Device    deviceC;     // Device for the complementary line
    Logger    log;         // The logger

   /**
    ***********************************************************************
    **
    **  Private lookup maps.
    **
    ***********************************************************************
    */
    private final static Map<String,Integer> typeMap = new HashMap<>();
    static {
        typeMap.put("MAIN",  TYPE_MAIN_POWER);
        typeMap.put("LOAD",  TYPE_LOAD_POWER);
        typeMap.put("ALARM", TYPE_ALARM);
    }


   /**
    ***********************************************************************
    **
    **  Constructor.
    **
    ***********************************************************************
    */
    public OutputLine(String type, String devcName, int chan,
                      String devcNameC, int chanC)
    {
        this.typeS     = type;
        this.devcName  = devcName;
        this.chan      = chan;
        this.devcNameC = devcNameC;
        this.chanC     = chanC;
    }


   /**
    ***********************************************************************
    **
    **  Configures the line.
    **
    ***********************************************************************
    */
    void configure(RefrigTest refg, int id)
    {
        try {
            Integer iType = typeMap.get(typeS);
            if (iType == null) {
                refg.reportError(getName(), "type", typeS);
            }
            type = iType;
            if (devcName != null) {
                device = refg.getDevice(devcName);
                if (device == null) {
                    refg.reportError(getName(), "device", devcName);
                }
                device.checkLine(getName(), chan);
                device.addLine(id);
            }
            if (devcNameC != null) {
                deviceC = refg.getDevice(devcNameC);
                if (deviceC == null) {
                    refg.reportError(getName(), "device", devcNameC);
                }
                deviceC.checkLine(getName(), chanC);
                deviceC.addLine(id);
            }
        }
        catch(Exception e) {
        }
    }


   /**
    ***********************************************************************
    **
    **  Sets the line value.
    **
    ***********************************************************************
    */
    void set(boolean on)
    {
        if (device != null && device.online) {
            device.setLine(chan, on);
        }
        if (deviceC != null && deviceC.online) {
            deviceC.setLine(chanC, !on);
        }
    }


   /**
    ***********************************************************************
    **
    **  Gets the line value.
    **
    ***********************************************************************
    */
    boolean isSet()
    {
        boolean state = false;
        if (device != null && device.online) {
            state = device.isLineSet(chan);
        }
        else if (deviceC != null && deviceC.online) {
            state = !deviceC.isLineSet(chanC);
        }

        return state;
    }

}
