package org.lsst.ccs.subsystem.refrig;

import java.util.BitSet;
import java.util.Timer;
import java.util.TimerTask;
import org.lsst.ccs.subsystem.refrig.data.RefrigState;

/**
 ******************************************************************************
 **
 **  Implements monitoring.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public abstract class Monitor {

   /**
    ***************************************************************************
    **
    **  Constants.
    **
    ***************************************************************************
    */
    final static long
        UPDATE_PERIOD = 1000;

   /**
    ***************************************************************************
    **
    **  Data fields.
    **
    ***************************************************************************
    */
    int numChans;
    MonChannel[] chanData;
    BitSet loLimChange = new BitSet(), hiLimChange = new BitSet(),
           goodChans = new BitSet(), onlineChans = new BitSet();
    int state;


   /**
    ***************************************************************************
    **
    **  Inner class to implement a timer task to update the channel state
    **  in a timely manner.
    **
    ***************************************************************************
    */
    private class UpdateState extends TimerTask {

        @Override
        public void run()
        {
            /*
            **  Read the sensor values
            */
            readSensors();

            /*
            **  Check the limits and set the channel state
            */
            checkLimits();
        }

    }

                
   /**
    ***************************************************************************
    **
    **  Starts monitoring.
    **
    ***************************************************************************
    */
    public void start()
    {
        (new Timer(true)).schedule(new UpdateState(), 0L, UPDATE_PERIOD);
    }


   /**
    ***************************************************************************
    **
    **  Publishes all the limit values of the refrigeration module.
    **
    **  This is intended to be called at startup time.
    **
    ***************************************************************************
    */
    void publishLimits()
    {
        for (MonChannel ch : chanData) {
            /*
            getSubsystem().publishMetaData(ch.getName(), "alarmLow",
                                           String.valueOf(ch.limitLo));
            getSubsystem().publishMetaData(ch.getName(), "alarmHigh",
                                           String.valueOf(ch.limitHi));
            */
        }
    }    


   /**
    ***************************************************************************
    **
    **  Checks whether all current channel values are within limits.
    **
    ***************************************************************************
    */
    void checkLimits()
    {
        /*
        **  Generate the current channel state
        */
        int sState = (state | RefrigState.LOAD_POWER_STATE)
                       & ~RefrigState.MAIN_TRIPPED_STATE;
        BitSet chanState = new BitSet(numChans);
        for (MonChannel ch : chanData) {
            sState = ch.checkLimits(sState, chanState);
        }
                
        /*
        **  If state changed
        **    Turn main power off if it was tripped off
        **    Turn load power off or on appropriately
        **    Set alarms to new values
        **    If running
        **      Publish state
        **      Log channel data
        */
        boolean update = false;
        sState ^= state;
        state ^= sState;
        if (sState != 0) {
            if ((sState & RefrigState.MAIN_TRIPPED_STATE) != 0) {
                if ((state & RefrigState.MAIN_TRIPPED_STATE) != 0) {
                    // setMainPowerEnable(0, false);
                }
            }
            if ((sState & RefrigState.LOAD_POWER_STATE) != 0) {
                // setLoadPower((state & RefrigState.LOAD_POWER_STATE) != 0);
            }
            update = true;
        }
        chanState.xor(goodChans);
        goodChans.xor(chanState);
        if (!chanState.isEmpty()) {
            // setDisplay(chanState, goodChans);
            update = true;
        }
    }


   /**
    ***************************************************************************
    **
    **  Finds the ID of a channel given its name.
    **
    ***************************************************************************
    */
    int findChannelId(String name)
    {
        for (MonChannel ch : chanData) {
            if (ch.getName().equals(name)) return ch.id;
        }

        return -1;
    }


   /**
    ***************************************************************************
    **
    **  Gets a channel given its ID.
    **
    ***************************************************************************
    */
    MonChannel getChannel(int id)
    {
        return chanData[id];
    }


   /**
    ***************************************************************************
    **
    **  Sets channels on- or off-line.
    **
    ***************************************************************************
    */
    void setOnline(BitSet mask, boolean online)
    {
        BitSet prevOnline = (BitSet)onlineChans.clone();
        if (online) {
            onlineChans.or(mask);
        }
        else {
            onlineChans.andNot(mask);
        }
        if (!onlineChans.equals(prevOnline)) {
            // publishState();
        }
    }


   /**
    ***************************************************************************
    **
    **  Gets the number of channels.
    **
    ***************************************************************************
    */
    int getNumChans()
    {
        return numChans;
    }


   /**
    ***************************************************************************
    **
    **  Reports parameter errors.
    **
    ***************************************************************************
    */
    void reportError(String cName, String pName, Object pValue)
        throws Exception
    {
        // log.error("Invalid " + pName + " (" + pValue + ") for " + cName);
        throw new Exception();
    }


   /**
    ***************************************************************************
    **
    **  Initializes the refrigeration configuration.
    **
    ***************************************************************************
    */
    abstract void initConfiguration();


   /**
    ***************************************************************************
    **
    **  Initializes the refrigeration sensors.
    **
    ***************************************************************************
    */
    abstract void initSensors();


   /**
    ***************************************************************************
    **
    **  Reads the sensor data.
    **
    ***************************************************************************
    */
    abstract void readSensors();
    
}
