package org.lsst.ccs.subsystem.refrig;

import java.util.BitSet;
import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.framework.ConfigurableComponent;
import org.lsst.ccs.framework.annotations.ConfigChanger;
import org.lsst.ccs.subsystem.refrig.data.RefrigState;
import org.lsst.ccs.utilities.logging.Logger;

/**
 ******************************************************************************
 **
 **  Handles a monitoring channel.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class MonChannel extends ConfigurableComponent {

   /**
    ***************************************************************************
    **
    **  Constants.
    **
    ***************************************************************************
    */
    final static int
        TYPE_UNKNOWN = -1,
        TYPE_TEMP    = 0,
        TYPE_VOLTS   = 1,
        TYPE_POWER   = 2,
        TYPE_SWITCH  = 3,
        TYPE_DIFF    = 4;

    final static int
        LIMIT_CHECK_NONE      = 0,   // Don't check limit
        LIMIT_CHECK_FLAG      = 1,   // Flag if limit exceeded
        LIMIT_CHECK_MAIN_TRIP = 2,   // Trip main power if limit exceeded
        LIMIT_CHECK_LOAD_TRIP = 3;   // Trip load power if limit exceeded

   /**
    ***************************************************************************
    **
    **  Private lookup maps.
    **
    ***************************************************************************
    */
    private final static Map<String,Integer> checkMap = new HashMap<>();
    static {
        checkMap.put("NONE", LIMIT_CHECK_NONE);
        checkMap.put("FLAG", LIMIT_CHECK_FLAG);
        checkMap.put("MTRIP", LIMIT_CHECK_MAIN_TRIP);
        checkMap.put("LTRIP", LIMIT_CHECK_LOAD_TRIP);
    }

   /**
    ***************************************************************************
    **
    **  Data fields.
    **
    ***************************************************************************
    */
    // Supplied immutable fields
    String     description;   // Channel description
    String     units;         // Units name
    String     loCheckS;      // Low limit check option string
    String     hiCheckS;      // High limit check option string
    double     deadbandLo;    // Low limit dead-band
    double     deadbandHi;    // High limit dead-band
    String     devcName;      // The name of the associated device
    int        hwChan;        // Hardware channel
    String     typeS;         // Channel type string (TEMP, VOLTS, POWER...)
    String     subtypeS;      // Channel subtype string (extra config info)
    int        alarm;         // Hardware alarm number
    double     offset;        // Value offset
    double     scale;         // Value scale

    // Supplied configurable fields
    double     limitLo;       // Low limit value
    double     limitHi;       // High limit value

    // Derived immutable fields
    int        loCheck;       // Low limit check option
    int        hiCheck;       // High limit check option
    Device     devc;          // The associated device
    int        type;          // Encoded channel type
    int        subtype;       // Encoded channel subtype
    int        id;            // External ID
    RefrigTest refg;          // Associated refrigeration object
    Logger     log;           // The logger

    // Volatile fields
    double     value;         // Current value read from channel
    boolean    trippedLo;     // Whether low limit has caused a trip
    boolean    trippedHi;     // Whether high limit has caused a trip



   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    ***************************************************************************
    */
    public MonChannel(String desc, String units,
                      String loCheck, double limitLo, double deadbandLo,
                      String hiCheck, double limitHi, double deadbandHi,
                      String devcName, int hwChan, String type,
                      String subtype, int alarm, double offset, double scale)
    {
        this.description = desc;
        this.units       = units;
        this.loCheckS    = loCheck;
        this.hiCheckS    = hiCheck;
        this.limitLo     = limitLo;
        this.deadbandLo  = deadbandLo;
        this.limitHi     = limitHi;
        this.deadbandHi  = deadbandHi;
        this.devcName    = devcName;
        this.hwChan      = hwChan;
        this.typeS       = type;
        this.subtypeS    = subtype;
        this.alarm       = alarm;
        this.offset      = offset;
        this.scale       = scale;
    }


   /**
    ***************************************************************************
    **
    **  Configures channel description.
    **
    ***************************************************************************
    */
    void configure(RefrigTest refg, int id)
    {
        this.refg = refg;
        this.id = id;
        log = refg.getLogger();

        int iLoCheck = LIMIT_CHECK_FLAG, iHiCheck = LIMIT_CHECK_FLAG;
        int iType = TYPE_UNKNOWN, iSubtype = -1, iChan = -1, iAlarm = -1;
        try {
            Integer iCheck = checkMap.get(loCheckS.toUpperCase());
            if (iCheck == null) {
                refg.reportError(getName(), "locheck", loCheckS);
            }
            iLoCheck = iCheck;
            iCheck = checkMap.get(hiCheckS.toUpperCase());
            if (iCheck == null) {
                refg.reportError(getName(), "hicheck", hiCheckS);
            }
            iHiCheck = iCheck;

            devc = refg.getDevice(devcName);
            if (devc == null) {
                refg.reportError(getName(), "device", devcName);
            }

            int[] types = devc.checkChannel(getName(), hwChan, typeS, subtypeS);
            iChan = hwChan;
            iType = types[0];
            iSubtype = types[1];

            devc.addChannel(id);

            if (!refg.checkAlarm(alarm)) {
                refg.reportError(getName(), "alarm number", alarm);
            }
            iAlarm = alarm;
        }
        catch (Exception e) {
        }

        loCheck = iLoCheck;
        hiCheck = iHiCheck;
        type    = iType;
        subtype = iSubtype;
        hwChan  = iChan;
        alarm   = iAlarm;
    }


   /**
    ***************************************************************************
    **
    **  Configures derived channel data.
    **
    ***************************************************************************
    */
    void configDerived()
    {
        if (type != TYPE_DIFF) return;
        String[] names = subtypeS.split(":", -1);
        boolean okay = false;
        if (names.length == 2) {
            int id0 = refg.findChannelId(names[0]);
            int id1 = refg.findChannelId(names[1]);
            if (id0 >= 0 && id1 >= 0) {
                subtype = (id0 << 16) | id1;
                okay = true;
                BitSet mask = new BitSet();
                mask.set(id);
                refg.setOnline(mask, true);
            }
        }
        if (!okay) {
            log.error("Invalid subtype (" + subtypeS + ") for "
                        + getName());
            type = TYPE_UNKNOWN;
        }
    }


   /**
    ***************************************************************************
    **
    **  Initializes the hardware sensor.
    **
    ***************************************************************************
    */
    void initSensor()
    {
        if (devc != null) {
            devc.initChannel(hwChan, type, subtype);
        }
    }


   /**
    ***************************************************************************
    **
    **  Reads the hardware sensor.
    **
    ***************************************************************************
    */
    void readSensor()
    {
        if (devc != null) {
            value = offset + scale * devc.readChannel(hwChan);
        }
    }


   /**
    ***************************************************************************
    **
    **  Calculates a derived value.
    **
    ***************************************************************************
    */
    void calcDerived()
    {
        if (type != TYPE_DIFF) return;
        value = refg.getChannel(subtype >> 16).value
                  - refg.getChannel(subtype & 0xffff).value;
    }


   /**
    ***************************************************************************
    **
    **  Sets the low limit.
    **
    ***************************************************************************
    */
    @ConfigChanger
    public void setLimitLo(double limit)
    {
        if (limit == limitLo) return;
        limitLo = limit;
        refg.loLimChange.set(id);
        refg.publishState();
        refg.getSubsystem().publishMetaData(getName(), "alarmLow",
                                            String.valueOf(limit));
    }


   /**
    ***************************************************************************
    **
    **  Sets the high limit.
    **
    ***************************************************************************
    */
    @ConfigChanger
    public void setLimitHi(double limit)
    {
        if (limit == limitHi) return;
        limitHi = limit;
        refg.hiLimChange.set(id);
        refg.publishState();
        refg.getSubsystem().publishMetaData(getName(), "alarmHigh",
                                            String.valueOf(limit));
    }


   /**
    ***************************************************************************
    **
    **  Checks whether current value is within limits.
    **
    ***************************************************************************
    */
    int checkLimits(int sState, BitSet cState)
    {
        boolean ok = false;
        if (value < limitLo) {
            if (loCheck == LIMIT_CHECK_NONE) {
                ok = true;
            }
            else if (loCheck == LIMIT_CHECK_MAIN_TRIP) {
                sState |= RefrigState.MAIN_TRIPPED_STATE;
            }
            else if (loCheck == LIMIT_CHECK_LOAD_TRIP) {
                if ((sState & RefrigState.LOAD_TRIP_ENAB_STATE) != 0) {
                    sState &= ~RefrigState.LOAD_POWER_STATE;
                    trippedLo = true;
                }
            }
        }
        else if (value > limitHi) {
            if (hiCheck == LIMIT_CHECK_NONE) {
                ok = true;
            }
            else if (hiCheck == LIMIT_CHECK_MAIN_TRIP) {
                sState |= RefrigState.MAIN_TRIPPED_STATE;
            }
            else if (hiCheck == LIMIT_CHECK_LOAD_TRIP) {
                if ((sState & RefrigState.LOAD_TRIP_ENAB_STATE) != 0) {
                    sState &= ~RefrigState.LOAD_POWER_STATE;
                    trippedHi = true;
                }
            }
        }
        else {
            ok = true;
            if (loCheck == LIMIT_CHECK_LOAD_TRIP) {
                boolean tripped = false;
                if ((sState & RefrigState.LOAD_TRIP_ENAB_STATE) != 0) {
                    if (trippedLo && value < limitLo + deadbandLo) {
                        sState &= ~RefrigState.LOAD_POWER_STATE;
                        tripped = true;
                    }
                }
                trippedLo = tripped;
            }
            if (hiCheck == LIMIT_CHECK_LOAD_TRIP) {
                boolean tripped = false;
                if ((sState & RefrigState.LOAD_TRIP_ENAB_STATE) != 0) {
                    if (trippedHi && value > limitHi - deadbandHi) {
                        sState &= ~RefrigState.LOAD_POWER_STATE;
                        tripped = true;
                    }
                }
                trippedHi = tripped;
            }
        }
        if (ok) {
            cState.set(id);
        }

        return sState;
    }

}
