package org.lsst.ccs.subsystem.refrig;

import java.util.HashMap;
import java.util.Map;
import javax.usb.UsbDisconnectedException;
import javax.usb.UsbEncodingException;
import javax.usb.UsbException;
import javax.usb.UsbNotActiveException;
import javax.usb.UsbNotClaimedException;
import javax.usb.UsbNotOpenException;
import org.lsst.ccs.drivers.mcc.MccUsb;

/**
 ******************************************************************************
 **
 **  Handles a Measurement Computing DAQ box.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class MCCDevice extends Device {

   /**
    ***************************************************************************
    **
    **  Constants.
    **
    ***************************************************************************
    */
    final static int
        N_DEV_CHANS = 8;
    final static float[]
        RTD_COEFFS = {100.0F, 0.003908F, -5.8019E-7F, -4.2735E-12F};

   /**
    ***************************************************************************
    **
    **  Private lookup maps.
    **
    ***************************************************************************
    */
    private final static Map<String,Integer> didMap = new HashMap<>();
    static {
        didMap.put("TC_AI", MccUsb.USB_TC_AI_DID);
        didMap.put("TC",    MccUsb.USB_TC_DID);
        didMap.put("TEMP",  MccUsb.USB_TEMP_DID);
    }

    private final static Map<String,Integer> typeMap = new HashMap<>();
    static {
        typeMap.put("TEMP",   MonChannel.TYPE_TEMP);
        typeMap.put("VOLTS",  MonChannel.TYPE_VOLTS);
        typeMap.put("SWITCH", MonChannel.TYPE_SWITCH);
    }

    private final static Map<String,Integer> tSbTypMap = new HashMap<>();
    static {
        tSbTypMap.put("TC",  MccUsb.STP_THERMOCOUPLE);
        tSbTypMap.put("RTD", MccUsb.STP_RTD);
    }

    private final static Map<String,Integer> tcSbTypMap = new HashMap<>();
    static {
        tcSbTypMap.put("J", MccUsb.TC_TYPE_J);
        tcSbTypMap.put("K", MccUsb.TC_TYPE_K);
        tcSbTypMap.put("T", MccUsb.TC_TYPE_T);
        tcSbTypMap.put("E", MccUsb.TC_TYPE_E);
        tcSbTypMap.put("R", MccUsb.TC_TYPE_R);
        tcSbTypMap.put("S", MccUsb.TC_TYPE_S);
        tcSbTypMap.put("B", MccUsb.TC_TYPE_B);
        tcSbTypMap.put("N", MccUsb.TC_TYPE_N);
    }

    private final static Map<String,Integer> rtSbTypMap = new HashMap<>();
    static {
        rtSbTypMap.put("2WIRE1", MccUsb.TCT_2WIRE_1SENSOR);
        rtSbTypMap.put("2WIRE2", MccUsb.TCT_2WIRE_2SENSOR);
        rtSbTypMap.put("3WIRE",  MccUsb.TCT_3WIRE);
        rtSbTypMap.put("4WIRE",  MccUsb.TCT_4WIRE);
    }

    private final static Map<String,Integer> vSbTypMap = new HashMap<>();
    static {
        vSbTypMap.put("10V",   MccUsb.RANGE_10V);
        vSbTypMap.put("5V",    MccUsb.RANGE_5V);
        vSbTypMap.put("2.5V",  MccUsb.RANGE_2_5V);
        vSbTypMap.put("1.25V", MccUsb.RANGE_1_25V);
    }


   /**
    ***************************************************************************
    **
    **  Data fields.
    **
    ***************************************************************************
    */
    String  dName;    // MCC device name
    String  serial;   // USB serial number
    int     did;      // MCC device ID
    MccUsb  mcc;      // Associated USB object
    int     tcMask;   // Mask of available thermocouple channels
    int     rtMask;   // Mask of available RT channels
    int     vMask;    // Mask of available voltage channels
    int     lineMask; // Mask of output line numbers on this device


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    ***************************************************************************
    */
    public MCCDevice(String devName, String serialNo)
    {
        type = TYPE_MCC;
        dName = devName;
        serial = serialNo;
        fullName = "MCC " + dName + " module"
                     + (serial == null ? "" : " (" + serial + ")");
        Integer iDid = didMap.get(dName);
        if (iDid == null) {
            log.error("Invalid MCC device type (" + dName + ")");
            did = -1;
        }
        else {
            did = iDid;
            if (did == MccUsb.USB_TC_AI_DID) {
                tcMask = 0x0f;
                vMask = 0xf0;
            }
            else if (did == MccUsb.USB_TC_DID) {
                tcMask = 0xff;
            }
            else if (did == MccUsb.USB_TEMP_DID) {
                tcMask = 0xff;
                rtMask = 0xff;
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Checks a channel's parameters for validity.
    **
    ***************************************************************************
    */
    @Override
    int[] checkChannel(String name, int hwChan, String type, String subtype)
        throws Exception
    {
        Integer iSubtype = 0, iType = typeMap.get(type.toUpperCase());
        if (iType == null) {
            refg.reportError(name, "type", type);
        }

        String sbtype = subtype.toUpperCase();
        if (iType == MonChannel.TYPE_TEMP) {
            String[] sbtypes = sbtype.split(":", -1);
            if (sbtypes.length < 2) {
                refg.reportError(name, "subtype", subtype);
            }
            iSubtype = tSbTypMap.get(sbtypes[0]);
            if (iSubtype == null) {
                refg.reportError(name, "subtype", subtype);
            }
            Integer iStp = (iSubtype == MccUsb.STP_RTD)
                             ? rtSbTypMap.get(sbtypes[1])
                             : tcSbTypMap.get(sbtypes[1]);
            if (iStp == null) {
                refg.reportError(name, "subtype", subtype);
            }
            iSubtype |= (iStp << 8);
        }
        else if (iType == MonChannel.TYPE_VOLTS) {
            iSubtype = vSbTypMap.get(sbtype);
            if (iSubtype == null) {
                refg.reportError(name, "subtype", subtype);
            }
        }

        boolean chanOk = true;
        if (iType == MonChannel.TYPE_TEMP) {
            int mask = (iSubtype & 0xff) == MccUsb.STP_RTD ? rtMask : tcMask;
            if ((1 << hwChan & mask) == 0) {
                chanOk = false;
            }
        }
        else if (iType == MonChannel.TYPE_VOLTS) {
            if ((1 << hwChan & vMask) == 0) {
                chanOk = false;
            }
        }
        else {
            if (hwChan < 0 || hwChan >= N_DEV_CHANS) {
                chanOk = false;
            }
        }
        if (!chanOk) {
            refg.reportError(name, "hw channel number", hwChan);
        }

        return new int[]{iType, iSubtype};
    }


   /**
    ***************************************************************************
    **
    **  Performs basic initialization.
    **
    ***************************************************************************
    */
    @Override
    void initialize()
    {
        if (did < 0) return;
        try {
            if (!inited || mcc == null) {
                mcc = new MccUsb();
            }
            mcc.open(did, serial, true);
            mcc.blink();
            mcc.dioConfig(0);
            if (did == MccUsb.USB_TC_AI_DID) {
                for (int j = 0; j < N_DEV_CHANS; j++) {
                    mcc.configAlarm(j, 0, 0, 0f, 0f);
                }
            }
            else if (did == MccUsb.USB_TEMP_DID) {
                for (int j = 0; j < N_DEV_CHANS; j += 2) {
                    mcc.setSensorType(j, MccUsb.STP_DISABLED);
                }
            }
            setOnline(true);
            initSensors();
            if (lineMask != 0) {
                setOutputLines();
            }
            String message = "Connected to " + fullName;
            if (!inited) {
                log.info(message);
            }
            else {
                log.error(message);
            }
        }
        catch (Exception e) {
            checkUsbException(e);
            if (!inited) {
                log.error("Error connecting to " + fullName + ": " + e);
            }
            if (mcc != null) {
                close();
            }
        }
        finally {
            inited = true;
        }
    }


   /**
    ***************************************************************************
    **
    **  Initializes a channel.
    **
    ***************************************************************************
    */
    @Override
    void initChannel(int chan, int type, int subtype)
    {
        if (!online) return;

        try {
            if (type == MonChannel.TYPE_TEMP) {
                int sType = subtype & 0xff, tType = subtype >> 8;
                mcc.setSensorType(chan, sType);
                if (sType == MccUsb.STP_RTD) {
                    mcc.setThermConnType(chan, tType);
                    mcc.setExcitation(chan, MccUsb.CEX_210UA);
                    mcc.setGain(chan, MccUsb.GAIN_4X);
                    for (int j = 0; j < RTD_COEFFS.length; j++) {
                        mcc.setCoefficient(chan, j, RTD_COEFFS[j]);
                    }
                }
                else {
                    mcc.setTCType(chan, tType);
                }
            }
            else if (type == MonChannel.TYPE_VOLTS) {
                mcc.setGain(chan, subtype);
                mcc.setVoltageConnType(chan, MccUsb.VCT_DIFFERENTIAL);
            }
            else if (type == MonChannel.TYPE_SWITCH) {
                mcc.dioConfigBit(chan, 1);
            }
        }
        catch (Exception e) {
            checkUsbException(e);
            log.error("Error configuring " + fullName + ": " + e);
            setOnline(false);
        }
    }


   /**
    ***************************************************************************
    **
    **  Reads a channel.
    **
    ***************************************************************************
    */
    @Override
    double readChannel(int chan)
    {
        double value = 0;
        if (!online) return value;

        if (type == MonChannel.TYPE_SWITCH) {
            try {
                value = mcc.dioInBit(chan);
            }
            catch (Exception e) {
                checkUsbException(e);
                log.error("Error reading DIO line: " + e);
                setOnline(false);
            }
        }
        else {
            try {
                value = mcc.adcIn(chan, 0);
            }
            catch (Exception e) {
                checkUsbException(e);
                log.error("Error reading ADC: " + e);
                setOnline(false);
            }
        }

        return value;
    }


   /**
    ***************************************************************************
    **
    **  Checks an output line number.
    **
    ***************************************************************************
    */
    @Override
    void checkLine(String name, int line) throws Exception
    {
        if (line < 0 || line >= N_DEV_CHANS) {
            refg.reportError(name, "line", line);
        }
    }


   /**
    ***************************************************************************
    **
    **  Adds an output line.
    **
    ***************************************************************************
    */
    @Override
    void addLine(int id)
    {
        lineMask |= 1 << id;
    }


   /**
    ***************************************************************************
    **
    **  Sets an output line on or off.
    **
    ***************************************************************************
    */
    @Override
    void setLine(int id, boolean on)
    {
        try {
            mcc.dioOutBit(id, on ? 1 : 0);
        }
        catch (Exception e) {
            checkUsbException(e);
            log.error("Error setting DIO line: " + e);
            setOnline(false);
        }
    }


   /**
    ***************************************************************************
    **
    **  Gets the set state of an output line.
    **
    ***************************************************************************
    */
    @Override
    boolean isLineSet(int id)
    {
        try {
            return mcc.dioInBit(id) != 0;
        }
        catch (Exception e) {
            checkUsbException(e);
            log.error("Error reading DIO line: " + e);
            setOnline(false);
            return false;
        }
    }


   /**
    ***************************************************************************
    **
    **  Closes the connection.
    **
    ***************************************************************************
    */
    @Override
    void close()
    {
        try {
            mcc.close();
        }
        catch (Exception e) {
            checkUsbException(e);
        }
    }


   /**
    ***************************************************************************
    **
    **  Checks whether an exception is USB-related
    **
    ***************************************************************************
    */
    static void checkUsbException(Exception e)
    {
        if (e instanceof UsbException
            || e instanceof UsbDisconnectedException
            || e instanceof UsbEncodingException
            || e instanceof UsbNotActiveException
            || e instanceof UsbNotOpenException
            || e instanceof UsbNotClaimedException) return;
        throw (RuntimeException)e;
    }

}
