package org.lsst.ccs.subsystem.refrig;

import java.util.BitSet;
import org.lsst.ccs.framework.ConfigurableComponent;
import org.lsst.ccs.utilities.logging.Logger;

/**
 ******************************************************************************
 **
 **  Handles refrigeration devices
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public abstract class Device extends ConfigurableComponent {

   /**
    ***************************************************************************
    **
    **  Constants.
    **
    ***************************************************************************
    */
    final static int
        TYPE_MCC     = 0,
        TYPE_WATTS   = 1;

   /**
    ***************************************************************************
    **
    **  Data fields.
    **
    ***************************************************************************
    */
    Logger        log;       // The logger
    int           type;      // Device type code
    String        fullName;  // Full device name
    BitSet        chanMask;  // Channel IDs on this device
    RefrigTest    refg;      // Associated refrigeration object
    boolean       inited;    // True if initialized
    boolean       online;    // True if online (running successfully)


   /**
    ***************************************************************************
    **
    **  Performs configuration.
    **
    ***************************************************************************
    */
    void configure(RefrigTest refg)
    {
        this.refg = refg;
        log = refg.getLogger();
        chanMask = new BitSet(refg.getNumChans());
    }


   /**
    ***************************************************************************
    **
    **  Performs basic initialization.
    **
    ***************************************************************************
    */
    abstract void initialize();


   /**
    ***************************************************************************
    **
    **  Checks a channel's parameters for validity.
    **
    ***************************************************************************
    */
    abstract int[] checkChannel(String name, int hwChan, String type,
                                String subtype) throws Exception;


   /**
    ***************************************************************************
    **
    **  Adds a channel to the list.
    **
    ***************************************************************************
    */
    void addChannel(int id)
    {
        chanMask.set(id);
    }


   /**
    ***************************************************************************
    **
    **  Initializes a channel.
    **
    ***************************************************************************
    */
    abstract void initChannel(int chan, int type, int subtype);


   /**
    ***************************************************************************
    **
    **  Reads a channel.
    **
    ***************************************************************************
    */
    abstract double readChannel(int chan);


   /**
    ***************************************************************************
    **
    **  Checks the online state.
    **
    ***************************************************************************
    */
    void checkOnline()
    {
        if (online) return;
        initialize();
    }


   /**
    ***************************************************************************
    **
    **  Sets the online state.
    **
    ***************************************************************************
    */
    void setOnline(boolean online)
    {
        this.online = online;
        refg.setOnline(chanMask, online);
        if (!online) {
            log.error("Disconnected from " + fullName);
            close();
        }
    }


   /**
    ***************************************************************************
    **
    **  Initializes all attached sensors.
    **
    ***************************************************************************
    */
    void initSensors()
    {
        for (int id = chanMask.nextSetBit(0); id >= 0;
             id = chanMask.nextSetBit(id + 1)) {
            refg.getChannel(id).initSensor();
        }
    }


   /**
    ***************************************************************************
    **
    **  Sets all output lines.
    **
    ***************************************************************************
    */
    void setOutputLines()
    {
        refg.setOutputLines();
    }


   /**
    ***************************************************************************
    **
    **  Closes the connection.
    **
    ***************************************************************************
    */
    abstract void close();


   /**
    ***************************************************************************
    **
    **  Checks an output line number.
    **
    ***************************************************************************
    */
    void checkLine(String name, int line) throws Exception
    {
    }


   /**
    ***************************************************************************
    **
    **  Adds an output line ID to the known list.
    **
    ***************************************************************************
    */
    void addLine(int id)
    {
    }


   /**
    ***************************************************************************
    **
    **  Sets an output line on or off.
    **
    ***************************************************************************
    */
    void setLine(int line, boolean on)
    {
    }


   /**
    ***************************************************************************
    **
    **  Gets an output line.
    **
    ***************************************************************************
    */
    boolean isLineSet(int line)
    {
        return false;
    }

}
