package org.lsst.ccs.subsystem.refrig;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;
import org.lsst.ccs.bus.ValueNotification;
import org.lsst.ccs.framework.ConfigurableComponent;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.framework.annotations.ConfigChanger;
import org.lsst.ccs.subsystem.refrig.data.RefrigChannel;
import org.lsst.ccs.subsystem.refrig.data.RefrigFullState;
import org.lsst.ccs.subsystem.refrig.data.RefrigState;
import org.lsst.ccs.subsystem.refrig.status.RefrigStateStatus;

/**
 ******************************************************************************
 **
 **  Implements the refrigeration long lines test modular subsystem.
 **
 **  This is an abstract class implementing the interactions with the CCS
 **  buses, but deferring the hardware interaction, real or simulated, to
 **  sub-classes.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public abstract class RefrigTest extends Module {

   /**
    ***************************************************************************
    **
    **  Constants.
    **
    ***************************************************************************
    */
    final static long
        UPDATE_PERIOD = 1000;

   /**
    ***************************************************************************
    **
    **  Data fields.
    **
    ***************************************************************************
    */
    Channel[] chanData;
    private String configName = "test";
    private boolean coldStart, running;
    private int loLimChange, hiLimChange, setState, chanState, chanOnline;


   /**
    ***************************************************************************
    **
    **  Inner container class for channel parameters.
    **
    ***************************************************************************
    */
    public static class Channel extends ConfigurableComponent {

       /**
        ***********************************************************************
        **
        **  Data fields.
        **
        ***********************************************************************
        */
        String     description;   // Channel description
        String     units;         // Units name
        String     loCheckS;      // Low limit check option string
        String     hiCheckS;      // High limit check option string
        int        loCheck;       // Low limit check option
        int        hiCheck;       // High limit check option
        float      deadbandLo;    // Low limit dead-band
        float      deadbandHi;    // High limit dead-band
        float      limitLo;       // Low limit value
        float      limitHi;       // High limit value
        float      value;         // Current value read from channel
        boolean    trippedLo;     // Whether low limit has caused a trip
        boolean    trippedHi;     // Whether high limit has caused a trip
        int        id;            // External ID
        RefrigTest refg;          // Associated refrigeration object

       /**
        ***********************************************************************
        **
        **  Package constants.
        **
        ***********************************************************************
        */
        final static int
            LIMIT_CHECK_NONE      = 0,   // Don't check limit
            LIMIT_CHECK_FLAG      = 1,   // Flag if limit exceeded
            LIMIT_CHECK_MAIN_TRIP = 2,   // Trip main power if limit exceeded
            LIMIT_CHECK_LOAD_TRIP = 3;   // Trip load power if limit exceeded

       /**
        ***********************************************************************
        **
        **  Private lookup maps.
        **
        ***********************************************************************
        */
        private final static Map<String,Integer> checkMap = new HashMap<>();
        static {
            checkMap.put("NONE", LIMIT_CHECK_NONE);
            checkMap.put("FLAG", LIMIT_CHECK_FLAG);
            checkMap.put("MTRIP", LIMIT_CHECK_MAIN_TRIP);
            checkMap.put("LTRIP", LIMIT_CHECK_LOAD_TRIP);
        }


       /**
        ***********************************************************************
        **
        **  Constructor.
        **
        ***********************************************************************
        */
        public Channel(String desc, String units,
                       String loCheck, double limitLo, double deadbandLo,
                       String hiCheck, double limitHi, double deadbandHi)
        {
            this.description = desc;
            this.units = units;
            this.loCheckS = loCheck;
            this.hiCheckS = hiCheck;
            this.limitLo = (float)limitLo;
            this.deadbandLo = (float)deadbandLo;
            this.limitHi = (float)limitHi;
            this.deadbandHi = (float)deadbandHi;
        }


       /**
        ***********************************************************************
        **
        **  Configures channel description.
        **
        ***********************************************************************
        */
        void configure(RefrigTest refg, int id)
        {
            this.refg = refg;
            this.id = id;
            int iLoCheck = LIMIT_CHECK_FLAG, iHiCheck = LIMIT_CHECK_FLAG;
            try {
                Integer iCheck = checkMap.get(loCheckS.toUpperCase());
                if (iCheck == null) {
                    reportError(getName(), "locheck", loCheckS);
                }
                iLoCheck = iCheck;
                iCheck = checkMap.get(hiCheckS.toUpperCase());
                if (iCheck == null) {
                    reportError(getName(), "hicheck", hiCheckS);
                }
                iHiCheck = iCheck;
            }
            catch (Exception e) {
            }

            loCheck = iLoCheck;
            hiCheck = iHiCheck;
        }


       /**
        ***********************************************************************
        **
        **  Sets the low limit.
        **
        ***********************************************************************
        */
        @ConfigChanger
        public void setLimitLo(double limit)
        {
            float fLimit = (float)limit;
            if (fLimit == limitLo) return;
            limitLo = fLimit;
            refg.loLimChange |= 1 << id;
            refg.publishState();
            refg.getSubsystem().publishMetaData(getName(), "alarmLow",
                                                String.valueOf(fLimit));
        }


       /**
        ***********************************************************************
        **
        **  Sets the high limit.
        **
        ***********************************************************************
        */
        @ConfigChanger
        public void setLimitHi(double limit)
        {
            float fLimit = (float)limit;
            if (fLimit == limitHi) return;
            limitHi = fLimit;
            refg.hiLimChange |= 1 << id;
            refg.publishState();
            refg.getSubsystem().publishMetaData(getName(), "alarmHigh",
                                                String.valueOf(fLimit));
        }


       /**
        ***********************************************************************
        **
        **  Checks whether current value is within limits.
        **
        ***********************************************************************
        */
        int checkLimits(int cState, int setState)
        {
            boolean ok = false;
            if (value < limitLo) {
                if (loCheck == Channel.LIMIT_CHECK_NONE) {
                    ok = true;
                }
                else if (loCheck == Channel.LIMIT_CHECK_MAIN_TRIP) {
                    cState |= RefrigState.MAIN_TRIPPED_STATE;
                }
                else if (loCheck == Channel.LIMIT_CHECK_LOAD_TRIP) {
                    if ((setState & RefrigState.LOAD_TRIP_ENAB_STATE) != 0) {
                        cState &= ~RefrigState.LOAD_POWER_STATE;
                        trippedLo = true;
                    }
                }
            }
            else if (value > limitHi) {
                if (hiCheck == Channel.LIMIT_CHECK_NONE) {
                    ok = true;
                }
                else if (hiCheck == Channel.LIMIT_CHECK_MAIN_TRIP) {
                    cState |= RefrigState.MAIN_TRIPPED_STATE;
                }
                else if (hiCheck == Channel.LIMIT_CHECK_LOAD_TRIP) {
                    if ((setState & RefrigState.LOAD_TRIP_ENAB_STATE) != 0) {
                        cState &= ~RefrigState.LOAD_POWER_STATE;
                        trippedHi = true;
                    }
                }
            }
            else {
                ok = true;
                if (loCheck == Channel.LIMIT_CHECK_LOAD_TRIP) {
                    boolean tripped = false;
                    if ((setState & RefrigState.LOAD_TRIP_ENAB_STATE) != 0) {
                        if (trippedLo && value < limitLo + deadbandLo) {
                            cState &= ~RefrigState.LOAD_POWER_STATE;
                            tripped = true;
                        }
                    }
                    trippedLo = tripped;
                }
                if (hiCheck == Channel.LIMIT_CHECK_LOAD_TRIP) {
                    boolean tripped = false;
                    if ((setState & RefrigState.LOAD_TRIP_ENAB_STATE) != 0) {
                        if (trippedHi && value > limitHi - deadbandHi) {
                            cState &= ~RefrigState.LOAD_POWER_STATE;
                            tripped = true;
                        }
                    }
                    trippedHi = tripped;
                }
            }
            if (ok) {
                cState |= (1 << id);
            }

            return cState;
        }

    }


   /**
    ***************************************************************************
    **
    **  Inner class to implement a timer task to update the channel state
    **  in a timely manner.
    **
    ***************************************************************************
    */
    private class UpdateState extends TimerTask {

        @Override
        public void run()
        {
            /*
            **  Read the sensor values
            */
            readSensors();

            /*
            **  Check the limits and set the channel state
            */
            checkLimits();
        }

    }

                
   /**
    ***************************************************************************
    **
    **  Main constructor.
    **
    ***************************************************************************
    */
    public RefrigTest(String name, int tickMillis, String configName)
    {
        super(name, tickMillis);
        this.configName = configName;
        String cold = System.getProperty("lsst.ccs.refrig.coldstart", "");
        coldStart = cold.equals("true");
    }


   /**
    ***************************************************************************
    **
    **  Initializes the refrigeration subsystem.
    **
    ***************************************************************************
    */
    @Override
    public void initModule()
    {
        /*
        **  Initialize all configuration data
        */
        initConfiguration();

        /*
        **  Initialize the hardware
        */
        initSensors();

        /*
        **  Set output lines
        */
        setOutputLines();
        coldStart = false;

        /*
        **  Start the timer task which updates the channel state periodically
        */
        (new Timer(true)).schedule(new UpdateState(), 0L, UPDATE_PERIOD);
    }


   /**
    ***************************************************************************
    **
    **  Performs periodic trending data broadcast.
    **
    ***************************************************************************
    */
    @Override
    public void tick()
    {
        /*
        **  Broadcast the state if first time
        */
        if (!running) {
            System.out.println("Refrigeration test system started");
            publishState();    // For any GUIs
            publishLimits();   // For GUIs and the trending database
            running = true;
        }

        /*
        **  Broadcast the trending data
        */
        long timeStamp = System.currentTimeMillis();
        List<ValueNotification> trendingValues = new ArrayList<>();
        for (Channel ch : chanData) {
            trendingValues.add(new ValueNotification(ch.getName(), ch.value,
                                                     timeStamp));
        }
        trendingValues.add(new ValueNotification("state", getState(),
                                                 timeStamp));
        publishData(trendingValues);
    }


   /**
    ***************************************************************************
    **
    **  Sets the tick period.
    **
    **  Overrides the method in Module in order to publish a status update.
    **
    **  @param  value  The tick period (milliseconds) to set.
    **
    ***************************************************************************
    */
    @Override
    public void setTickMillis(int value)
    {
        super.setTickMillis(value);
        if (running) {
            publishState();
        }
    }


   /**
    ***************************************************************************
    **
    **  Sets the main power enabled state on or off.
    **
    **  @param  value  The enabled state value to set: 0 = off, ~0 = on.
    **
    ***************************************************************************
    */
    //@BusCommand
    public void setMainPowerEnable(int value)
    {
        setMainPowerEnable(value, true);
    }


   /**
    ***************************************************************************
    **
    **  Sets the main power enabled state on or off.
    **
    **  @param  value    The enabled state value to set: 0 = off, ~0 = on.
    **
    **  @param  publish  If true, publish the new state on the status bus.
    **
    ***************************************************************************
    */
    public void setMainPowerEnable(int value, boolean publish)
    {
        boolean on = false;
        if (value != 0) {
            if ((chanState & RefrigState.MAIN_TRIPPED_STATE) == 0) {
                setState |= RefrigState.MAIN_POWER_STATE;
                on = true;
            }
        }
        else {
            setState &= ~RefrigState.MAIN_POWER_STATE;
        }
        setMainPower(on);
        if (publish) {
            publishState();
        }
    }


   /**
    ***************************************************************************
    **
    **  Sets the load power trip enabled state on or off.
    **
    **  @param  value  The load trip enabled state value to set: 0 = off;
    **                 ~0 = on.
    **
    ***************************************************************************
    */
    //@BusCommand
    public void setLoadTripEnable(int value)
    {
        if (value != 0) {
            setState |= RefrigState.LOAD_TRIP_ENAB_STATE;
        }
        else {
            setState &= ~RefrigState.LOAD_TRIP_ENAB_STATE;
        }
        publishState();
    }


   /**
    ***************************************************************************
    **
    **  Sets the low limit for a channel.
    **
    **  @param  id     The channel ID
    **
    **  @param  limit  The low limit value.
    **
    ***************************************************************************
    */
    //@BusCommand
    public void setLowLimit(int id, double limit) throws Exception
    {
        chanData[id].change("limitLo", limit);
    }


   /**
    ***************************************************************************
    **
    **  Sets the high limit for a channel.
    **
    **  @param  id     The channel ID
    **
    **  @param  limit  The high limit value.
    **
    ***************************************************************************
    */
    //@BusCommand
    public void setHighLimit(int id, double limit) throws Exception
    {
        chanData[id].change("limitHi", limit);
    }


   /**
    ***************************************************************************
    **
    **  Saves the configuration data.
    **
    ***************************************************************************
    */
    //@BusCommand
    public void saveConfiguration() throws IOException
    {
        register(configName, "");
        loLimChange = 0;
        hiLimChange = 0;
        publishState();         // Must always do this
    }


   /**
    ***************************************************************************
    **
    **  Gets the full state of the refrigeration module.
    **
    **  This is intended to be called by GUIs during initialization
    **
    ***************************************************************************
    */
    //@BusCommand
    public RefrigFullState getFullState()
    {
        RefrigFullState status = new RefrigFullState();
        for (Channel ch : chanData) {
            status.addChannel(new RefrigChannel(ch.getName(), ch.description,
                                                ch.units, ch.limitLo,
                                                ch.limitHi, ch.value));
        }
        status.setRefrigState(new RefrigState(getState(), chanOnline,
                                              loLimChange, hiLimChange,
                                              getTickMillis()));
        return status;
    }    


   /**
    ***************************************************************************
    **
    **  Sets the output lines from the current state.
    **
    ***************************************************************************
    */
    void setOutputLines()
    {
        /*
        **  Set the power state
        */
        setState &= ~RefrigState.MAIN_POWER_STATE;
        if (!coldStart) {
            if (isMainPowerOn()) {
                setState |= RefrigState.MAIN_POWER_STATE;
            }
        }
        setMainPower((setState & RefrigState.MAIN_POWER_STATE) != 0);
        setLoadPower((chanState & RefrigState.LOAD_POWER_STATE) != 0);

        /*
        **  Set the display
        */
        setDisplay(-1, chanState);
    }


   /**
    ***************************************************************************
    **
    **  Gets the operating state word.
    **
    **  @return  The current value of the operating state
    **
    ***************************************************************************
    */
    int getState()
    {
        return setState | chanState;
    }


   /**
    ***************************************************************************
    **
    **  Publishes the state of the refrigeration module.
    **
    **  This is intended to be called whenever any element of the state is
    **  changed.
    **
    ***************************************************************************
    */
    void publishState()
    {
        RefrigState refrigState = new RefrigState(getState(), chanOnline,
                                                  loLimChange, hiLimChange,
                                                  getTickMillis());
        sendToStatus(new RefrigStateStatus(refrigState));
    }    


   /**
    ***************************************************************************
    **
    **  Publishes all the limit values of the refrigeration module.
    **
    **  This is intended to be called at startup time.
    **
    ***************************************************************************
    */
    void publishLimits()
    {
        for (Channel ch : chanData) {
            getSubsystem().publishMetaData(ch.getName(), "alarmLow",
                                           String.valueOf(ch.limitLo));
            getSubsystem().publishMetaData(ch.getName(), "alarmHigh",
                                           String.valueOf(ch.limitHi));
        }
    }    


   /**
    ***************************************************************************
    **
    **  Checks whether all current channel values are within limits.
    **
    ***************************************************************************
    */
    void checkLimits()
    {
        /*
        **  Generate the current channel state
        */
        int cState = RefrigState.LOAD_POWER_STATE;
        for (Channel ch : chanData) {
            cState = ch.checkLimits(cState, setState);
        }
                
        /*
        **  If state changed
        **    Turn main power off if it was tripped off
        **    Turn load power off or on appropriately
        **    Set alarms to new values
        **    If running
        **      Publish state
        **      Log channel data
        */
        cState ^= chanState;
        chanState ^= cState;
        if (cState != 0) {
            int tr;
            if ((cState & RefrigState.MAIN_TRIPPED_STATE) != 0) {
                if ((chanState & RefrigState.MAIN_TRIPPED_STATE) != 0) {
                    setMainPowerEnable(0, false);
                }
            }
            if ((cState & RefrigState.LOAD_POWER_STATE) != 0) {
                setLoadPower((chanState & RefrigState.LOAD_POWER_STATE) != 0);
            }
            setDisplay(cState, chanState);
            if (running) {
                publishState();
                tick();
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Finds the ID of a channel given its name.
    **
    ***************************************************************************
    */
    int findChannelId(String name)
    {
        for (Channel ch : chanData) {
            if (ch.getName().equals(name)) return ch.id;
        }

        return -1;
    }


   /**
    ***************************************************************************
    **
    **  Gets a channel given its ID.
    **
    ***************************************************************************
    */
    Channel getChannel(int id)
    {
        return chanData[id];
    }


   /**
    ***************************************************************************
    **
    **  Sets channels on- or off-line.
    **
    ***************************************************************************
    */
    void setOnline(int mask, boolean online)
    {
        int prevOnline = chanOnline;
        if (online) {
            chanOnline |= mask;
        }
        else {
            chanOnline &= ~mask;
        }
        if (chanOnline != prevOnline) {
            publishState();
        }
    }


   /**
    ***************************************************************************
    **
    **  Reports parameter errors.
    **
    ***************************************************************************
    */
    static void reportError(String cName, String pName, Object pValue)
        throws Exception
    {
        log.error("Invalid " + pName + " (" + pValue + ") for " + cName);
        throw new Exception();
    }


   /**
    ***************************************************************************
    **
    **  Initializes the refrigeration configuration.
    **
    ***************************************************************************
    */
    abstract void initConfiguration();


   /**
    ***************************************************************************
    **
    **  Initializes the refrigeration sensors.
    **
    ***************************************************************************
    */
    abstract void initSensors();


   /**
    ***************************************************************************
    **
    **  Reads the sensor data.
    **
    ***************************************************************************
    */
    abstract void readSensors();


   /**
    ***************************************************************************
    **
    **  Turns the main power on or off.
    **
    ***************************************************************************
    */
    abstract void setMainPower(boolean on);
            

   /**
    ***************************************************************************
    **
    **  Turns the load power on or off.
    **
    ***************************************************************************
    */
    abstract void setLoadPower(boolean on);


   /**
    ***************************************************************************
    **
    **  Gets the hardware main power state.
    **
    ***************************************************************************
    */
    abstract boolean isMainPowerOn();
            

   /**
    ***************************************************************************
    **
    **  Gets the hardware load power state.
    **
    ***************************************************************************
    */
    abstract boolean isLoadPowerOn();
            

   /**
    ***************************************************************************
    **
    **  Sets the hardware display, according to the current state.
    **
    ***************************************************************************
    */
    abstract void setDisplay(int mask, int state);
    
}
