package org.lsst.ccs.subsystem.refrig;

import java.util.Iterator;
import java.util.Map;
import org.lsst.ccs.subsystem.refrig.data.RefrigState;

/**
 ******************************************************************************
 **
 **  Implements the simulated refrigeration long lines test modular subsystem
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class RefrigTestSim extends RefrigTest {

   /**
    ***************************************************************************
    **
    **  Private fields.
    **
    ***************************************************************************
    */
    private final static float LOAD_POWER_FCTR = 0.0006667F;
    private ChannelSim[] chanDataSim;
    private int loadPwrId = -1, loadTmpId = -1;
    private long currTime = System.currentTimeMillis();
    private double loadFract = 0.6;

   /**
    ***************************************************************************
    **
    **  Inner container class for channel simulation data.
    **
    ***************************************************************************
    */
    public static class ChannelSim extends Channel {

       /**
        ***********************************************************************
        **
        **  Data fields
        **
        ***********************************************************************
        */
        float         onValue;    // On (operating) value
        float         onTime;     // Time (secs) to reach on value
        float         offValue;   // Off (ambient) value
        float         offTime;    // Time (secs) to reach off value
        float         endValue;   // Ending value
        float         rate;       // Rate of change
        boolean       negative;   // True if nominal change is negative
        RefrigTestSim refSim;     // Associated refrigeration object


       /**
        ***********************************************************************
        **
        **  Constructor
        **
        ***********************************************************************
        */
        public ChannelSim(String desc, String units,
                          String loCheck, double limitLo, double deadbandLo,
                          String hiCheck, double limitHi, double deadbandHi,
                          double onValue, double onTime,
                          double offValue, double offTime)
        {
            super(desc, units, loCheck, limitLo, deadbandLo,
                  hiCheck, limitHi, deadbandHi);
            this.onValue  = (float)onValue;
            this.onTime   = (float)onTime;
            this.offValue = (float)offValue;
            this.offTime  = (float)offTime;
            this.endValue = 0f;
            this.rate     = 0f;
            this.negative = false;
        }


       /**
        ***********************************************************************
        **
        **  Configures direct channel data
        **
        ***********************************************************************
        */
        void configure(RefrigTestSim ref, int id)
        {
            super.configure(ref, id);
            refSim = ref;
        }


       /**
        ***********************************************************************
        **
        **  Initializes the simulated sensor
        **
        ***********************************************************************
        */
        void initSensor()
        {
            if (id == refSim.loadPwrId) {
                onTime = 0F;
                offValue = 0F;
                offTime = 0F;
            }
            endValue = offValue;
            value    = offValue;
        }


       /**
        ***********************************************************************
        **
        **  Reads the simulated sensor
        **
        ***********************************************************************
        */
        void readSensor(double intvl)
        {
            value += rate * intvl;
            if (negative) {
                if (value < endValue) {
                    value = endValue;
                }
            }
            else {
                if (value > endValue) {
                    value = endValue;
                }
            }
        }


       /**
        ***********************************************************************
        **
        **  Sets sensor to power-on value
        **
        ***********************************************************************
        */
        void powerOn()
        {
            rate = 0F;
            if ((refSim.setState & RefrigState.MAIN_POWER_STATE) != 0) {
                endValue = onValue;
                if (onTime != 0F) {
                    rate = (onValue - offValue) / onTime;
                }
            }
            else {
                endValue = offValue;
                if (offTime != 0F) {
                    rate = (offValue - onValue) / offTime;
                }
            }
            if (rate == 0F) {
                value = endValue;
            }
            negative = rate < 0F;
        }


       /**
        ***********************************************************************
        **
        **  Gets the off value
        **
        ***********************************************************************
        */
        float getOffValue()
        {
            return offValue;
        }


       /**
        ***********************************************************************
        **
        **  Gets the on value
        **
        ***********************************************************************
        */
        float getOnValue()
        {
            return onValue;
        }


       /**
        ***********************************************************************
        **
        **  Gets the on time
        **
        ***********************************************************************
        */
        float getOnTime()
        {
            return onTime;
        }


       /**
        ***********************************************************************
        **
        **  Sets the end value
        **
        ***********************************************************************
        */
        void setEndValue(float value)
        {
            endValue = value;
        }


       /**
        ***********************************************************************
        **
        **  Sets the change rate
        **
        ***********************************************************************
        */
        void setRate(float value)
        {
            rate = value;
        }


       /**
        ***********************************************************************
        **
        **  Sets the current value
        **
        ***********************************************************************
        */
        void setValue(float value)
        {
            this.value = value;
        }

    }


   /**
    ***************************************************************************
    **
    **  Main Constructor
    **
    ***************************************************************************
    */
    public RefrigTestSim(String name, int tickMillis, String configName)
    {
        super(name, tickMillis, configName);
    }


   /**
    ***************************************************************************
    **
    **  Sets the load power fraction
    **
    ***************************************************************************
    */
    public void setLoadFract(double fract)
    {
        loadFract = (float)fract;
        setLoadPower();
    }


   /**
    ***************************************************************************
    **
    **  Gets the load power fraction
    **
    ***************************************************************************
    */
    public double getLoadFract()
    {
        return loadFract;
    }


   /**
    ***************************************************************************
    **
    **  Initializes the simulated channels with description data.
    **
    ***************************************************************************
    */
    @Override
    void initConfiguration()
    {
        Map<String, ChannelSim> chData = getChildren(ChannelSim.class);
        nChan = chData.size();
        chanData = new ChannelSim[nChan];
        chanDataSim = (ChannelSim[])chanData;
        Iterator<ChannelSim> values = chData.values().iterator();
        for (int id = 0; id < nChan; id++) {
            ChannelSim ch = chanDataSim[id] = values.next();
            ch.configure(this, id);
            if (ch.getName().equals("LoadPower")) {
                loadPwrId = id;
            }
            if (ch.getName().equals("LoadTmp")) {
                loadTmpId = id;
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Initializes the simulated sensors.
    **
    ***************************************************************************
    */
    @Override
    void initSensors()
    {
        for (ChannelSim ch : chanDataSim) {
            ch.initSensor();
        }
    }


   /**
    ***************************************************************************
    **
    **  Reads the simulated sensors.
    **
    ***************************************************************************
    */
    @Override
    void readSensors()
    {
        long thisTime = System.currentTimeMillis();
        double intvl = (thisTime - currTime) / 1000.0;
        currTime = thisTime;
        for (ChannelSim ch : chanDataSim) {
            ch.readSensor(intvl);
        }
    }


   /**
    ***************************************************************************
    **
    **  Turns the main power on or off.
    **
    ***************************************************************************
    */
    @Override
    void setMainPower()
    {
        currTime = System.currentTimeMillis();
        for (ChannelSim ch : chanDataSim) {
            if (ch.getId() == loadPwrId) continue;
            ch.powerOn();
        }
        setLoadPower();
    }
            

   /**
    ***************************************************************************
    **
    **  Turns the load power on or off, setting the correct level.
    **
    ***************************************************************************
    */
    @Override
    void setLoadPower()
    {
        ChannelSim pc = chanDataSim[loadPwrId];
        float endValue = 0F;
        if ((setState & RefrigState.MAIN_POWER_STATE) != 0) {
            if ((chanState & RefrigState.LOAD_POWER_STATE) != 0) {
                endValue = (float)(loadFract * pc.getOnValue());
            }
            ChannelSim tc = chanDataSim[loadTmpId];
            if (tc.getOnTime() != 0F) {
                tc.setRate((tc.getOnValue() - tc.getOffValue()) / tc.getOnTime()
                             + LOAD_POWER_FCTR * endValue);
            }
        }
        pc.setEndValue(endValue);
        pc.setValue(endValue);
    }


   /**
    ***************************************************************************
    **
    **  Sets the hardware display
    **
    ***************************************************************************
    */
    @Override
    void setDisplay()
    {
    }

}
