package org.lsst.ccs.subsystem.refrig;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import javax.usb.UsbException;
import org.lsst.ccs.drivers.mcc.MccUsb;
import org.lsst.ccs.drivers.wattsup.WattsUp;
import org.lsst.ccs.subsystem.refrig.data.RefrigState;

/**
 ******************************************************************************
 **
 **  Implements the refrigeration long lines test modular subsystem
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class RefrigTestHdw extends RefrigTest {

   /**
    ***************************************************************************
    **
    **  Private constants.
    **
    ***************************************************************************
    */
    private final static int
        N_MCC_DEV_CHANS   = 8,
        N_ALARMS          = 6,
        MAIN_POWER_BIT    = 7,
        LOAD_POWER_BIT    = 6,
        ALARMS_MASK = ((1 << N_ALARMS) - 1),
        LEDS_MASK = ALARMS_MASK | (1 << LOAD_POWER_BIT),
        N_WATTS_DEV_CHANS = 2,
        WATTS_LOG_PERIOD  = 1,
        CHAN_TYPE_UNKNOWN = -1,
        CHAN_TYPE_TEMP    = 0,
        CHAN_TYPE_VOLTS   = 1,
        CHAN_TYPE_POWER   = 2,
        CHAN_TYPE_SWITCH  = 3,
        CHAN_TYPE_DIFF    = 4;

    private final static float[]
        RTD_COEFFS = {100.0F, 0.003908F, -5.8019E-7F, -4.2735E-12F};

   /**
    ***************************************************************************
    **
    **  Private fields.
    **
    ***************************************************************************
    */
    private MCC[] mccData;
    private Watts[] wuData;
    private ChannelHdw[] chanDataHdw;
    private int nMcc, nWatts;
    private int tcChanMask, rtChanMask, vChanMask, mainPowerDev, loadPowerDev,
                loadPowerDev2, alarmsDev, alarmsDev2;


   /**
    ***************************************************************************
    **
    **  Inner container class for MCC device description data.
    **
    ***************************************************************************
    */
    public static class MCC {

       /**
        ***********************************************************************
        **
        **  Data fields
        **
        ***********************************************************************
        */
        String  type;     // Device type name
        int     did;      // Device ID
        String  serial;   // Device serial number
        MccUsb  mcc;      // Associated USB object
        boolean inited;   // true if initialized successfully

       /**
        ***********************************************************************
        **
        **  Private lookup maps
        **
        ***********************************************************************
        */
        private final static HashMap<String,Integer> didMap;
        static {
            didMap = new HashMap<String,Integer>();
            didMap.put("TC_AI", MccUsb.USB_TC_AI_DID);
            didMap.put("TC",    MccUsb.USB_TC_DID);
            didMap.put("TEMP",  MccUsb.USB_TEMP_DID);
        }


       /**
        ***********************************************************************
        **
        **  Constructor
        **
        ***********************************************************************
        */
        public MCC(String type, String serial)
        {
            this.type = type;
            this.serial = serial;
            Integer iDid = didMap.get(type);
            if (iDid == null) {
                log.error("Invalid MCC device type (" + type + ")");
                did = -1;
            }
            else {
                did = iDid;
            }
        }


       /**
        ***********************************************************************
        **
        **  Configures
        **
        ***********************************************************************
        */
        private void configure(RefrigTestHdw ref, int id)
        {
            int shift = N_MCC_DEV_CHANS * id;
            if (did == MccUsb.USB_TC_AI_DID) {
                ref.tcChanMask |= 0x0f << shift;
                ref.vChanMask |= 0xf0 << shift;
                ref.mainPowerDev = id;
                ref.loadPowerDev = id;
                ref.alarmsDev = id;
            }
            else if (did == MccUsb.USB_TC_DID) {
                ref.tcChanMask |= 0xff << shift;
                ref.loadPowerDev2 = id;
                ref.alarmsDev2 = id;
            }
            else if (did == MccUsb.USB_TEMP_DID) {
                ref.tcChanMask |= 0xff << shift;
                ref.rtChanMask |= 0xff << shift;
            }
        }


       /**
        ***********************************************************************
        **
        **  Performs basic initialization
        **
        ***********************************************************************
        */
        private void initialize()
        {
            if (did < 0) return;
            try {
                mcc = new MccUsb();
                mcc.open(did, serial, true);
                mcc.blink();
                mcc.dioConfig(0);
                mcc.dioOut(0);
                if (did == MccUsb.USB_TC_AI_DID) {
                    for (int j = 0; j < N_ALARMS; j++) {
                        mcc.configAlarm(j, 0, 0, 0f, 0f);
                    }
                }
                else if (did == MccUsb.USB_TEMP_DID) {
                    for (int j = 0; j < N_MCC_DEV_CHANS; j += 2) {
                        mcc.setSensorType(j, MccUsb.STP_DISABLED);
                    }
                }
                inited = true;
            }
            catch (UsbException e) {
                log.error("Error initializing " + type + " module: " + e);
            }
        }

    }


   /**
    ***************************************************************************
    **
    **  Inner container class for WattsUp meter description data.
    **
    ***************************************************************************
    */
    public static class Watts implements WattsUp.Listener {

       /**
        ***********************************************************************
        **
        **  Data fields
        **
        ***********************************************************************
        */
        String  node;        // Name of the node attached to the meter
        String  serial;      // The serial number of the meter
        boolean inited;      // true if successfully initialized
        WattsUp wtu;         // WattsUp hardware object
        float[] value;       // Current values: power, voltage


       /**
        ***********************************************************************
        **
        **  Constructor
        **
        ***********************************************************************
        */
        public Watts(String node, String serial)
        {
            this.node = node;
            this.serial = serial;
            value = new float[N_WATTS_DEV_CHANS];
        }


       /**
        ***********************************************************************
        **
        **  Performs basic initialization
        **
        ***********************************************************************
        */
        private void initialize()
        {
            try {
                wtu = new WattsUp();
                wtu.addListener(this);
                wtu.open(node, 0, serial);
                inited = true;
            }
            catch (Exception e) {
                log.error("Cannot access WattsUp meter: " + e);
                for (int j = 0; j < N_WATTS_DEV_CHANS; j++) {
                    value[j] = -1F;
                }
            }
        }


       /**
        ***********************************************************************
        **
        **  Changes the WattsUp? meter powered state.
        **
        **  <p>
        **  Performs meter setup upon power-on.
        **
        **  @param  on  Whether the meter is powered on or not
        **
        ***********************************************************************
        */
        @Override
        public void setPowered(boolean on)
        {
            if (on) {
                try {
                    wtu.setLoggedFields((1 << WattsUp.FLD_WATTS)
                                          | (1 << WattsUp.FLD_VOLTS));
                    wtu.setExternalLogging(WATTS_LOG_PERIOD);
                }
                catch (Exception e) {
                    log.error("Cannot configure WattsUp meter: " + e);
                }
            }
            else {
                for (int j = 0; j < N_WATTS_DEV_CHANS; j++) {
                    value[j] = 0F;
                }
            }
        }


       /**
        ***********************************************************************
        **
        **  Changes the WattsUp? meter open state.
        **
        ***********************************************************************
        */
        @Override
        public void setClosed()
        {
            for (int j = 0; j < N_WATTS_DEV_CHANS; j++) {
                value[j] = -1F;
            }
            inited = false;
        }


       /**
        ***********************************************************************
        **
        **  Receives data periodically from the WattsUp? meter.
        **
        **  @param  data  The array of data from the meter.
        **
        ***********************************************************************
        */
        @Override
        public void processData(float[] data)
        {
            value[0] = data[WattsUp.FLD_WATTS];
            value[1] = data[WattsUp.FLD_VOLTS];
        }

    }


   /**
    ***************************************************************************
    **
    **  Inner container class for channel hardware data.
    **
    ***************************************************************************
    */
    public static class ChannelHdw extends Channel {

       /**
        ***********************************************************************
        **
        **  Data fields
        **
        ***********************************************************************
        */
        String        typeS;      // Channel type string (TEMP, VOLTS, POWER...)
        String        subtypeS;   // Channel subtype string (extra config info)
        int           chan;       // Hardware channel for MCC channel
        int           alarm;      // Hardware alarm number
        float         offset;     // Value offset
        float         scale;      // Value scale
        int           type;       // Encoded channel type
        int           subtype;    // Encoded channel subtype
        RefrigTestHdw refHdw;     // Associated refrigeration object

       /**
        ***********************************************************************
        **
        **  Private lookup maps
        **
        ***********************************************************************
        */
        private final static HashMap<String,Integer> typeMap;
        static {
            typeMap = new HashMap<String,Integer>();
            typeMap.put("TEMP",   CHAN_TYPE_TEMP);
            typeMap.put("VOLTS",  CHAN_TYPE_VOLTS);
            typeMap.put("POWER",  CHAN_TYPE_POWER);
            typeMap.put("SWITCH", CHAN_TYPE_SWITCH);
            typeMap.put("DIFF",   CHAN_TYPE_DIFF);
        }

        private final static HashMap<String,Integer> tSubtypeMap;
        static {
            tSubtypeMap = new HashMap<String,Integer>();
            tSubtypeMap.put("TC",  MccUsb.STP_THERMOCOUPLE);
            tSubtypeMap.put("RTD", MccUsb.STP_RTD);
        }

        private final static HashMap<String,Integer> tcSubtypeMap;
        static {
            tcSubtypeMap = new HashMap<String,Integer>();
            tcSubtypeMap.put("J", MccUsb.TC_TYPE_J);
            tcSubtypeMap.put("K", MccUsb.TC_TYPE_K);
            tcSubtypeMap.put("T", MccUsb.TC_TYPE_T);
            tcSubtypeMap.put("E", MccUsb.TC_TYPE_E);
            tcSubtypeMap.put("R", MccUsb.TC_TYPE_R);
            tcSubtypeMap.put("S", MccUsb.TC_TYPE_S);
            tcSubtypeMap.put("B", MccUsb.TC_TYPE_B);
            tcSubtypeMap.put("N", MccUsb.TC_TYPE_N);
        }

        private final static HashMap<String,Integer> rtSubtypeMap;
        static {
            rtSubtypeMap = new HashMap<String,Integer>();
            rtSubtypeMap.put("2WIRE1", MccUsb.TCT_2WIRE_1SENSOR);
            rtSubtypeMap.put("2WIRE2", MccUsb.TCT_2WIRE_2SENSOR);
            rtSubtypeMap.put("3WIRE",  MccUsb.TCT_3WIRE);
            rtSubtypeMap.put("4WIRE",  MccUsb.TCT_4WIRE);
        }

        private final static HashMap<String,Integer> vSubtypeMap;
        static {
            vSubtypeMap = new HashMap<String,Integer>();
            vSubtypeMap.put("10V",   MccUsb.RANGE_10V);
            vSubtypeMap.put("5V",    MccUsb.RANGE_5V);
            vSubtypeMap.put("2.5V",  MccUsb.RANGE_2_5V);
            vSubtypeMap.put("1.25V", MccUsb.RANGE_1_25V);
        }


       /**
        ***********************************************************************
        **
        **  Constructor
        **
        ***********************************************************************
        */
        public ChannelHdw(String desc, String units,
                          String loCheck, double limitLo, double deadbandLo,
                          String hiCheck, double limitHi, double deadbandHi,
                          String type, String subtype,
                          int chan, int alarm, double offset, double scale)
        {
            super(desc, units, loCheck, limitLo, deadbandLo, hiCheck, limitHi,
                  deadbandHi);
            this.typeS    = type;
            this.subtypeS = subtype;
            this.chan     = chan;
            this.alarm    = alarm;
            this.offset   = (float)offset;
            this.scale    = (float)scale;
        }


       /**
        ***********************************************************************
        **
        **  Configures direct channel data
        **
        ***********************************************************************
        */
        void configure(RefrigTestHdw ref, int id)
        {
            super.configure(ref, id);
            refHdw = ref;
            int iType = CHAN_TYPE_UNKNOWN, iSubtype = -1, iChan = -1,
                iAlarm = -1;
            try {
                Integer iTyp = typeMap.get(typeS.toUpperCase());
                if (iTyp == null) {
                    reportError(getName(), "type", typeS);
                }
                iType = iTyp;

                String sbtype = subtypeS.toUpperCase();
                if (iType == CHAN_TYPE_TEMP) {
                    String[] sbtypes = sbtype.split(":", -1);
                    if (sbtypes.length < 2) {
                        reportError(getName(), "subtype", subtypeS);
                    }
                    Integer iStp = tSubtypeMap.get(sbtypes[0]);
                    if (iStp == null) {
                        reportError(getName(), "subtype", subtypeS);
                    }
                    iSubtype = iStp;
                    if (iStp == MccUsb.STP_RTD) {
                        iStp = rtSubtypeMap.get(sbtypes[1]);
                    }
                    else {
                        iStp = tcSubtypeMap.get(sbtypes[1]);
                    }
                    if (iStp == null) {
                        reportError(getName(), "subtype", subtypeS);
                    }
                    iSubtype |= (iStp << 8);
                }
                else if (iType == CHAN_TYPE_VOLTS) {
                    Integer iStp = vSubtypeMap.get(sbtype);
                    if (iStp == null) {
                        reportError(getName(), "subtype", subtypeS);
                    }
                    iSubtype = iStp;
                }

                if (iType == CHAN_TYPE_TEMP) {
                    int mask = (iSubtype & 0xff) == MccUsb.STP_RTD
                                 ? refHdw.rtChanMask : refHdw.tcChanMask;
                    if (((1 << chan) & mask) == 0) {
                        reportError(getName(), "hw channel number", chan);
                    }
                }
                else if (iType == CHAN_TYPE_VOLTS) {
                    if (((1 << chan) & refHdw.vChanMask) == 0) {
                        reportError(getName(), "hw channel number", chan);
                    }
                }
                else if (iType == CHAN_TYPE_SWITCH) {
                    if (chan < 0 || chan >= N_MCC_DEV_CHANS * refHdw.nMcc) {
                        reportError(getName(), "hw channel number", chan);
                    }
                }
                iChan = chan;

                if (alarm >= N_ALARMS) {
                    reportError(getName(), "alarm number", alarm);
                }
                iAlarm = alarm;
            }
            catch (Exception e) {
            }

            type     = iType;
            subtype  = iSubtype;
            chan     = iChan;
            alarm    = iAlarm;
        }


       /**
        ***********************************************************************
        **
        **  Configures derived channel data
        **
        ***********************************************************************
        */
        void configDerived()
        {
            if (type != CHAN_TYPE_DIFF) return;
            String[] names = subtypeS.split(":", -1);
            boolean okay = false;
            if (names.length == 2) {
                int id0 = refHdw.findChannelId(names[0]);
                int id1 = refHdw.findChannelId(names[1]);
                if (id0 >= 0 && id1 >= 0) {
                    subtype = (id0 << 16) | id1;
                    okay = true;
                }
            }
            if (!okay) {
                log.error("Invalid subtype (" + subtypeS + ") for "
                            + getName());
                type = CHAN_TYPE_UNKNOWN;
            }
        }


       /**
        ***********************************************************************
        **
        **  Initializes the hardware sensor
        **
        ***********************************************************************
        */
        void initSensor()
        {
            if (type != CHAN_TYPE_SWITCH && type != CHAN_TYPE_TEMP
                  && type != CHAN_TYPE_VOLTS) {
                return;
            }

            MCC mccDev = refHdw.mccData[chan / N_MCC_DEV_CHANS];
            if (!mccDev.inited) return;
            int chn = chan % N_MCC_DEV_CHANS;
            MccUsb mcc = mccDev.mcc;
            try {
                if (type == CHAN_TYPE_TEMP) {
                    int sType = subtype & 0xff, tType = subtype >> 8;
                    mcc.setSensorType(chn, sType);
                    if (sType == MccUsb.STP_RTD) {
                        mcc.setThermConnType(chn, tType);
                        mcc.setExcitation(chn, MccUsb.CEX_210UA);
                        mcc.setGain(chn, MccUsb.GAIN_4X);
                        for (int j = 0; j < RTD_COEFFS.length; j++) {
                            mcc.setCoefficient(chn, j, RTD_COEFFS[j]);
                        }
                    }
                    else
                        mcc.setTCType(chn, tType);
                }
                else if (type == CHAN_TYPE_VOLTS) {
                    mcc.setGain(chn, subtype);
                    mcc.setVoltageConnType(chn, MccUsb.VCT_DIFFERENTIAL);
                }
                else if (type == CHAN_TYPE_SWITCH) {
                    mcc.dioConfigBit(chn, 1);
                }
            }
            catch (UsbException e) {
                log.error("Error configuring " + mccDev.type + " module: " + e);
                mccDev.inited = false;
            }
        }


       /**
        ***********************************************************************
        **
        **  Reads the hardware sensor
        **
        ***********************************************************************
        */
        void readSensor()
        {
            if (type == CHAN_TYPE_POWER) {
                Watts wuDev = refHdw.wuData[chan / N_WATTS_DEV_CHANS];
                value = wuDev.value[chan % N_WATTS_DEV_CHANS];
                return;
            }

            if (type != CHAN_TYPE_SWITCH && type != CHAN_TYPE_TEMP
                  && type != CHAN_TYPE_VOLTS) {
                return;
            }

            MCC mccDev = refHdw.mccData[chan / N_MCC_DEV_CHANS];
            if (!mccDev.inited) return;
            int chn = chan % N_MCC_DEV_CHANS;
            if (type == CHAN_TYPE_SWITCH) {
                try {
                    value = mccDev.mcc.dioInBit(chn);
                }
                catch (UsbException e) {
                    log.error("Error reading DIO line: " + e);
                }
            }
            else {
                try {
                    value = offset + scale * mccDev.mcc.adcIn(chn, 0);
                }
                catch (UsbException e) {
                    log.error("Error reading ADC: " + e);
                }
            }
        }


       /**
        ***********************************************************************
        **
        **  Calculates derived values
        **
        ***********************************************************************
        */
        void calcDerived()
        {
            if (type != CHAN_TYPE_DIFF) return;
            value = refHdw.chanDataHdw[subtype >> 16].getValue()
                      - refHdw.chanDataHdw[subtype & 0xffff].getValue();
        }


       /**
        ***********************************************************************
        **
        **  Gets the alarm number
        **
        ***********************************************************************
        */
        int getAlarm()
        {
            return alarm;
        }

    }


   /**
    ***************************************************************************
    **
    **  Main constructor.
    **
    ***************************************************************************
    */
    public RefrigTestHdw(String name, int tickMillis, String configName)
    {
        super(name, tickMillis, configName);
    }


   /**
    ***************************************************************************
    **
    **  Initializes the refrigeration channels with description data.
    **
    ***************************************************************************
    */
    @Override
    void initConfiguration()
    {
        Map<String, MCC> mccMap = getChildren(MCC.class);
        nMcc = mccMap.size();
        mccData = new MCC[nMcc];
        Iterator<MCC> iMcc = mccMap.values().iterator();
        for (int id = 0; id < nMcc; id++) {
            (mccData[id] = iMcc.next()).configure(this, id);
        }

        Map<String, Watts> wuMap = getChildren(Watts.class);
        nWatts = wuMap.size();
        wuData = new Watts[nWatts];
        Iterator<Watts> iWatts = wuMap.values().iterator();
        for (int id = 0; id < nWatts; id++) {
            wuData[id] = iWatts.next();
        }

        Map<String, ChannelHdw> chanMap = getChildren(ChannelHdw.class);
        nChan = chanMap.size();
        chanData = new ChannelHdw[nChan];
        chanDataHdw = (ChannelHdw[])chanData;
        Iterator<ChannelHdw> values = chanMap.values().iterator();
        for (int id = 0; id < nChan; id++) {
            (chanDataHdw[id] = values.next()).configure(this, id);
        }
        for (ChannelHdw ch : chanDataHdw) {
            ch.configDerived();
        }

    }


   /**
    ***************************************************************************
    **
    **  Initializes the refrigeration sensors.
    **
    ***************************************************************************
    */
    @Override
    void initSensors()
    {
        /*
        **  Perform basic configuration on the MCC DAQ boxes
        */
        for (MCC mcc : mccData) {
            mcc.initialize();
        }

        /*
        **  Perform basic configuration of the WattsUp? meters
        */
        for (Watts wu : wuData) {
            wu.initialize();
        }

        /*
        **  Configure all the hardware channels
        */
        for (ChannelHdw ch : chanDataHdw) {
            ch.initSensor();
        }
    }


   /**
    ***************************************************************************
    **
    **  Reads the sensor data.
    **
    ***************************************************************************
    */
    @Override
    void readSensors()
    {
        /*
        **  Read all the hardware channels
        */
        for (ChannelHdw ch : chanDataHdw) {
            ch.readSensor();
        }

        /*
        **  Compute derived values
        */
        for (ChannelHdw ch : chanDataHdw) {
            ch.calcDerived();
        }
    }


   /**
    ***************************************************************************
    **
    **  Turns the main power on or off.
    **
    ***************************************************************************
    */
    @Override
    void setMainPower()
    {
        MCC mccDev = mccData[mainPowerDev];
        if (mccDev.inited) {
            try {
                int on = (setState >> RefrigState.MAIN_POWER_STATE_BIT) & 1;
                mccDev.mcc.dioOutBit(MAIN_POWER_BIT, on);
            }
            catch (UsbException e) {
                log.error("Error setting DIO line: " + e);
            }
        }
    }
            

   /**
    ***************************************************************************
    **
    **  Turns the load power on or off.
    **
    ***************************************************************************
    */
    @Override
    void setLoadPower()
    {
        int on = (chanState >> RefrigState.LOAD_POWER_STATE_BIT) & 1;
        MCC mccDev = mccData[loadPowerDev];
        if (mccDev.inited) {
            try {
                mccDev.mcc.dioOutBit(LOAD_POWER_BIT, on);
            }
            catch (UsbException e) {
                log.error("Error setting DIO line: " + e);
            }
        }
        mccDev = mccData[loadPowerDev2];
        if (mccDev.inited) {
            try {
                mccDev.mcc.dioOutBit(LOAD_POWER_BIT, on ^ 1);
            }
            catch (UsbException e) {
                log.error("Error setting DIO line: " + e);
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Sets the front-panel display lights.
    **
    ***************************************************************************
    */
    @Override
    void setDisplay()
    {
        int alarms = 0;
        for (ChannelHdw ch : chanDataHdw) {
            int alarm = ch.getAlarm();
            if (alarm < 0) continue;
            if (((chanState >> ch.getId()) & 1) == 0) {
                alarms |= 1 << alarm;
            }
        }
        MCC mccDev = mccData[alarmsDev];
        if (mccDev.inited) {
            try {
                int value = mccDev.mcc.dioIn();
                mccDev.mcc.dioOut((value & ~ALARMS_MASK) | alarms);
            }
            catch (UsbException e) {
                log.error("Error setting DIO line: " + e);
            }
        }
        mccDev = mccData[alarmsDev2];
        if (mccDev.inited) {
            try {
                int value = mccDev.mcc.dioIn();
                mccDev.mcc.dioOut((value & ~ALARMS_MASK) | ~alarms);
            }
            catch (UsbException e) {
                log.error("Error setting DIO line: " + e);
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Finds the ID of a channel given its name.
    **
    ***************************************************************************
    */
    private int findChannelId(String name)
    {
        for (ChannelHdw ch : chanDataHdw) {
            if (ch.getName().equals(name)) return ch.getId();
        }

        return -1;
    }

}
