package org.lsst.ccs.subsystem.refrig;

import java.io.IOException;
import java.io.InputStream;
import java.net.ServerSocket;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import javax.usb.UsbException;
import org.lsst.ccs.drivers.mcc.MccUsb;
import org.lsst.ccs.drivers.wattsup.WattsUp;
import org.lsst.ccs.subsystem.refrig.data.RefrigState;
import org.lsst.ccs.utilities.conv.Convert;

/**
 ******************************************************************************
 **
 **  Implements the refrigeration long lines test modular subsystem
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class RefrigTestHdw extends RefrigTest implements WattsUp.Listener {

   /**
    ***************************************************************************
    **
    **  Private constants.
    **
    ***************************************************************************
    */
    private final static int
        MCC_DEV_TC_AI = 0,
        MCC_DEV_TC    = 1,
        MCC_DEV_TEMP  = 2,
        N_MCC_DEVS    = 3,
        N_MCC_DEV_CHANS = 8,
        N_MCC_CHANS     = N_MCC_DEV_CHANS * N_MCC_DEVS,
        TC_CHAN_MASK = (0x0f << (N_MCC_DEV_CHANS * MCC_DEV_TC_AI))
                         | (0xff << (N_MCC_DEV_CHANS * MCC_DEV_TC))
                         | (0xff << (N_MCC_DEV_CHANS * MCC_DEV_TEMP)),
        RT_CHAN_MASK = 0xff << (N_MCC_DEV_CHANS * MCC_DEV_TEMP),
        V_CHAN_MASK = 0xf0 << (N_MCC_DEV_CHANS * MCC_DEV_TC_AI),
        N_ALARMS    = 6,
        MAIN_POWER_DEV = MCC_DEV_TC_AI,
        MAIN_POWER_BIT = 7,
        LOAD_POWER_DEV = MCC_DEV_TC_AI,
        LOAD_POWER_DEV2 = MCC_DEV_TC,
        LOAD_POWER_BIT = 6,
        ALARMS_MASK = ((1 << N_ALARMS) - 1),
        LEDS_MASK = ALARMS_MASK | (1 << LOAD_POWER_BIT),
        LOAD_POWER_CHAN = 0,
        CMPR_POWER_CHAN = 1,
        CMPR_VOLTS_CHAN = 2,
        CHAN_TYPE_UNKNOWN = -1,
        CHAN_TYPE_TEMP    = 0,
        CHAN_TYPE_VOLTS   = 1,
        CHAN_TYPE_POWER   = 2,
        CHAN_TYPE_SWITCH  = 3,
        CHAN_TYPE_DIFF    = 4;

    private final static float[]
        RTD_COEFFS = {100.0F, 0.003908F, -5.8019E-7F, -4.2735E-12F};

    private final static String[] DEV_NAMES = new String[N_MCC_DEVS];
    static {
        DEV_NAMES[MCC_DEV_TC_AI] = "USB_TC_AI";
        DEV_NAMES[MCC_DEV_TC]    = "USB_TC";
        DEV_NAMES[MCC_DEV_TEMP]  = "USB_TEMP";
    }

    private final static int[] DEV_DIDS = new int[N_MCC_DEVS];
    static {
        DEV_DIDS[MCC_DEV_TC_AI] = MccUsb.USB_TC_AI_DID;
        DEV_DIDS[MCC_DEV_TC]    = MccUsb.USB_TC_DID;
        DEV_DIDS[MCC_DEV_TEMP]  = MccUsb.USB_TEMP_DID;
    }

   /**
    ***************************************************************************
    **
    **  Private fields.
    **
    ***************************************************************************
    */
    private ChannelHdw[] chanDataHdw;
    private MccUsb[] mcc = new MccUsb[N_MCC_DEVS];
    private WattsUp wtu;
    private float[] power = {-1F, -1F, -1F};
    private boolean[] mccInited = new boolean[N_MCC_DEVS];
    private boolean wattsInited;

   /**
    ***************************************************************************
    **
    **  Inner container class for channel hardware data.
    **
    ***************************************************************************
    */
    public static class ChannelHdw extends Channel {

       /**
        ***********************************************************************
        **
        **  Data fields
        **
        ***********************************************************************
        */
        String        typeS;      // Channel type string (TEMP, VOLTS, POWER...)
        String        subtypeS;   // Channel subtype string (extra config info)
        int           chan;       // Hardware channel for MCC channel
        int           alarm;      // Hardware alarm number
        float         offset;     // Value offset
        float         scale;      // Value scale
        int           type;       // Encoded channel type
        int           subtype;    // Encoded channel subtype
        RefrigTestHdw refHdw;     // Associated refrigeration object

       /**
        ***********************************************************************
        **
        **  Private lookup maps
        **
        ***********************************************************************
        */
        private final static HashMap<String,Integer> typeMap;
        static {
            typeMap = new HashMap<String,Integer>();
            typeMap.put("TEMP",   CHAN_TYPE_TEMP);
            typeMap.put("VOLTS",  CHAN_TYPE_VOLTS);
            typeMap.put("POWER",  CHAN_TYPE_POWER);
            typeMap.put("SWITCH", CHAN_TYPE_SWITCH);
            typeMap.put("DIFF",   CHAN_TYPE_DIFF);
        }

        private final static HashMap<String,Integer> tSubtypeMap;
        static {
            tSubtypeMap = new HashMap<String,Integer>();
            tSubtypeMap.put("TC",  MccUsb.STP_THERMOCOUPLE);
            tSubtypeMap.put("RTD", MccUsb.STP_RTD);
        }

        private final static HashMap<String,Integer> tcSubtypeMap;
        static {
            tcSubtypeMap = new HashMap<String,Integer>();
            tcSubtypeMap.put("J", MccUsb.TC_TYPE_J);
            tcSubtypeMap.put("K", MccUsb.TC_TYPE_K);
            tcSubtypeMap.put("T", MccUsb.TC_TYPE_T);
            tcSubtypeMap.put("E", MccUsb.TC_TYPE_E);
            tcSubtypeMap.put("R", MccUsb.TC_TYPE_R);
            tcSubtypeMap.put("S", MccUsb.TC_TYPE_S);
            tcSubtypeMap.put("B", MccUsb.TC_TYPE_B);
            tcSubtypeMap.put("N", MccUsb.TC_TYPE_N);
        }

        private final static HashMap<String,Integer> rtSubtypeMap;
        static {
            rtSubtypeMap = new HashMap<String,Integer>();
            rtSubtypeMap.put("2WIRE1", MccUsb.TCT_2WIRE_1SENSOR);
            rtSubtypeMap.put("2WIRE2", MccUsb.TCT_2WIRE_2SENSOR);
            rtSubtypeMap.put("3WIRE",  MccUsb.TCT_3WIRE);
            rtSubtypeMap.put("4WIRE",  MccUsb.TCT_4WIRE);
        }

        private final static HashMap<String,Integer> vSubtypeMap;
        static {
            vSubtypeMap = new HashMap<String,Integer>();
            vSubtypeMap.put("10V",   MccUsb.RANGE_10V);
            vSubtypeMap.put("5V",    MccUsb.RANGE_5V);
            vSubtypeMap.put("2.5V",  MccUsb.RANGE_2_5V);
            vSubtypeMap.put("1.25V", MccUsb.RANGE_1_25V);
        }


       /**
        ***********************************************************************
        **
        **  Constructor
        **
        ***********************************************************************
        */
        public ChannelHdw(String desc, String units,
                          String loCheck, double limitLo, double deadbandLo,
                          String hiCheck, double limitHi, double deadbandHi,
                          String type, String subtype,
                          int chan, int alarm, double offset, double scale)
        {
            super(desc, units, loCheck, limitLo, deadbandLo, hiCheck, limitHi,
                  deadbandHi);
            this.typeS    = type;
            this.subtypeS = subtype;
            this.chan     = chan;
            this.alarm    = alarm;
            this.offset   = (float)offset;
            this.scale    = (float)scale;
        }


       /**
        ***********************************************************************
        **
        **  Configures direct channel data
        **
        ***********************************************************************
        */
        void configure(RefrigTestHdw ref, int id)
        {
            super.configure(ref, id);
            refHdw = ref;
            int iType = CHAN_TYPE_UNKNOWN, iSubtype = -1, iChan = -1,
                iAlarm = -1;
            try {
                Integer iTyp = typeMap.get(typeS.toUpperCase());
                if (iTyp == null) {
                    reportError(getName(), "type", typeS);
                }
                iType = iTyp;

                String sbtype = subtypeS.toUpperCase();
                if (iType == CHAN_TYPE_TEMP) {
                    String[] sbtypes = sbtype.split(":", -1);
                    if (sbtypes.length < 2) {
                        reportError(getName(), "subtype", subtypeS);
                    }
                    Integer iStp = tSubtypeMap.get(sbtypes[0]);
                    if (iStp == null) {
                        reportError(getName(), "subtype", subtypeS);
                    }
                    iSubtype = iStp;
                    if (iStp == MccUsb.STP_RTD) {
                        iStp = rtSubtypeMap.get(sbtypes[1]);
                    }
                    else {
                        iStp = tcSubtypeMap.get(sbtypes[1]);
                    }
                    if (iStp == null) {
                        reportError(getName(), "subtype", subtypeS);
                    }
                    iSubtype |= (iStp << 8);
                }
                else if (iType == CHAN_TYPE_VOLTS) {
                    Integer iStp = vSubtypeMap.get(sbtype);
                    if (iStp == null) {
                        reportError(getName(), "subtype", subtypeS);
                    }
                    iSubtype = iStp;
                }

                if (iType == CHAN_TYPE_TEMP) {
                    int mask = (iSubtype & 0xff) == MccUsb.STP_RTD
                                 ? RT_CHAN_MASK : TC_CHAN_MASK;
                    if (((1 << chan) & mask) == 0) {
                        reportError(getName(), "hw channel number", chan);
                    }
                }
                else if (iType == CHAN_TYPE_VOLTS) {
                    if (((1 << chan) & V_CHAN_MASK) == 0) {
                        reportError(getName(), "hw channel number", chan);
                    }
                }
                else if (iType == CHAN_TYPE_SWITCH) {
                    if (chan < 0 || chan >= N_MCC_CHANS) {
                        reportError(getName(), "hw channel number", chan);
                    }
                }
                iChan = chan;

                if (alarm >= N_ALARMS) {
                    reportError(getName(), "alarm number", alarm);
                }
                iAlarm = alarm;
            }
            catch (Exception e) {
            }

            type     = iType;
            subtype  = iSubtype;
            chan     = iChan;
            alarm    = iAlarm;
        }


       /**
        ***********************************************************************
        **
        **  Configures derived channel data
        **
        ***********************************************************************
        */
        void configDerived()
        {
            if (type != CHAN_TYPE_DIFF) return;
            String[] names = subtypeS.split(":", -1);
            boolean okay = false;
            if (names.length == 2) {
                int id0 = refHdw.findChannelId(names[0]);
                int id1 = refHdw.findChannelId(names[1]);
                if (id0 >= 0 && id1 >= 0) {
                    subtype = (id0 << 16) | id1;
                    okay = true;
                }
            }
            if (!okay) {
                log.error("Invalid subtype (" + subtypeS + ") for "
                            + getName());
                type = CHAN_TYPE_UNKNOWN;
            }
        }


       /**
        ***********************************************************************
        **
        **  Initializes the hardware sensor
        **
        ***********************************************************************
        */
        void initSensor()
        {
            if (type != CHAN_TYPE_SWITCH && type != CHAN_TYPE_TEMP
                  && type != CHAN_TYPE_VOLTS) {
                return;
            }

            int dev = chan / N_MCC_DEV_CHANS;
            if (!refHdw.mccInited[dev]) return;
            int chn = chan % N_MCC_DEV_CHANS;
            MccUsb mcc = refHdw.mcc[dev];
            try {
                if (type == CHAN_TYPE_TEMP) {
                    int sType = subtype & 0xff, tType = subtype >> 8;
                    mcc.setSensorType(chn, sType);
                    if (sType == MccUsb.STP_RTD) {
                        mcc.setThermConnType(chn, tType);
                        mcc.setExcitation(chn, MccUsb.CEX_210UA);
                        mcc.setGain(chn, MccUsb.GAIN_4X);
                        for (int j = 0; j < RTD_COEFFS.length; j++) {
                            mcc.setCoefficient(chn, j, RTD_COEFFS[j]);
                        }
                    }
                    else
                        mcc.setTCType(chn, tType);
                }
                else if (type == CHAN_TYPE_VOLTS) {
                    mcc.setGain(chn, subtype);
                    mcc.setVoltageConnType(chn, MccUsb.VCT_DIFFERENTIAL);
                }
                else if (type == CHAN_TYPE_SWITCH) {
                    mcc.dioConfigBit(chn, 1);
                }
            }
            catch (UsbException e) {
                log.error("Error configuring " + DEV_NAMES[dev] + " module: "
                            + e);
                refHdw.mccInited[dev] = false;
            }
        }


       /**
        ***********************************************************************
        **
        **  Reads the hardware sensor
        **
        ***********************************************************************
        */
        void readSensor()
        {
            if (type == CHAN_TYPE_POWER) {
                value = refHdw.power[chan];
                return;
            }

            if (type != CHAN_TYPE_SWITCH && type != CHAN_TYPE_TEMP
                  && type != CHAN_TYPE_VOLTS) {
                return;
            }

            int dev = chan / N_MCC_DEV_CHANS, chn = chan % N_MCC_DEV_CHANS;
            if (type == CHAN_TYPE_SWITCH) {
                try {
                    value = refHdw.mcc[dev].dioInBit(chn);
                }
                catch (UsbException e) {
                    if (refHdw.mccInited[dev]) {
                        log.error("Error reading DIO line: " + e);
                    }
                }
            }
            else {
                try {
                    value = offset + scale * refHdw.mcc[dev].adcIn(chn, 0);
                }
                catch (UsbException e) {
                    if (refHdw.mccInited[dev]) {
                        log.error("Error reading ADC: " + e);
                    }
                }
            }
        }


       /**
        ***********************************************************************
        **
        **  Calculates derived values
        **
        ***********************************************************************
        */
        void calcDerived()
        {
            if (type != CHAN_TYPE_DIFF) return;
            value = refHdw.chanDataHdw[subtype >> 16].getValue()
                      - refHdw.chanDataHdw[subtype & 0xffff].getValue();
        }


       /**
        ***********************************************************************
        **
        **  Gets the alarm number
        **
        ***********************************************************************
        */
        int getAlarm()
        {
            return alarm;
        }

    }


   /**
    ***************************************************************************
    **
    **  Inner class implementing task to obtain remote WattsUp meter data
    **
    ***************************************************************************
    */
    private class RemotePower extends Thread {

        @Override
        public void run()
        {
            ServerSocket server;
            try {
                server = new ServerSocket(RefrigPower.POWER_PORT, 1);
            }
            catch (IOException e) {
                log.error("Error creating remote power listener: " + e);
                return;
            }

            byte[] data = new byte[12];
            InputStream netInp;

            while (true) {
                try {
                    netInp = server.accept().getInputStream();
                }
                catch (IOException e) {
                    log.error("Error creating remote power socket: " + e);
                    continue;
                }
                while (true) {
                    int nRead;
                    try {
                        nRead = netInp.read(data);
                    }
                    catch (IOException e) {
                        log.error("Error reading remote power data: " + e);
                        nRead = -2;
                    }
                    if (nRead < 0) {
                        try {
                            netInp.close();
                        }
                        catch (IOException e) {
                        }
                        power[CMPR_POWER_CHAN] = -1F;
                        power[CMPR_VOLTS_CHAN] = -1F;
                        break;
                    }
                    if (nRead < data.length) {
                        continue;
                    }
                    if (Convert.bytesToInt(data, 0) != RefrigPower.POWER_ID) {
                        continue;
                    }
                    power[CMPR_POWER_CHAN] = Convert.bytesToFloat(data, 4);
                    power[CMPR_VOLTS_CHAN] = Convert.bytesToFloat(data, 8);
                }
            }
        }

    }


   /**
    ***************************************************************************
    **
    **  Main constructor.
    **
    ***************************************************************************
    */
    public RefrigTestHdw(String name, int tickMillis, String configName)
    {
        super(name, tickMillis, configName);
    }


   /**
    ***************************************************************************
    **
    **  Initializes the refrigeration channels with description data.
    **
    ***************************************************************************
    */
    @Override
    void initConfiguration()
    {
        Map<String, ChannelHdw> chData = getChildren(ChannelHdw.class);
        nChan = chData.size();
        chanData = new ChannelHdw[nChan];
        chanDataHdw = (ChannelHdw[])chanData;
        Iterator<ChannelHdw> values = chData.values().iterator();
        for (int id = 0; id < nChan; id++) {
            (chanDataHdw[id] = values.next()).configure(this, id);
        }
        for (ChannelHdw ch : chanDataHdw) {
            ch.configDerived();
        }
    }


   /**
    ***************************************************************************
    **
    **  Initializes the refrigeration sensors.
    **
    ***************************************************************************
    */
    @Override
    void initSensors()
    {
        /*
        **  Perform basic configuration on the MCC DAQ boxes
        */
        for (int dev = 0; dev < N_MCC_DEVS; dev++) {
            try {
                mcc[dev] = new MccUsb();
                mcc[dev].open(DEV_DIDS[dev], null, true);
                mcc[dev].blink();
                mcc[dev].dioConfig(0);
                mcc[dev].dioOut(0);
                if (dev == MCC_DEV_TC_AI) {
                    for (int j = 0; j < N_ALARMS; j++) {
                        mcc[dev].configAlarm(j, 0, 0, 0f, 0f);
                    }
                }
                else if (dev == MCC_DEV_TEMP) {
                    for (int j = 0; j < N_MCC_DEV_CHANS; j += 2) {
                        mcc[dev].setSensorType(j, MccUsb.STP_DISABLED);
                    }
                }
                mccInited[dev] = true;
            }
            catch (UsbException e) {
                log.error("Error initializing " + DEV_NAMES[dev] + " module: "
                            + e);
            }
        }

        /*
        **  Perform basic configuration of the WattsUp? meters
        */
        try {
            wtu = new WattsUp();
            wtu.addListener(this);
            wtu.open();
            wattsInited = true;
            power[LOAD_POWER_CHAN] = 0F;
        }
        catch (Exception e) {
            log.error("Cannot access WattsUp meter: " + e);
        }

        /*
        **  Configure all the hardware channels
        */
        for (ChannelHdw ch : chanDataHdw) {
            ch.initSensor();
        }

        /*
        **  Start the thread that receives remote power data
        */
        (new RemotePower()).start();
    }


   /**
    ***************************************************************************
    **
    **  Reads the sensor data.
    **
    ***************************************************************************
    */
    @Override
    void readSensors()
    {
        /*
        **  Read all the hardware channels
        */
        for (ChannelHdw ch : chanDataHdw) {
            ch.readSensor();
        }

        /*
        **  Compute derived values
        */
        for (ChannelHdw ch : chanDataHdw) {
            ch.calcDerived();
        }
    }


   /**
    ***************************************************************************
    **
    **  Turns the main power on or off.
    **
    ***************************************************************************
    */
    @Override
    void setMainPower()
    {
        try {
            int on = (setState >> RefrigState.MAIN_POWER_STATE_BIT) & 1;
            mcc[MAIN_POWER_DEV].dioOutBit(MAIN_POWER_BIT, on);
        }
        catch (UsbException e) {
            if (mccInited[MAIN_POWER_DEV]) {
                log.error("Error setting DIO line: " + e);
            }
        }
    }
            

   /**
    ***************************************************************************
    **
    **  Turns the load power on or off.
    **
    ***************************************************************************
    */
    @Override
    void setLoadPower()
    {
        int on = (chanState >> RefrigState.LOAD_POWER_STATE_BIT) & 1;
        try {
            mcc[LOAD_POWER_DEV].dioOutBit(LOAD_POWER_BIT, on);
        }
        catch (UsbException e) {
            if (mccInited[LOAD_POWER_DEV]) {
                log.error("Error setting DIO line: " + e);
            }
        }
        try {
            mcc[LOAD_POWER_DEV2].dioOutBit(LOAD_POWER_BIT, on ^ 1);
        }
        catch (UsbException e) {
            if (mccInited[LOAD_POWER_DEV2]) {
                log.error("Error setting DIO line: " + e);
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Sets the front-panel display lights.
    **
    ***************************************************************************
    */
    @Override
    void setDisplay()
    {
        int alarms = 0;
        for (ChannelHdw ch : chanDataHdw) {
            int alarm = ch.getAlarm();
            if (alarm < 0) continue;
            if (((chanState >> ch.getId()) & 1) == 0) {
                alarms |= 1 << alarm;
            }
        }
        try {
            int value = mcc[MCC_DEV_TC_AI].dioIn();
            mcc[MCC_DEV_TC_AI].dioOut((value & ~ALARMS_MASK) | alarms);
        }
        catch (UsbException e) {
            if (mccInited[MCC_DEV_TC_AI]) {
                log.error("Error setting DIO line: " + e);
            }
        }
        try {
            int value = mcc[MCC_DEV_TC].dioIn();
            mcc[MCC_DEV_TC].dioOut((value & ~ALARMS_MASK) | ~alarms);
        }
        catch (UsbException e) {
            if (mccInited[MCC_DEV_TC]) {
                log.error("Error setting DIO line: " + e);
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Changes the WattsUp? meter powered state.
    **
    **  <p>
    **  Performs meter setup upon power-on.
    **
    **  @param  on  Whether the meter is powered on or not
    **
    ***************************************************************************
    */
    @Override
    public void setPowered(boolean on)
    {
        if (on) {
            try {
                wtu.setLoggedFields(1 << WattsUp.FLD_WATTS);
                wtu.setExternalLogging(1);
            }
            catch (Exception e) {
                log.error("Cannot initialize WattsUp meter: " + e);
            }
        }
        else {
            power[LOAD_POWER_CHAN] = 0F;
        }
    }


   /**
    ***************************************************************************
    **
    **  Changes the WattsUp? meter open state.
    **
    ***************************************************************************
    */
    @Override
    public void setClosed()
    {
        power[LOAD_POWER_CHAN] = -1F;
        wattsInited = false;
    }


   /**
    ***************************************************************************
    **
    **  Receives data periodically from the WattsUp? meter.
    **
    **  @param  data  The array of data from the meter.
    **
    ***************************************************************************
    */
    @Override
    public void processData(float[] data)
    {
        power[LOAD_POWER_CHAN] = data[WattsUp.FLD_WATTS];
    }


   /**
    ***************************************************************************
    **
    **  Finds the ID of a channel given its name.
    **
    ***************************************************************************
    */
    private int findChannelId(String name)
    {
        for (ChannelHdw ch : chanDataHdw) {
            if (ch.getName().equals(name)) return ch.getId();
        }

        return -1;
    }

}
