package org.lsst.ccs.subsystem.refrig;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Timer;
import java.util.TimerTask;
import org.lsst.ccs.bus.ValueNotification;
import org.lsst.ccs.framework.ConfigurableComponent;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.framework.annotations.BusCommand;
import org.lsst.ccs.framework.annotations.ConfigChanger;
import org.lsst.ccs.subsystem.refrig.data.RefrigChannel;
import org.lsst.ccs.subsystem.refrig.data.RefrigFullState;
import org.lsst.ccs.subsystem.refrig.data.RefrigState;
import org.lsst.ccs.subsystem.refrig.status.RefrigStateStatus;

/**
 ******************************************************************************
 **
 **  Implements the refrigeration long lines test modular subsystem
 **
 **  <p>
 **  This is an abstract class implementing the interactions with the CCS
 **  buses, but deferring the hardware interaction, real or simulated, to
 **  sub-classes.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public abstract class RefrigTest extends Module {

   /**
    ***************************************************************************
    **
    **  Package constants
    **
    ***************************************************************************
    */
    final static int
        MAX_CHANS = 16;
    final static long
        UPDATE_PERIOD = 1000;

   /**
    ***************************************************************************
    **
    **  Package fields
    **
    ***************************************************************************
    */
    String configName = "test";
    Channel[] chanData;
    int nChan, loLimChange = 0, hiLimChange = 0, setState = 0, chanState = 0;
    boolean running = false;

   /**
    ***************************************************************************
    **
    **  Inner container class for channel parameters
    **
    ***************************************************************************
    */
    public static class Channel extends ConfigurableComponent {

       /**
        ***********************************************************************
        **
        **  Data fields
        **
        ***********************************************************************
        */
        String     description;   // Channel description
        String     units;         // Units name
        String     loCheckS;      // Low limit check option string
        String     hiCheckS;      // High limit check option string
        int        loCheck;       // Low limit check option
        int        hiCheck;       // High limit check option
        float      deadbandLo;    // Low limit dead-band
        float      deadbandHi;    // High limit dead-band
        float      limitLo;       // Low limit value
        float      limitHi;       // High limit value
        float      value;         // Current value read from channel
        boolean    trippedLo;     // Whether low limit has caused a trip
        boolean    trippedHi;     // Whether high limit has caused a trip
        int        id;            // External ID
        RefrigTest ref;           // Associated refrigeration object

       /**
        ***********************************************************************
        **
        **  Package constants
        **
        ***********************************************************************
        */
        final static int
            LIMIT_CHECK_NONE      = 0,   // Don't check limit
            LIMIT_CHECK_FLAG      = 1,   // Flag if limit exceeded
            LIMIT_CHECK_MAIN_TRIP = 2,   // Trip main power if limit exceeded
            LIMIT_CHECK_LOAD_TRIP = 3;   // Trip load power if limit exceeded

       /**
        ***********************************************************************
        **
        **  Private lookup maps
        **
        ***********************************************************************
        */
        private final static HashMap<String,Integer> checkMap;
        static {
            checkMap = new HashMap<String,Integer>();
            checkMap.put("NONE", LIMIT_CHECK_NONE);
            checkMap.put("FLAG", LIMIT_CHECK_FLAG);
            checkMap.put("MTRIP", LIMIT_CHECK_MAIN_TRIP);
            checkMap.put("LTRIP", LIMIT_CHECK_LOAD_TRIP);
        }


       /**
        ***********************************************************************
        **
        **  Constructor
        **
        ***********************************************************************
        */
        public Channel(String desc, String units,
                       String loCheck, double limitLo, double deadbandLo,
                       String hiCheck, double limitHi, double deadbandHi)
        {
            this.description = desc;
            this.units = units;
            this.loCheckS = loCheck;
            this.hiCheckS = hiCheck;
            this.limitLo = (float)limitLo;
            this.deadbandLo = (float)deadbandLo;
            this.limitHi = (float)limitHi;
            this.deadbandHi = (float)deadbandHi;
        }


       /**
        ***********************************************************************
        **
        **  Configures channel description
        **
        ***********************************************************************
        */
        void configure(RefrigTest ref, int id)
        {
            this.ref = ref;
            this.id = id;
            int iLoCheck = LIMIT_CHECK_FLAG, iHiCheck = LIMIT_CHECK_FLAG;
            try {
                Integer iCheck = checkMap.get(loCheckS.toUpperCase());
                if (iCheck == null)
                    reportError(getName(), "locheck", loCheck);
                iLoCheck = iCheck;
                iCheck = checkMap.get(hiCheckS.toUpperCase());
                if (iCheck == null) {
                    reportError(getName(), "hicheck", hiCheck);
                }
                iHiCheck = iCheck;
            }
            catch (Exception e) {
            }

            loCheck = iLoCheck;
            hiCheck = iHiCheck;
        }


       /**
        ***********************************************************************
        **
        **  Sets the low limit
        **
        ***********************************************************************
        */
        @ConfigChanger
        public void setLimitLo(double limit)
        {
            float fLimit = (float)limit;
            if (fLimit == limitLo) return;
            limitLo = fLimit;
            ref.loLimChange |= 1 << id;
            ref.publishState();
            ref.getSubsystem().publishMetaData(getName(), "alarmLow",
                                               String.valueOf(fLimit));
        }


       /**
        ***********************************************************************
        **
        **  Sets the high limit
        **
        ***********************************************************************
        */
        @ConfigChanger
        public void setLimitHi(double limit)
        {
            float fLimit = (float)limit;
            if (fLimit == limitHi) return;
            limitHi = fLimit;
            ref.hiLimChange |= 1 << id;
            ref.publishState();
            ref.getSubsystem().publishMetaData(getName(), "alarmHigh",
                                               String.valueOf(fLimit));
        }


       /**
        ***********************************************************************
        **
        **  Checks whether current value is within limits
        **
        ***********************************************************************
        */
        int checkLimits(int cState, int setState)
        {
            boolean ok = false;
            if (value < limitLo) {
                if (loCheck == Channel.LIMIT_CHECK_NONE) {
                    ok = true;
                }
                else if (loCheck == Channel.LIMIT_CHECK_MAIN_TRIP) {
                    cState |= RefrigState.MAIN_TRIPPED_STATE;
                }
                else if (loCheck == Channel.LIMIT_CHECK_LOAD_TRIP) {
                    if ((setState & RefrigState.LOAD_TRIP_ENAB_STATE) != 0) {
                        cState &= ~RefrigState.LOAD_POWER_STATE;
                        trippedLo = true;
                    }
                }
            }
            else if (value > limitHi) {
                if (hiCheck == Channel.LIMIT_CHECK_NONE) {
                    ok = true;
                }
                else if (hiCheck == Channel.LIMIT_CHECK_MAIN_TRIP) {
                    cState |= RefrigState.MAIN_TRIPPED_STATE;
                }
                else if (hiCheck == Channel.LIMIT_CHECK_LOAD_TRIP) {
                    if ((setState & RefrigState.LOAD_TRIP_ENAB_STATE) != 0) {
                        cState &= ~RefrigState.LOAD_POWER_STATE;
                        trippedHi = true;
                    }
                }
            }
            else {
                ok = true;
                if (loCheck == Channel.LIMIT_CHECK_LOAD_TRIP) {
                    boolean tripped = false;
                    if ((setState & RefrigState.LOAD_TRIP_ENAB_STATE) != 0) {
                        if (trippedLo && value < limitLo + deadbandLo) {
                            cState &= ~RefrigState.LOAD_POWER_STATE;
                            tripped = true;
                        }
                    }
                    trippedLo = tripped;
                }
                if (hiCheck == Channel.LIMIT_CHECK_LOAD_TRIP) {
                    boolean tripped = false;
                    if ((setState & RefrigState.LOAD_TRIP_ENAB_STATE) != 0) {
                        if (trippedHi && value > limitHi - deadbandHi) {
                            cState &= ~RefrigState.LOAD_POWER_STATE;
                            tripped = true;
                        }
                    }
                    trippedHi = tripped;
                }
            }
            if (ok) cState |= (1 << id);

            return cState;
        }


       /**
        ***********************************************************************
        **
        **  Gets the ID
        **
        ***********************************************************************
        */
        int getId()
        {
            return id;
        }


       /**
        ***********************************************************************
        **
        **  Gets the current value
        **
        ***********************************************************************
        */
        float getValue()
        {
            return value;
        }

    }

   /**
    ***************************************************************************
    **
    **  Inner class to implement a timer task to update the channel state
    **  in a timely manner.
    **
    ***************************************************************************
    */
    private class UpdateState extends TimerTask {

        @Override
        public void run()
        {
            /*
            **  Read the sensor values
            */
            readSensors();

            /*
            **  Check the limits and set the channel state
            */
            checkLimits();

            /*
            **  Update the hardware display
            */
            setDisplay();
        }

    }

                
   /**
    ***************************************************************************
    **
    **  Main constructor
    **
    ***************************************************************************
    */
    public RefrigTest(String name, int tickMillis, String configName)
    {
        super(name, tickMillis);
        this.configName = configName;
    }


   /**
    ***************************************************************************
    **
    **  Initializes the refrigeration subsystem
    **
    ***************************************************************************
    */
    @Override
    public void initModule()
    {
        /*
        **  Initialize all configuration data
        */
        initConfiguration();

        /*
        **  Initialize the hardware
        */
        initSensors();

        /*
        **  Start the timer task which updates the channel state periodically
        */
        (new Timer()).schedule(new UpdateState(), 0L, UPDATE_PERIOD);
    }


   /**
    ***************************************************************************
    **
    **  Performs periodic trending data broadcast
    **
    ***************************************************************************
    */
    @Override
    public void tick()
    {
        /*
        **  Broadcast the state if first time
        */
        if (!running) {
            System.out.println("Refrigeration test system started");
            publishState();    // For any GUIs
            publishLimits();   // For GUIs and the trending database
            running = true;
        }

        /*
        **  Broadcast the trending data
        */
        long timeStamp = System.currentTimeMillis();
        ArrayList<ValueNotification> trendingValues
          = new ArrayList<ValueNotification>();
        for (Channel ch : chanData) {
            trendingValues.add(new ValueNotification(ch.getName(), ch.value,
                                                     timeStamp));
        }
        trendingValues.add(new ValueNotification("state", getState(),
                                                 timeStamp));
        publishData(trendingValues);
    }


   /**
    ***************************************************************************
    **
    **  Sets the tick period
    **
    **  <p>
    **  Overrides the method in Module in order to publish a status update.
    **
    **  @param  value  The tick period (milliseconds) to set.
    **
    ***************************************************************************
    */
    @Override
    @BusCommand
    public void setTickMillis(int value)
    {
        super.setTickMillis(value);
        if (running) publishState();
    }


   /**
    ***************************************************************************
    **
    **  Sets the main power enabled state on or off
    **
    **  @param  value  The enabled state value to set: 0 = off, ~0 = on.
    **
    ***************************************************************************
    */
    @BusCommand
    public void setMainPowerEnable(int value)
    {
        setMainPowerEnable(value, true);
    }


   /**
    ***************************************************************************
    **
    **  Sets the main power enabled state on or off
    **
    **  @param  value    The enabled state value to set: 0 = off, ~0 = on.
    **
    **  @param  publish  If true, publish the new state on the status bus.
    **
    ***************************************************************************
    */
    @BusCommand
    public void setMainPowerEnable(int value, boolean publish)
    {
        if (value != 0) {
            if ((chanState & RefrigState.MAIN_TRIPPED_STATE) == 0)
                setState |= RefrigState.MAIN_POWER_STATE;
        }
        else
            setState &= ~RefrigState.MAIN_POWER_STATE;
        setMainPower();
        if (publish) publishState();
    }


   /**
    ***************************************************************************
    **
    **  Sets the load power trip enabled state on or off
    **
    **  @param  value  The load trip enabled state value to set: 0 = off;
    **                 ~0 = on.
    **
    ***************************************************************************
    */
    @BusCommand
    public void setLoadTripEnable(int value)
    {
        if (value != 0)
            setState |= RefrigState.LOAD_TRIP_ENAB_STATE;
        else
            setState &= ~RefrigState.LOAD_TRIP_ENAB_STATE;
        publishState();
    }


   /**
    ***************************************************************************
    **
    **  Sets the low limit for a channel
    **
    **  @param  id     The channel ID
    **
    **  @param  limit  The low limit value.
    **
    ***************************************************************************
    */
    @BusCommand
    public void setLowLimit(int id, double limit) throws Exception
    {
        chanData[id].change("limitLo", limit);
    }


   /**
    ***************************************************************************
    **
    **  Sets the high limit for a channel
    **
    **  @param  id     The channel ID
    **
    **  @param  limit  The high limit value.
    **
    ***************************************************************************
    */
    @BusCommand
    public void setHighLimit(int id, double limit) throws Exception
    {
        chanData[id].change("limitHi", limit);
    }


   /**
    ***************************************************************************
    **
    **  Saves the configuration data
    **
    ***************************************************************************
    */
    @BusCommand
    public void saveConfiguration() throws IOException
    {
        register(configName);
        loLimChange = 0;
        hiLimChange = 0;
        publishState();         // Must always do this
    }


   /**
    ***************************************************************************
    **
    **  Gets the full state of the refrigeration module
    **
    **  <p>
    **  This is intended to be called by GUIs during initialization
    **
    ***************************************************************************
    */
    @BusCommand
    public RefrigFullState getFullState()
    {
        RefrigFullState status = new RefrigFullState();
        for (Channel ch : chanData) {
            status.addChannel(new RefrigChannel(ch.getName(), ch.description,
                                                ch.units, ch.limitLo,
                                                ch.limitHi, ch.value));
        }
        status.setRefrigState(new RefrigState(getState(), loLimChange,
                                              hiLimChange, getTickMillis()));
        return status;
    }    


   /**
    ***************************************************************************
    **
    **  Gets the operating state word
    **
    **  @return  The current value of the operating state
    **
    ***************************************************************************
    */
    int getState()
    {
        return setState | chanState;
    }


   /**
    ***************************************************************************
    **
    **  Publishes the state of the refrigeration module
    **
    **  <p>
    **  This is intended to be called whenever any element of the state is
    **  changed.
    **
    ***************************************************************************
    */
    void publishState()
    {
        RefrigState refrigState = new RefrigState(getState(), loLimChange,
                                                  hiLimChange, getTickMillis());
        sendToStatus(new RefrigStateStatus(refrigState));
    }    


   /**
    ***************************************************************************
    **
    **  Publishes all the limit values of the refrigeration module
    **
    **  <p>
    **  This is intended to be called at startup time.
    **
    ***************************************************************************
    */
    void publishLimits()
    {
        for (Channel ch : chanData) {
            getSubsystem().publishMetaData(ch.getName(), "alarmLow",
                                           String.valueOf(ch.limitLo));
            getSubsystem().publishMetaData(ch.getName(), "alarmHigh",
                                           String.valueOf(ch.limitHi));
        }
    }    


   /**
    ***************************************************************************
    **
    **  Checks whether all current channel values are within limits
    **
    ***************************************************************************
    */
    void checkLimits()
    {
        /*
        **  Generate the current channel state
        */
        int cState = RefrigState.LOAD_POWER_STATE;
        for (Channel ch : chanData) {
            cState = ch.checkLimits(cState, setState);
        }
                
        /*
        **  If state changed
        **    Turn main power off if it was tripped off
        **    Turn load power off or on appropriately
        **    If running
        **      Publish state
        **      Log channel data
        */
        cState ^= chanState;
        chanState ^= cState;
        if (cState != 0) {
            int tr;
            if ((cState & RefrigState.MAIN_TRIPPED_STATE) != 0) {
                tr = (chanState >> RefrigState.MAIN_TRIPPED_STATE_BIT) & 1;
                if (tr != 0) setMainPowerEnable(0, false);
            }
            if ((cState & RefrigState.LOAD_POWER_STATE) != 0)
                setLoadPower();
            if (running) {
                publishState();
                tick();
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Reports parameter errors
    **
    ***************************************************************************
    */
    static void reportError(String cName, String pName, Object pValue)
        throws Exception
    {
        log.error("Invalid " + pName + " (" + pValue + ") for " + cName);
        throw new Exception();
    }


   /**
    ***************************************************************************
    **
    **  Initializes the refrigeration configuration.
    **
    ***************************************************************************
    */
    abstract void initConfiguration();


   /**
    ***************************************************************************
    **
    **  Initializes the refrigeration sensors.
    **
    ***************************************************************************
    */
    abstract void initSensors();


   /**
    ***************************************************************************
    **
    **  Reads the sensor data.
    **
    ***************************************************************************
    */
    abstract void readSensors();


   /**
    ***************************************************************************
    **
    **  Turns the main power on or off, according to the set value.
    **
    ***************************************************************************
    */
    abstract void setMainPower();
            

   /**
    ***************************************************************************
    **
    **  Turns the load power on or off, according to the set value.
    **
    ***************************************************************************
    */
    abstract void setLoadPower();


   /**
    ***************************************************************************
    **
    **  Sets the hardware display, according to the current state
    **
    ***************************************************************************
    */
    abstract void setDisplay();
    
}
