package org.lsst.ccs.subsystem.refrig;

import java.io.Serializable;
import org.lsst.ccs.subsystem.refrig.data.RefrigState;

/**
 ******************************************************************************
 **
 **  Implements the simulated refrigeration long lines test modular subsystem
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class RefrigTestSim extends RefrigTest {

   /**
    ***************************************************************************
    **
    **  Private fields.
    **
    ***************************************************************************
    */
    private final static float LOAD_POWER_FCTR = 0.0006667F;
    private final ChannelSim[] chanDataSim;
    private int loadPwrId = -1, loadTmpId = -1;
    private long currTime = System.currentTimeMillis();
    private double loadFract = 0.6;

   /**
    ***************************************************************************
    **
    **  Inner container class for channel simulation data.
    **
    ***************************************************************************
    */
    public static class ChannelSim extends Channel implements Serializable {

       /**
        ***********************************************************************
        **
        **  Data fields
        **
        ***********************************************************************
        */
        float onValue;       // On (operating) value
        float onTime;        // Time (secs) to reach on value
        float offValue;      // Off (ambient) value
        float offTime;       // Time (secs) to reach off value
        float endValue;      // Ending value
        float rate;          // Rate of change
        boolean negative;    // True if nominal change is negative


       /**
        ***********************************************************************
        **
        **  Constructor
        **
        ***********************************************************************
        */
        public ChannelSim(String name, String desc, String units,
                          String loCheck, String hiCheck, float deadbandLo,
                          float deadbandHi, float onValue, float onTime,
                          float offValue, float offTime)
        {
            super(name, desc, units, loCheck, hiCheck, deadbandLo, deadbandHi);
            this.onValue  = onValue;
            this.onTime   = onTime;
            this.offValue = offValue;
            this.offTime  = offTime;
            this.endValue = 0f;
            this.rate     = 0f;
            this.negative = false;
        }


       /**
        ***********************************************************************
        **
        **  Converts data fields to a string
        **
        ***********************************************************************
        */
        @Override
        public String toString()
        {
            return "[" + name + ",\"" + description + "\"," + units + ","
                     + loCheckS + "," + hiCheckS + "," + deadbandLo + ","
                     + deadbandHi + ","  + onValue + "," + onTime + ","
                     + offValue + "," + offTime + "]";
        }
                
    }


   /**
    ***************************************************************************
    **
    **  Main Constructor
    **
    ***************************************************************************
    */
    public RefrigTestSim(String name, int tickMillis, Object[] chData,
                         String configFile)
    {
        super(name, tickMillis, configFile);
        nChan = chData.length;
        chanData = new ChannelSim[nChan];
        chanDataSim = (ChannelSim[])chanData;
        for (int j = 0; j < nChan; j++) {
            chanDataSim[j] = (ChannelSim)chData[j];
            if (chanDataSim[j].name.equals("LoadPower")) loadPwrId = j;
            if (chanDataSim[j].name.equals("LoadTmp")) loadTmpId = j;
        }
    }


   /**
    ***************************************************************************
    **
    **  Sets the load power fraction
    **
    ***************************************************************************
    */
    public void setLoadFract(double fract)
    {
        loadFract = (float)fract;
        setLoadPower();
    }


   /**
    ***************************************************************************
    **
    **  Gets the load power fraction
    **
    ***************************************************************************
    */
    public double getLoadFract()
    {
        return loadFract;
    }


   /**
    ***************************************************************************
    **
    **  Initializes the simulated sensors.
    **
    ***************************************************************************
    */
    @Override
    void initSensors()
    {
        /*
        **  Set initial simulator values
        */
        for (int j = 0; j < nChan; j++) {
            ChannelSim ch = chanDataSim[j];
            if (j == loadPwrId) {
                ch.onTime = 0F;
                ch.offValue = 0F;
                ch.offTime = 0F;
            }
            ch.endValue = ch.offValue;
            ch.value    = ch.offValue;
        }
    }


   /**
    ***************************************************************************
    **
    **  Reads the simulated sensors.
    **
    ***************************************************************************
    */
    @Override
    void readSensors()
    {
        long thisTime = System.currentTimeMillis();
        double intvl = (thisTime - currTime) / 1000.0;
        currTime = thisTime;
        for (int j = 0; j < nChan; j++) {
            ChannelSim ch = chanDataSim[j];
            ch.value += ch.rate * intvl;
            if (ch.negative) {
                if (ch.value < ch.endValue) ch.value = ch.endValue;
            }
            else {
                if (ch.value > ch.endValue) ch.value = ch.endValue;
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Turns the main power on or off.
    **
    ***************************************************************************
    */
    @Override
    void setMainPower()
    {
        currTime = System.currentTimeMillis();

        for (int j = 0; j < nChan; j++) {
            if (j == loadPwrId) continue;
            ChannelSim ch = chanDataSim[j];
            ch.rate = 0F;
            if ((setState & RefrigState.MAIN_POWER_STATE) != 0) {
                ch.endValue = ch.onValue;
                if (ch.onTime != 0F)
                    ch.rate = (ch.onValue - ch.offValue) / ch.onTime;
            }
            else {
                ch.endValue = ch.offValue;
                if (ch.offTime != 0F)
                    ch.rate = (ch.offValue - ch.onValue) / ch.offTime;
            }
            if (ch.rate == 0F)
                ch.value = ch.endValue;
            ch.negative = ch.rate < 0F;
        }

        setLoadPower();
    }
            

   /**
    ***************************************************************************
    **
    **  Turns the load power on or off, setting the correct level.
    **
    ***************************************************************************
    */
    @Override
    void setLoadPower()
    {
        ChannelSim pc = chanDataSim[loadPwrId];
        pc.endValue = 0F;
        if ((setState & RefrigState.MAIN_POWER_STATE) != 0) {
            if ((chanState & RefrigState.LOAD_POWER_STATE) != 0)
                pc.endValue = (float)(loadFract * pc.onValue);
            ChannelSim tc = chanDataSim[loadTmpId];
            if (tc.onTime != 0F)
                tc.rate = (tc.onValue - tc.offValue) / tc.onTime
                             + LOAD_POWER_FCTR * pc.endValue;
        }
        pc.value = pc.endValue;
    }


   /**
    ***************************************************************************
    **
    **  Sets the hardware display
    **
    ***************************************************************************
    */
    @Override
    void setDisplay()
    {
    }

}
