package org.lsst.ccs.subsystem.refrig;

import java.io.IOException;
import java.io.InputStream;
import java.io.Serializable;
import java.net.ServerSocket;
import java.util.HashMap;
import javax.usb.UsbException;
import org.lsst.ccs.drivers.mcc.MccUsb;
import org.lsst.ccs.drivers.wattsup.WattsUp;
import org.lsst.ccs.subsystem.refrig.data.RefrigState;
import org.lsst.ccs.utilities.conv.Convert;

/**
 ******************************************************************************
 **
 **  Implements the refrigeration long lines test modular subsystem
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class RefrigTestHdw extends RefrigTest implements WattsUp.Listener {

   /**
    ***************************************************************************
    **
    **  Private constants.
    **
    ***************************************************************************
    */
    private final static int
        MCC_DEV_TC_AI = 0,
        MCC_DEV_TC    = 1,
        MCC_DEV_TEMP  = 2,
        N_MCC_DEVS    = 3,
        N_MCC_DEV_CHANS = 8,
        N_MCC_CHANS     = N_MCC_DEV_CHANS * N_MCC_DEVS,
        TC_CHAN_MASK = (0x0f << (N_MCC_DEV_CHANS * MCC_DEV_TC_AI))
                         | (0xff << (N_MCC_DEV_CHANS * MCC_DEV_TC))
                         | (0xff << (N_MCC_DEV_CHANS * MCC_DEV_TEMP)),
        RT_CHAN_MASK = 0xff << (N_MCC_DEV_CHANS * MCC_DEV_TEMP),
        V_CHAN_MASK = 0xf0 << (N_MCC_DEV_CHANS * MCC_DEV_TC_AI),
        N_ALARMS    = 6,
        MAIN_POWER_DEV = MCC_DEV_TC_AI,
        MAIN_POWER_BIT = 7,
        LOAD_POWER_DEV = MCC_DEV_TC_AI,
        LOAD_POWER_DEV2 = MCC_DEV_TC,
        LOAD_POWER_BIT = 6,
        ALARMS_MASK = ((1 << N_ALARMS) - 1),
        LEDS_MASK = ALARMS_MASK | (1 << LOAD_POWER_BIT),
        LOAD_POWER_CHAN = 0,
        CMPR_POWER_CHAN = 1,
        CMPR_VOLTS_CHAN = 2,
        CHAN_TYPE_UNKNOWN = -1,
        CHAN_TYPE_TEMP    = 0,
        CHAN_TYPE_VOLTS   = 1,
        CHAN_TYPE_POWER   = 2,
        CHAN_TYPE_SWITCH  = 3,
        CHAN_TYPE_DIFF    = 4;

    private final static float[]
        RTD_COEFFS = {100.0F, 0.003908F, -5.8019E-7F, -4.2735E-12F};

    private final static String[] DEV_NAMES = new String[N_MCC_DEVS];
    static {
        DEV_NAMES[MCC_DEV_TC_AI] = "USB_TC_AI";
        DEV_NAMES[MCC_DEV_TC]    = "USB_TC";
        DEV_NAMES[MCC_DEV_TEMP]  = "USB_TEMP";
    }

    private final static int[] DEV_DIDS = new int[N_MCC_DEVS];
    static {
        DEV_DIDS[MCC_DEV_TC_AI] = MccUsb.USB_TC_AI_DID;
        DEV_DIDS[MCC_DEV_TC]    = MccUsb.USB_TC_DID;
        DEV_DIDS[MCC_DEV_TEMP]  = MccUsb.USB_TEMP_DID;
    }

   /**
    ***************************************************************************
    **
    **  Private fields.
    **
    ***************************************************************************
    */
    private final ChannelHdw[] chanDataHdw;
    private MccUsb[] mcc = new MccUsb[N_MCC_DEVS];
    private WattsUp wtu;
    private float[] power = {-1F, -1F, -1F};
    private boolean[] mccInited = new boolean[N_MCC_DEVS];
    private boolean wattsInited;

   /**
    ***************************************************************************
    **
    **  Inner container class for channel hardware data.
    **
    ***************************************************************************
    */
    public static class ChannelHdw extends Channel implements Serializable {

       /**
        ***********************************************************************
        **
        **  Data fields
        **
        ***********************************************************************
        */
        String typeS;      // Channel type string (TEMP, VOLTS, POWER...)
        String subtypeS;   // Channel subtype string (extra config info)
        int    chan;       // Hardware channel for MCC channel
        int    alarm;      // Hardware alarm number
        float  offset;     // Value offset
        float  scale;      // Value scale
        int    type;       // Encoded channel type
        int    subtype;    // Encoded channel subtype

       /**
        ***********************************************************************
        **
        **  Private lookup maps
        **
        ***********************************************************************
        */
        private final static HashMap<String,Integer> typeMap;
        static {
            typeMap = new HashMap<String,Integer>();
            typeMap.put("TEMP",   CHAN_TYPE_TEMP);
            typeMap.put("VOLTS",  CHAN_TYPE_VOLTS);
            typeMap.put("POWER",  CHAN_TYPE_POWER);
            typeMap.put("SWITCH", CHAN_TYPE_SWITCH);
            typeMap.put("DIFF",   CHAN_TYPE_DIFF);
        }

        private final static HashMap<String,Integer> tSubtypeMap;
        static {
            tSubtypeMap = new HashMap<String,Integer>();
            tSubtypeMap.put("TC",  MccUsb.STP_THERMOCOUPLE);
            tSubtypeMap.put("RTD", MccUsb.STP_RTD);
        }

        private final static HashMap<String,Integer> tcSubtypeMap;
        static {
            tcSubtypeMap = new HashMap<String,Integer>();
            tcSubtypeMap.put("J", MccUsb.TC_TYPE_J);
            tcSubtypeMap.put("K", MccUsb.TC_TYPE_K);
            tcSubtypeMap.put("T", MccUsb.TC_TYPE_T);
            tcSubtypeMap.put("E", MccUsb.TC_TYPE_E);
            tcSubtypeMap.put("R", MccUsb.TC_TYPE_R);
            tcSubtypeMap.put("S", MccUsb.TC_TYPE_S);
            tcSubtypeMap.put("B", MccUsb.TC_TYPE_B);
            tcSubtypeMap.put("N", MccUsb.TC_TYPE_N);
        }

        private final static HashMap<String,Integer> rtSubtypeMap;
        static {
            rtSubtypeMap = new HashMap<String,Integer>();
            rtSubtypeMap.put("2WIRE1", MccUsb.TCT_2WIRE_1SENSOR);
            rtSubtypeMap.put("2WIRE2", MccUsb.TCT_2WIRE_2SENSOR);
            rtSubtypeMap.put("3WIRE",  MccUsb.TCT_3WIRE);
            rtSubtypeMap.put("4WIRE",  MccUsb.TCT_4WIRE);
        }

        private final static HashMap<String,Integer> vSubtypeMap;
        static {
            vSubtypeMap = new HashMap<String,Integer>();
            vSubtypeMap.put("10V",   MccUsb.RANGE_10V);
            vSubtypeMap.put("5V",    MccUsb.RANGE_5V);
            vSubtypeMap.put("2.5V",  MccUsb.RANGE_2_5V);
            vSubtypeMap.put("1.25V", MccUsb.RANGE_1_25V);
        }


       /**
        ***********************************************************************
        **
        **  Constructor
        **
        ***********************************************************************
        */
        public ChannelHdw(String name, String desc, String units,
                          String loCheck, String hiCheck, float deadbandLo,
                          float deadbandHi, String type, String subtype,
                          int chan, int alarm, float offset, float scale)
        {
            super(name, desc, units, loCheck, hiCheck, deadbandLo, deadbandHi);

            int iType = CHAN_TYPE_UNKNOWN, iSubtype = -1, iChan = -1,
                iAlarm = -1;
            try {
                Integer iTyp = typeMap.get(type.toUpperCase());
                if (iTyp == null)
                    reportError(name, "type", type);
                iType = iTyp;

                String sbtype = subtype.toUpperCase();
                if (iType == CHAN_TYPE_TEMP) {
                    String[] sbtypes = sbtype.split(":", -1);
                    if (sbtypes.length < 2)
                        reportError(name, "subtype", subtype);
                    Integer iStp = tSubtypeMap.get(sbtypes[0]);
                    if (iStp == null)
                        reportError(name, "subtype", subtype);
                    iSubtype = iStp;
                    if (iStp == MccUsb.STP_RTD)
                        iStp = rtSubtypeMap.get(sbtypes[1]);
                    else
                        iStp = tcSubtypeMap.get(sbtypes[1]);
                    if (iStp == null)
                        reportError(name, "subtype", subtype);
                    iSubtype |= (iStp << 8);
                }
                else if (iType == CHAN_TYPE_VOLTS) {
                    Integer iStp = vSubtypeMap.get(sbtype);
                    if (iStp == null)
                        reportError(name, "subtype", subtype);
                    iSubtype = iStp;
                }

                if (iType == CHAN_TYPE_TEMP) {
                    int mask = (iSubtype & 0xff) == MccUsb.STP_RTD
                                 ? RT_CHAN_MASK : TC_CHAN_MASK;
                    if (((1 << chan) & mask) == 0)
                        reportError(name, "hw channel number", chan);
                }
                else if (iType == CHAN_TYPE_VOLTS) {
                    if (((1 << chan) & V_CHAN_MASK) == 0)
                        reportError(name, "hw channel number", chan);
                }
                else if (iType == CHAN_TYPE_SWITCH) {
                    if (chan < 0 || chan >= N_MCC_CHANS)
                        reportError(name, "hw channel number", chan);
                }
                iChan = chan;

                if (alarm >= N_ALARMS)
                    reportError(name, "alarm number", alarm);
                iAlarm = alarm;
            }
            catch (Exception e) {
            }

            this.typeS    = type;
            this.subtypeS = subtype;
            this.type     = iType;
            this.subtype  = iSubtype;
            this.chan     = iChan;
            this.alarm    = iAlarm;
            this.offset   = offset;
            this.scale    = scale;
        }


       /**
        ***********************************************************************
        **
        **  Converts data fields to a string
        **
        ***********************************************************************
        */
        @Override
        public String toString()
        {
            return "[" + name + ",\"" + description + "\"," + units + ","
                     + loCheckS + "," + hiCheckS + "," + deadbandLo + ","
                     + deadbandHi + "," + typeS + "," + subtypeS + ","
                     + chan + "," + alarm + "," + offset + "," + scale + "]";
        }
                
    }

   /**
    ***************************************************************************
    **
    **  Inner class implementing task to obtain remote WattsUp meter data
    **
    ***************************************************************************
    */
    private class RemotePower extends Thread {

        @Override
        public void run()
        {
            ServerSocket server;
            try {
                server = new ServerSocket(RefrigPower.POWER_PORT, 1);
            }
            catch (IOException e) {
                log.error("Error creating remote power listener: " + e);
                return;
            }

            byte[] data = new byte[12];
            InputStream netInp;

            while (true) {
                try {
                    netInp = server.accept().getInputStream();
                }
                catch (IOException e) {
                    log.error("Error creating remote power socket: " + e);
                    continue;
                }
                while (true) {
                    int nRead;
                    try {
                        nRead = netInp.read(data);
                    }
                    catch (IOException e) {
                        log.error("Error reading remote power data: " + e);
                        nRead = -2;
                    }
                    if (nRead < 0) {
                        try {
                            netInp.close();
                        }
                        catch (IOException e) {
                        }
                        power[CMPR_POWER_CHAN] = -1F;
                        power[CMPR_VOLTS_CHAN] = -1F;
                        break;
                    }
                    if (nRead < data.length)
                        continue;
                    if (Convert.bytesToInt(data, 0) != RefrigPower.POWER_ID)
                        continue;
                    power[CMPR_POWER_CHAN] = Convert.bytesToFloat(data, 4);
                    power[CMPR_VOLTS_CHAN] = Convert.bytesToFloat(data, 8);
                }
            }
        }

    }


   /**
    ***************************************************************************
    **
    **  Main constructor.
    **
    ***************************************************************************
    */
    public RefrigTestHdw(String name, int tickMillis, Object[] chData,
                         String configFile)
    {
        super(name, tickMillis, configFile);
        nChan = chData.length;
        chanData = new ChannelHdw[nChan];
        chanDataHdw = (ChannelHdw[])chanData;
        for (int j = 0; j < nChan; j++)
            chanDataHdw[j] = (ChannelHdw)chData[j];
        for (int j = 0; j < nChan; j++) {
            ChannelHdw ch = chanDataHdw[j];
            if (ch.type != CHAN_TYPE_DIFF) continue;
            String[] names = ch.subtypeS.split(":", -1);
            boolean okay = false;
            if (names.length == 2) {
                int indx0 = findChannel(names[0]);
                int indx1 = findChannel(names[1]);
                if (indx0 >= 0 && indx1 >= 0) {
                    ch.subtype = (indx0 << 16) | indx1;
                    okay = true;
                }
            }
            if (!okay) {
                log.error("Invalid subtype (" + ch.subtypeS + ") for "
                            + ch.name);
                ch.type = CHAN_TYPE_UNKNOWN;
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Finds a channel given its name.
    **
    ***************************************************************************
    */
    private int findChannel(String name)
    {
        for (int j = 0; j < nChan; j++) {
            ChannelHdw ch = chanDataHdw[j];
            if (name.equals(ch.name)) return j;
        }

        return -1;
    }

                
   /**
    ***************************************************************************
    **
    **  Initializes the refrigeration sensors.
    **
    ***************************************************************************
    */
    @Override
    void initSensors()
    {
        /*
        **  Perform basic configuration on the MCC DAQ boxes
        */
        for (int dev = 0; dev < N_MCC_DEVS; dev++) {
            try {
                mcc[dev] = new MccUsb();
                mcc[dev].open(DEV_DIDS[dev], null, true);
                mcc[dev].blink();
                mcc[dev].dioConfig(0);
                mcc[dev].dioOut(0);
                if (dev == MCC_DEV_TC_AI) {
                    for (int j = 0; j < N_ALARMS; j++)
                        mcc[dev].configAlarm(j, 0, 0, 0f, 0f);
                }
                else if (dev == MCC_DEV_TEMP) {
                    for (int j = 0; j < N_MCC_DEV_CHANS; j += 2)
                        mcc[dev].setSensorType(j, MccUsb.STP_DISABLED);
                }
                mccInited[dev] = true;
            }
            catch (UsbException e) {
                log.error("Error initializing " + DEV_NAMES[dev] + " module: "
                            + e);
            }
        }

        /*
        **  Perform basic configuration of the WattsUp? meter
        */
        try {
            wtu = new WattsUp();
            wtu.addListener(this);
            wtu.open();
            wattsInited = true;
            power[LOAD_POWER_CHAN] = 0F;
        }
        catch (Exception e) {
            log.error("Cannot access WattsUp meter: " + e);
        }

        /*
        **  Configure all the hardware channels
        */
        for (int j = 0; j < nChan; j++) {
            ChannelHdw ch = chanDataHdw[j];
            int dev = ch.chan / N_MCC_DEV_CHANS;
            if (!mccInited[dev]) continue;
            int chan = ch.chan % N_MCC_DEV_CHANS;
            try {
                if (ch.type == CHAN_TYPE_TEMP) {
                    int sType = ch.subtype & 0xff, tType = ch.subtype >> 8;
                    mcc[dev].setSensorType(chan, sType);
                    if (sType == MccUsb.STP_RTD) {
                        mcc[dev].setThermConnType(chan, tType);
                        mcc[dev].setExcitation(chan, MccUsb.CEX_210UA);
                        mcc[dev].setGain(chan, MccUsb.GAIN_4X);
                        for (int k = 0; k < RTD_COEFFS.length; k++)
                            mcc[dev].setCoefficient(chan, k, RTD_COEFFS[k]);
                    }
                    else
                        mcc[dev].setTCType(chan, tType);
                }
                else if (ch.type == CHAN_TYPE_VOLTS) {
                    mcc[dev].setGain(chan, ch.subtype);
                    mcc[dev].setVoltageConnType(chan, MccUsb.VCT_DIFFERENTIAL);
                }
                else if (ch.type == CHAN_TYPE_SWITCH)
                    mcc[dev].dioConfigBit(chan, 1);
            }
            catch (UsbException e) {
                log.error("Error configuring " + DEV_NAMES[dev] + " module: "
                          + e);
                mccInited[dev] = false;
            }
        }

        /*
        **  Start the thread that receives remote power data
        */
        (new RemotePower()).start();
    }


   /**
    ***************************************************************************
    **
    **  Reads the sensor data.
    **
    ***************************************************************************
    */
    @Override
    void readSensors()
    {
        /*
        **  Read all the hardware channels
        */
        for (int j = 0; j < nChan; j++) {
            ChannelHdw ch = chanDataHdw[j];
            if (ch.type == CHAN_TYPE_POWER) {
                ch.value = power[ch.chan];
                continue;
            }
            if (ch.type != CHAN_TYPE_SWITCH && ch.type != CHAN_TYPE_TEMP
                  && ch.type != CHAN_TYPE_VOLTS) continue;
            int dev = ch.chan / N_MCC_DEV_CHANS,
                chan = ch.chan % N_MCC_DEV_CHANS;
            if (ch.type == CHAN_TYPE_SWITCH) {
                try {
                    ch.value = mcc[dev].dioInBit(chan);
                }
                catch (UsbException e) {
                    if (mccInited[dev])
                        log.error("Error reading DIO line: " + e);
                }
                continue;
            }
            try {
                ch.value = ch.offset + ch.scale * mcc[dev].adcIn(chan, 0);
            }
            catch (UsbException e) {
                if (mccInited[dev])
                    log.error("Error reading ADC: " + e);
            }
        }

        /*
        **  Compute derived values
        */
        for (int j = 0; j < nChan; j++) {
            ChannelHdw ch = chanDataHdw[j];
            if (ch.type != CHAN_TYPE_DIFF) continue;
            ch.value = chanDataHdw[ch.subtype >> 16].value
                         - chanDataHdw[ch.subtype & 0xffff].value;
        }
    }


   /**
    ***************************************************************************
    **
    **  Turns the main power on or off.
    **
    ***************************************************************************
    */
    @Override
    void setMainPower()
    {
        try {
            int on = (setState >> RefrigState.MAIN_POWER_STATE_BIT) & 1;
            mcc[MAIN_POWER_DEV].dioOutBit(MAIN_POWER_BIT, on);
        }
        catch (UsbException e) {
            if (mccInited[MAIN_POWER_DEV])
                log.error("Error setting DIO line: " + e);
        }
    }
            

   /**
    ***************************************************************************
    **
    **  Turns the load power on or off.
    **
    ***************************************************************************
    */
    @Override
    void setLoadPower()
    {
        int on = (chanState >> RefrigState.LOAD_POWER_STATE_BIT) & 1;
        try {
            mcc[LOAD_POWER_DEV].dioOutBit(LOAD_POWER_BIT, on);
        }
        catch (UsbException e) {
            if (mccInited[LOAD_POWER_DEV])
                log.error("Error setting DIO line: " + e);
        }
        try {
            mcc[LOAD_POWER_DEV2].dioOutBit(LOAD_POWER_BIT, on ^ 1);
        }
        catch (UsbException e) {
            if (mccInited[LOAD_POWER_DEV2])
                log.error("Error setting DIO line: " + e);
        }
    }


   /**
    ***************************************************************************
    **
    **  Sets the front-panel display lights.
    **
    ***************************************************************************
    */
    @Override
    void setDisplay()
    {
        for (int j = 0; j < nChan; j++) {
            ChannelHdw ch = chanDataHdw[j];
            if (ch.alarm < 0) continue;
            int good = (chanState >> j) & 1;
            try {
                mcc[MCC_DEV_TC_AI].dioOutBit(ch.alarm, good ^ 1);
            }
            catch (UsbException e) {
                if (mccInited[MCC_DEV_TC_AI])
                    log.error("Error setting DIO line: " + e);
            }
            try {
                mcc[MCC_DEV_TC].dioOutBit(ch.alarm, good);
            }
            catch (UsbException e) {
                if (mccInited[MCC_DEV_TC])
                    log.error("Error setting DIO line: " + e);
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Changes the WattsUp? meter powered state.
    **
    **  <p>
    **  Performs meter setup upon power-on.
    **
    **  @param  on  Whether the meter is powered on or not
    **
    ***************************************************************************
    */
    @Override
    public void setPowered(boolean on)
    {
        if (on) {
            try {
                wtu.setLoggedFields(1 << WattsUp.FLD_WATTS);
                wtu.setExternalLogging(1);
            }
            catch (Exception e) {
                log.error("Cannot initialize WattsUp meter: " + e);
            }
        }
        else {
            power[LOAD_POWER_CHAN] = 0F;
        }
    }


   /**
    ***************************************************************************
    **
    **  Changes the WattsUp? meter open state.
    **
    ***************************************************************************
    */
    @Override
    public void setClosed()
    {
        power[LOAD_POWER_CHAN] = -1F;
        wattsInited = false;
    }


   /**
    ***************************************************************************
    **
    **  Receives data periodically from the WattsUp? meter.
    **
    **  @param  data  The array of data from the meter.
    **
    ***************************************************************************
    */
    @Override
    public void processData(float[] data)
    {
        power[LOAD_POWER_CHAN] = data[WattsUp.FLD_WATTS];
    }

}
