package org.lsst.ccs.subsystem.refrig;

import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Properties;
import java.util.Timer;
import java.util.TimerTask;
import org.lsst.ccs.bus.ValueNotification;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.subsystem.refrig.data.RefrigChannel;
import org.lsst.ccs.subsystem.refrig.data.RefrigFullState;
import org.lsst.ccs.subsystem.refrig.data.RefrigState;
import org.lsst.ccs.subsystem.refrig.status.RefrigStateStatus;

/**
 ******************************************************************************
 **
 **  Implements the refrigeration long lines test modular subsystem
 **
 **  <p>
 **  This is an abstract class implementing the interactions with the CCS
 **  buses, but deferring the hardware interaction, real or simulated, to
 **  sub-classes.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public abstract class RefrigTest extends Module {

   /**
    ***************************************************************************
    **
    **  Package constants
    **
    ***************************************************************************
    */
    final static int
        MAX_CHANS = 16;
    final static String
        PREFIX        = "org.lsst.ccs.refrig.";
    final static long
        UPDATE_PERIOD = 1000;

   /**
    ***************************************************************************
    **
    **  Package fields
    **
    ***************************************************************************
    */
    String configFile = "config.properties";
    Properties config;
    Channel[] chanData;
    int nChan, loLimChange = 0, hiLimChange = 0, setState = 0, chanState = 0;
    boolean running = false;

   /**
    ***************************************************************************
    **
    **  Inner container class for channel parameters
    **
    ***************************************************************************
    */
    public static class Channel implements Serializable {

       /**
        ***********************************************************************
        **
        **  Data fields
        **
        ***********************************************************************
        */
        String   name;          // Channel name
        String   description;   // Channel description
        String   units;         // Units name
        String   loCheckS;      // Low limit check option string
        String   hiCheckS;      // High limit check option string
        int      loCheck;       // Low limit check option
        int      hiCheck;       // High limit check option
        float    deadbandLo;    // Low limit dead-band
        float    deadbandHi;    // High limit dead-band
        float    limitLo;       // Low limit value
        float    limitHi;       // High limit value
        String   limitLoName;   // Low limit name (derived)
        String   limitHiName;   // High limit name (derived)
        float    value;         // Current value read from channel
        boolean  trippedLo;     // Whether low limit has caused a trip
        boolean  trippedHi;     // Whether high limit has caused a trip

       /**
        ***********************************************************************
        **
        **  Package constants
        **
        ***********************************************************************
        */
        final static int
            LIMIT_CHECK_NONE      = 0,   // Don't check limit
            LIMIT_CHECK_FLAG      = 1,   // Flag if limit exceeded
            LIMIT_CHECK_MAIN_TRIP = 2,   // Trip main power if limit exceeded
            LIMIT_CHECK_LOAD_TRIP = 3;   // Trip load power if limit exceeded

       /**
        ***********************************************************************
        **
        **  Private lookup maps
        **
        ***********************************************************************
        */
        private final static HashMap<String,Integer> checkMap;
        static {
            checkMap = new HashMap<String,Integer>();
            checkMap.put("NONE", LIMIT_CHECK_NONE);
            checkMap.put("FLAG", LIMIT_CHECK_FLAG);
            checkMap.put("MTRIP", LIMIT_CHECK_MAIN_TRIP);
            checkMap.put("LTRIP", LIMIT_CHECK_LOAD_TRIP);
        }


       /**
        ***********************************************************************
        **
        **  Constructor
        **
        ***********************************************************************
        */
        public Channel(String name, String desc, String units,
                       String loCheck, String hiCheck, float deadbandLo,
                       float deadbandHi)
        {
            int iLoCheck = LIMIT_CHECK_FLAG, iHiCheck = LIMIT_CHECK_FLAG;
            try {
                Integer iCheck = checkMap.get(loCheck.toUpperCase());
                if (iCheck == null)
                    reportError(name, "locheck", loCheck);
                iLoCheck = iCheck;
                iCheck = checkMap.get(hiCheck.toUpperCase());
                if (iCheck == null)
                    reportError(name, "hicheck", hiCheck);
                iHiCheck = iCheck;
            }
            catch (Exception e) {
            }

            this.name = name;
            this.description = desc;
            this.units = units;
            this.loCheckS = loCheck;
            this.hiCheckS = hiCheck;
            this.loCheck = iLoCheck;
            this.hiCheck = iHiCheck;
            this.deadbandLo = deadbandLo;
            this.deadbandHi = deadbandHi;
            this.limitLoName = name + "LoLim";
            this.limitHiName = name + "HiLim";
        }

    }

   /**
    ***************************************************************************
    **
    **  Inner class to implement a timer task to update the channel state
    **  in a timely manner.
    **
    ***************************************************************************
    */
    private class UpdateState extends TimerTask {

        @Override
        public void run()
        {
            /*
            **  Read the sensor values
            */
            readSensors();

            /*
            **  Check the limits and set the channel state
            */
            checkLimits();

            /*
            **  Update the hardware display
            */
            setDisplay();
        }

    }

                
   /**
    ***************************************************************************
    **
    **  Main constructor
    **
    ***************************************************************************
    */
    public RefrigTest(String name, int tickMillis, String configFile)
    {
        super();
        setName(name);
        this.tickMillis = tickMillis;
        this.configFile = configFile;
    }


   /**
    ***************************************************************************
    **
    **  Initializes the refrigeration subsystem
    **
    ***************************************************************************
    */
    @Override
    public void initModule()
    {
        /*
        **  Initialize all configuration data
        */
        initConfiguration();

        /*
        **  Initialize the hardware
        */
        initSensors();

        /*
        **  Start the timer task which updates the channel state periodically
        */
        (new Timer()).schedule(new UpdateState(), 0L, UPDATE_PERIOD);
    }


   /**
    ***************************************************************************
    **
    **  Performs periodic trending data broadcast
    **
    ***************************************************************************
    */
    @Override
    public void tick()
    {
        /*
        **  Broadcast the state if first time
        */
        if (!running) {
            System.out.println("Refrigeration test system started");
            publishState();    // For any GUIs
            publishLimits();   // For GUIs and the trending database
            running = true;
        }

        /*
        **  Broadcast the trending data
        */
        long timeStamp = System.currentTimeMillis();
        ArrayList<ValueNotification> trendingValues
          = new ArrayList<ValueNotification>();
        for (int j = 0; j < nChan; j++) {
            Channel chan = chanData[j];
            trendingValues.add(new ValueNotification(chan.name, chan.value,
                                                     timeStamp));
        }
        trendingValues.add(new ValueNotification("state", getState(),
                                                 timeStamp));
        publishData(trendingValues);
    }


   /**
    ***************************************************************************
    **
    **  Sets the tick period
    **
    **  <p>
    **  Overrides the method in Module in order to publish a status update.
    **
    **  @param  value  The tick period (milliseconds) to set.
    **
    ***************************************************************************
    */
    @Override
    public void setTickMillis(int value)
    {
        super.setTickMillis(value);
        if (running) publishState();
    }


   /**
    ***************************************************************************
    **
    **  Sets the main power enabled state on or off
    **
    **  @param  value  The enabled state value to set: 0 = off, ~0 = on.
    **
    ***************************************************************************
    */
    public void setMainPowerEnable(int value)
    {
        setMainPowerEnable(value, true);
    }


   /**
    ***************************************************************************
    **
    **  Sets the main power enabled state on or off
    **
    **  @param  value    The enabled state value to set: 0 = off, ~0 = on.
    **
    **  @param  publish  If true, publish the new state on the status bus.
    **
    ***************************************************************************
    */
    public void setMainPowerEnable(int value, boolean publish)
    {
        if (value != 0) {
            if ((chanState & RefrigState.MAIN_TRIPPED_STATE) == 0)
                setState |= RefrigState.MAIN_POWER_STATE;
        }
        else
            setState &= ~RefrigState.MAIN_POWER_STATE;
        setMainPower();
        if (publish) publishState();
    }


   /**
    ***************************************************************************
    **
    **  Sets the load power trip enabled state on or off
    **
    **  @param  value  The load trip enabled state value to set: 0 = off;
    **                 ~0 = on.
    **
    ***************************************************************************
    */
    public void setLoadTripEnable(int value)
    {
        if (value != 0)
            setState |= RefrigState.LOAD_TRIP_ENAB_STATE;
        else
            setState &= ~RefrigState.LOAD_TRIP_ENAB_STATE;
        publishState();
    }


   /**
    ***************************************************************************
    **
    **  Gets the operating state word
    **
    **  @return  The current value of the operating state
    **
    ***************************************************************************
    */
    public int getState()
    {
        return setState | chanState;
    }


   /**
    ***************************************************************************
    **
    **  Sets the low limit for a channel
    **
    **  @param  id     The channel ID
    **
    **  @param  limit  The low limit value.
    **
    ***************************************************************************
    */
    public void setLowLimit(int id, double limit)
    {
        Channel chan = chanData[id];
        float fLimit = (float)limit;
        if (fLimit == chan.limitLo) return;
        chan.limitLo = fLimit;
        String sLimit = String.valueOf(fLimit);
        config.setProperty(PREFIX + chan.limitLoName, sLimit);
        loLimChange |= 1 << id;
        publishState();
        getSubsystem().publishMetaData(chan.name, "alarmLow", sLimit);
    }


   /**
    ***************************************************************************
    **
    **  Sets the high limit for a channel
    **
    **  @param  id     The channel ID
    **
    **  @param  limit  The high limit value.
    **
    ***************************************************************************
    */
    public void setHighLimit(int id, double limit)
    {
        Channel chan = chanData[id];
        float fLimit = (float)limit;
        if (fLimit == chan.limitHi) return;
        chan.limitHi = fLimit;
        String sLimit = String.valueOf(fLimit);
        config.setProperty(PREFIX + chan.limitHiName, sLimit);
        hiLimChange |= 1 << id;
        publishState();
        getSubsystem().publishMetaData(chan.name, "alarmHigh", sLimit);
    }


   /**
    ***************************************************************************
    **
    **  Saves the configuration data
    **
    ***************************************************************************
    */
    public void saveConfiguration()
    {
        FileWriter writer = null;
        try {
            writer = new FileWriter(configFile);
        }
        catch (IOException e) {
            log.error("Error opening configuration file: " + e);
        }
        if (writer != null) {
            try {
                config.store(writer,
                             "Refrigeration long lines test configuration");
                loLimChange = hiLimChange = 0;
            }
            catch (IOException e) {
                log.error("Error writing configuration: " + e);
            }
            try {
                writer.close();
            }
            catch (IOException e) {
                log.error("Error closing configuration file: " + e);
            }
        }
        publishState();   // Must always do this
    }


   /**
    ***************************************************************************
    **
    **  Gets the full state of the refrigeration module
    **
    **  <p>
    **  This is intended to be called by GUIs during initialization
    **
    ***************************************************************************
    */
    public RefrigFullState getFullState()
    {
        RefrigFullState status = new RefrigFullState();
        for ( int i = 0; i < nChan; i++) {
            Channel c = chanData[i];
            status.addChannel(new RefrigChannel(c.name, c.description, c.units,
                                                c.limitLo, c.limitHi, c.value));
        }
        status.setRefrigState(new RefrigState(getState(), loLimChange,
                                              hiLimChange, getTickMillis()));
        return status;
    }    


   /**
    ***************************************************************************
    **
    **  Publishes the state of the refrigeration module
    **
    **  <p>
    **  This is intended to be called whenever any element of the state is
    **  changed.
    **
    ***************************************************************************
    */
    void publishState()
    {
        RefrigState refrigState = new RefrigState(getState(), loLimChange,
                                                  hiLimChange, getTickMillis());
        sendToStatus(new RefrigStateStatus(refrigState));
    }    


   /**
    ***************************************************************************
    **
    **  Publishes all the limit values of the refrigeration module
    **
    **  <p>
    **  This is intended to be called at startup time.
    **
    ***************************************************************************
    */
    void publishLimits()
    {
        for ( int i = 0; i < nChan; i++) {
            Channel c = chanData[i];
            getSubsystem().publishMetaData(c.name, "alarmLow",
                                           String.valueOf(c.limitLo));
            getSubsystem().publishMetaData(c.name, "alarmHigh",
                                           String.valueOf(c.limitHi));
        }
    }    


   /**
    ***************************************************************************
    **
    **  Checks whether current channel values are within limits
    **
    ***************************************************************************
    */
    void checkLimits()
    {
        int cState = RefrigState.LOAD_POWER_STATE;
        for (int j = 0; j < nChan; j++) {
            Channel ch = chanData[j];
            boolean ok = true;
            if (ch.value < ch.limitLo) {
                ok = false;
                if (ch.loCheck == Channel.LIMIT_CHECK_NONE)
                    ok = true;
                else if (ch.loCheck == Channel.LIMIT_CHECK_MAIN_TRIP)
                    cState |= RefrigState.MAIN_TRIPPED_STATE;
                else if (ch.loCheck == Channel.LIMIT_CHECK_LOAD_TRIP) {
                    if ((setState & RefrigState.LOAD_TRIP_ENAB_STATE) != 0) {
                        cState &= ~RefrigState.LOAD_POWER_STATE;
                        ch.trippedLo = true;
                    }
                }
            }
            else if (ch.value > ch.limitHi) {
                ok = false;
                if (ch.hiCheck == Channel.LIMIT_CHECK_NONE)
                    ok = true;
                else if (ch.hiCheck == Channel.LIMIT_CHECK_MAIN_TRIP)
                    cState |= RefrigState.MAIN_TRIPPED_STATE;
                else if (ch.hiCheck == Channel.LIMIT_CHECK_LOAD_TRIP) {
                    if ((setState & RefrigState.LOAD_TRIP_ENAB_STATE) != 0) {
                        cState &= ~RefrigState.LOAD_POWER_STATE;
                        ch.trippedHi = true;
                    }
                }
            }
            else {
                if (ch.loCheck == Channel.LIMIT_CHECK_LOAD_TRIP) {
                    boolean tripped = false;
                    if ((setState & RefrigState.LOAD_TRIP_ENAB_STATE) != 0) {
                        if (ch.trippedLo
                               && ch.value < ch.limitLo + ch.deadbandLo) {
                            cState &= ~RefrigState.LOAD_POWER_STATE;
                            tripped = true;
                        }
                    }
                    ch.trippedLo = tripped;
                }
                if (ch.hiCheck == Channel.LIMIT_CHECK_LOAD_TRIP) {
                    boolean tripped = false;
                    if ((setState & RefrigState.LOAD_TRIP_ENAB_STATE) != 0) {
                        if (ch.trippedHi
                              && ch.value > ch.limitHi - ch.deadbandHi) {
                            cState &= ~RefrigState.LOAD_POWER_STATE;
                            tripped = true;
                        }
                    }
                    ch.trippedHi = tripped;
                }
            }
            if (ok) cState |= (1 << j);
        }
                
        /*
        **  If state changed
        **    Turn main power off if it was tripped off
        **    Turn load power off or on appropriately
        **    If running
        **      Publish state
        **      Log channel data
        */
        cState ^= chanState;
        chanState ^= cState;
        if (cState != 0) {
            int tr;
            if ((cState & RefrigState.MAIN_TRIPPED_STATE) != 0) {
                tr = (chanState >> RefrigState.MAIN_TRIPPED_STATE_BIT) & 1;
                if (tr != 0) setMainPowerEnable(0, false);
            }
            if ((cState & RefrigState.LOAD_POWER_STATE) != 0)
                setLoadPower();
            if (running) {
                publishState();
                tick();
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Reports parameter errors
    **
    ***************************************************************************
    */
    static void reportError(String cName, String pName, Object pValue)
        throws Exception
    {
        log.error("Invalid " + pName + " (" + pValue + ") for " + cName);
        throw new Exception();
    }


   /**
    ***************************************************************************
    **
    **  Initializes the configuration data
    **
    ***************************************************************************
    */
    private void initConfiguration()
    {
        /*
        **  Generate a properties list containing default configuration values
        */
        Properties dfltConfig = new Properties();
        for (int j = 0; j < nChan; j++) {
            Channel chan = chanData[j];
            dfltConfig.setProperty(PREFIX + chan.limitLoName, "0.0");
            dfltConfig.setProperty(PREFIX + chan.limitHiName, "0.0");
        }

        /*
        **  Create the configuration properties list and load it
        */
        config = loadProperties(configFile, dfltConfig);

        /*
        **  Extract values from the properties list
        */
        for (int j = 0; j < nChan; j++) {
            Channel chan = chanData[j];
            chan.limitLo
              = Float.valueOf(config.getProperty(PREFIX + chan.limitLoName));
            chan.limitHi
              = Float.valueOf(config.getProperty(PREFIX + chan.limitHiName));
        }
    }


   /**
    ***************************************************************************
    **
    **  Loads a properties file
    **
    ***************************************************************************
    */
    private Properties loadProperties(String file, Properties dflt)
    {
        /*
        **  Create the properties list
        */
        Properties propList;
        if (dflt == null)
            propList = new Properties();
        else
            propList = new Properties(dflt);

        /*
        **  Open the properties file
        */
        FileReader reader = null;
        try {
            reader = new FileReader(file);
        }
        catch (FileNotFoundException e) {
            log.error("Error opening properties file: " + e);
        }

        /*
        **  Load the properties list
        */
        try {
            propList.load(reader);
        }
        catch (IOException e) {
            log.error("Error reading properties file: " + e);
        }
        try {
            reader.close();
        }
        catch (IOException e) {
            log.error("Error closing properties file: " + e);
        }

        return propList;
    }


   /**
    ***************************************************************************
    **
    **  Initializes the refrigeration sensors.
    **
    ***************************************************************************
    */
    abstract void initSensors();


   /**
    ***************************************************************************
    **
    **  Reads the sensor data.
    **
    ***************************************************************************
    */
    abstract void readSensors();


   /**
    ***************************************************************************
    **
    **  Turns the main power on or off, according to the set value.
    **
    ***************************************************************************
    */
    abstract void setMainPower();
            

   /**
    ***************************************************************************
    **
    **  Turns the load power on or off, according to the set value.
    **
    ***************************************************************************
    */
    abstract void setLoadPower();


   /**
    ***************************************************************************
    **
    **  Sets the hardware display, according to the current state
    **
    ***************************************************************************
    */
    abstract void setDisplay();
    
}
