package org.lsst.ccs.subsystem.refrig;

import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintStream;
import java.net.Socket;
import java.net.UnknownHostException;
import java.util.Timer;
import java.util.TimerTask;
import org.lsst.ccs.drivers.wattsup.WattsUp;
import org.lsst.ccs.utilities.conv.Convert;

/**
 ***************************************************************************
 **
 **  Server program to send WattsUp power data to DAQ program
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class RefrigPower implements WattsUp.Listener {

   /**
    ***************************************************************************
    **
    **  Package and private constants
    **
    ***************************************************************************
    */
    final static int
        POWER_PORT = 7070,
        POWER_ID   = 13572468;

    final private int
        UPDATE_PERIOD = 1,
        CHECK_OPEN_PERIOD = 10;

   /**
    ***************************************************************************
    **
    **  Private fields
    **
    ***************************************************************************
    */
    private static final PrintStream out = System.out;
    private OutputStream netOut = null;
    private WattsUp wtu;
    private String daqNode;
    private float power = -1F, volts = -1F;
    private boolean meterOpen = false;

   /**
    ***************************************************************************
    **
    **  Inner class to implement timer task to keep meter connection open
    **
    ***************************************************************************
    */
    private class OpenMeter extends TimerTask {

        @Override
        public void run()
        {
            if (meterOpen) return;
            try {
                out.println("Attempting meter connection");
                wtu.open();
                power = 0F;
                volts = 0F;
                out.println("Connected to WattsUp meter");
                meterOpen = true;
            }
            catch (RuntimeException e) {
                out.println("Cannot open meter (runtime): " + e);
            }
            catch (Exception e) {
                out.println("Cannot open meter (excptn): " + e);
            }
            catch (Error e) {
                out.println("Cannot open meter (error): " + e);
            }
        }

    }

   /**
    ***************************************************************************
    **
    **  Inner class to implement timer task to keep network connection open
    **
    ***************************************************************************
    */
    private class OpenNet extends TimerTask {

        @Override
        public void run()
        {
            if (netOut != null) return;
            try {
                netOut = (new Socket(daqNode, POWER_PORT)).getOutputStream();
                out.println("Connected to DAQ node (" + daqNode + ")");
            }
            catch (UnknownHostException e) {
                out.println(e);
            }
            catch (Exception e) {
            }
        }

    }

   /**
    ***************************************************************************
    **
    **  Inner class to implement timer task to send the current power values
    **
    ***************************************************************************
    */
    private class SendPower extends TimerTask {

        @Override
        public void run()
        {
            sendData();
        }

    }


   /**
    ***************************************************************************
    **
    **  Main program
    **
    ***************************************************************************
    */
    public static void main(String[] args)
    {
        if (args.length < 1) {
            out.println("No DAQ IP address supplied");
            System.exit(0);
        }
        (new RefrigPower()).run(args[0]);
        System.exit(0);
    }


   /**
    ***************************************************************************
    **
    **  Runs the power meter reader
    **
    **  @param  node  The name of the DAQ node to receive the power data
    **
    ***************************************************************************
    */
    private void run(String node)
    {
        /*
        **  Set up meter object, etc
        */
        wtu = new WattsUp();
        wtu.addListener(this);
        daqNode = node;

        /*
        **  Start the timer task to keep the meter connection open
        */
        (new Timer()).schedule(new OpenMeter(), 0L, 1000 * CHECK_OPEN_PERIOD);

        /*
        **  Start the timer task to keep the network connection open
        */
        (new Timer()).schedule(new OpenNet(), 0L, 1000 * CHECK_OPEN_PERIOD);

        /*
        **  Start the timer task to regularly send the current power value
        */
        (new Timer()).schedule(new SendPower(), 0L, 1000 * UPDATE_PERIOD);

        /*
        **  Sleep forever
        */
        try {
            Thread.sleep(Long.MAX_VALUE);
        }
        catch (InterruptedException e) {
        }
    }


   /**
    ***************************************************************************
    **
    **  Receives notification of powered state change
    **
    **  @param  on  Whether power is on or not
    **
    ***************************************************************************
    */
    @Override
    public void setPowered(boolean on)
    {
        if (on) {
            try {
                wtu.setLoggedFields(1 << WattsUp.FLD_WATTS
                                      | (1 << WattsUp.FLD_VOLTS));
                wtu.setExternalLogging(UPDATE_PERIOD);
            }
            catch (Exception e) {
                out.println("Cannot configure WattsUp meter: " + e);
            }
        }
        else {
            power = volts = 0F;
            sendData();
        }
    }


   /**
    ***************************************************************************
    **
    **  Receives notification of device closure
    **
    ***************************************************************************
    */
    @Override
    public void setClosed()
    {
        out.println("Disconnected from WattsUp meter");
        power = volts = -1F;
        sendData();
        meterOpen = false;
    }


   /**
    ***************************************************************************
    **
    **  Receives data periodically from the WattsUp? meter.
    **
    **  @param  data  The array of data from the meter.
    **
    ***************************************************************************
    */
    @Override
    public void processData(float[] data)
    {
        power = data[WattsUp.FLD_WATTS];
        volts = data[WattsUp.FLD_VOLTS];
    }


   /**
    ***************************************************************************
    **
    **  Sends the power data to the DAQ node
    **
    ***************************************************************************
    */
    private void sendData()
    {
        if (netOut != null) {
            byte[] mesg = new byte[12];
            Convert.intToBytes(POWER_ID, mesg, 0);
            Convert.floatToBytes(power, mesg, 4);
            Convert.floatToBytes(volts, mesg, 8);
            try {
                netOut.write(mesg);
            }
            catch (IOException e) {
                out.println("Lost connection to DAQ node");
                closeNet();
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Closes the connection to the DAQ node
    **
    ***************************************************************************
    */
    private void closeNet()
    {
        try {
            netOut.close();
        }
        catch (IOException e) {
        }
        netOut = null;
    }

}
