package org.lsst.ccs.subsystem.refrig;

import java.io.IOException;
import java.io.InputStream;
import java.net.ServerSocket;
import java.util.HashMap;
import java.util.Properties;
import java.util.Timer;
import java.util.TimerTask;
import javax.usb.UsbException;
import org.lsst.ccs.drivers.mcc.MccUsb;
import org.lsst.ccs.drivers.wattsup.WattsUp;
import org.lsst.ccs.utilities.conv.Convert;

/**
 ***************************************************************************
 **
 **  Implements the refrigeration long lines test modular subsystem
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class RefrigTestHdw extends RefrigTest implements WattsUp.Listener {

   /**
    ***************************************************************************
    **
    **  Container class for channel hardware data.
    **
    ***************************************************************************
    */
    private static class ChanHdw {

        int    type;       // Channel type (Temperature, voltage, power...)
        int    subtype;    // Channel subtype (extra config info)
        int    chan;       // Hardware channel for MCC channel
        int    alarm;      // Hardware alarm number
        float  offset;     // Value offset
        float  scale;      // Value scale

        public ChanHdw(int type, int subtype, int chan, int alarm,
                       float offset, float scale)
        {
            this.type    = type;
            this.subtype = subtype;
            this.chan    = chan;
            this.alarm   = alarm;
            this.offset  = offset;
            this.scale   = scale;
        }
    }

   /**
    ***************************************************************************
    **
    **  Inner class implementing task to obtain remote WattsUp meter data
    **
    ***************************************************************************
    */
    private class RemotePower extends Thread {

        @Override
        public void run()
        {
            ServerSocket server;
            try {
                server = new ServerSocket(RefrigPower.POWER_PORT, 1);
            }
            catch (IOException e) {
                log.error("Error creating remote power listener: " + e);
                return;
            }

            byte[] data = new byte[8];
            InputStream netInp;

            while (true) {
                try {
                    netInp = server.accept().getInputStream();
                }
                catch (IOException e) {
                    log.error("Error creating remote power socket: " + e);
                    continue;
                }
                while (true) {
                    int nRead;
                    try {
                        nRead = netInp.read(data);
                    }
                    catch (IOException e) {
                        log.error("Error reading remote power data: " + e);
                        nRead = -2;
                    }
                    if (nRead < 0) {
                        try {
                            netInp.close();
                        }
                        catch (IOException e) {
                        }
                        power[CMPR_POWER_CHAN] = -1F;
                        break;
                    }
                    if (nRead < data.length)
                        continue;
                    if (Convert.bytesToInt(data, 0) != RefrigPower.POWER_ID)
                        continue;
                    power[CMPR_POWER_CHAN] = Convert.bytesToFloat(data, 4);
                }
            }
        }

    }

   /**
    ***************************************************************************
    **
    **  Inner class to implement timer task to update the channel state and
    **  the green state LEDs.
    **
    ***************************************************************************
    */
    private class UpdateState extends TimerTask {

        @Override
        public void run()
        {
            /*
            **  Read channels without hardware alarms and check them
            */
            int cState = 0;
            for (int j = 0; j < nChan; j++) {
                Channel ch = chanData[j];
                ChanHdw hw = hdwData[j];
                if (hw.type == CHAN_TYPE_POWER) {
                    ch.value = power[hw.chan];
                }
                else {
                    int dev = hw.chan / N_MCC_DEV_CHANS,
                        chan = hw.chan % N_MCC_DEV_CHANS;
                    if (hw.type == CHAN_TYPE_SWITCH) {
                        try {
                            ch.value = mcc[dev].dioInBit(chan);
                        }
                        catch (UsbException e) {
                            if (mccInited[dev])
                                log.error("Error reading DIO line: " + e);
                        }
                    }
                    else {
                        if (hw.alarm >= 0 && dev == MCC_DEV_TC_AI) continue;
                        try {
                            ch.value
                              = hw.offset + hw.scale * mcc[dev].adcIn(chan, 0);
                        }
                        catch (UsbException e) {
                            if (mccInited[dev])
                                log.error("Error reading ADC: " + e);
                        }
                    }
                }
                boolean ok = (!ch.checkLo || ch.value >= ch.limitLo)
                               && (!ch.checkHi || ch.value <= ch.limitHi);
                if (ok) cState |= (1 << j);
                if (hw.alarm >= 0) {
                    try {
                        mcc[MCC_DEV_TC_AI].dioOutBit(hw.alarm, ok ? 0 : 1);
                    }
                    catch (UsbException e) {
                        if (mccInited[MCC_DEV_TC_AI])
                            log.error("Error setting DIO line: " + e);
                    }
                }
            }
                    
            /*
            **  Read LED/alarm lines and update other color
            */
            int lines = 0;
            try {
                lines = mcc[MCC_DEV_TC_AI].dioIn();
                try {
                    mcc[MCC_DEV_TC].dioOut((lines ^ LEDS_MASK) & LEDS_MASK);
                }
                catch (UsbException e) {
                    if (mccInited[MCC_DEV_TC])
                        log.error("Error writing DIO lines: " + e);
                }
            }
            catch (UsbException e) {
                if (mccInited[MCC_DEV_TC_AI])
                    log.error("Error reading DIO lines: " + e);
            }

            /*
            **  Update state from alarm lines
            */
            for (int j = 0; j < nChan; j++) {
                int alarm = hdwData[j].alarm;
                if (alarm >= 0 && ((1 << alarm) & lines) == 0)
                    cState |= (1 << j);
            }

            /*
            **  Do full readout and report if state changed and system running
            */
            if (cState != chanState && running) {
                chanState = cState;
                tick();
            }
        }

    }

   /**
    ***************************************************************************
    **
    **  Private constants.
    **
    ***************************************************************************
    */
    private final static int
        MCC_DEV_TC_AI = 0,
        MCC_DEV_TC    = 1,
        MCC_DEV_TEMP  = 2,
        N_MCC_DEVS    = 3,
        N_MCC_DEV_CHANS = 8,
        N_MCC_CHANS     = N_MCC_DEV_CHANS * N_MCC_DEVS,
        TC_CHAN_MASK = (0x0f << (N_MCC_DEV_CHANS * MCC_DEV_TC_AI))
                         | (0xff << (N_MCC_DEV_CHANS * MCC_DEV_TC))
                         | (0xff << (N_MCC_DEV_CHANS * MCC_DEV_TEMP)),
        RT_CHAN_MASK = 0xff << (N_MCC_DEV_CHANS * MCC_DEV_TEMP),
        V_CHAN_MASK = 0xf0 << (N_MCC_DEV_CHANS * MCC_DEV_TC_AI),
        N_ALARMS    = 6,
        MAIN_POWER_DEV = MCC_DEV_TC_AI,
        MAIN_POWER_BIT = 7,
        LOAD_POWER_DEV = MCC_DEV_TC_AI,
        LOAD_POWER_BIT = 6,
        ALARMS_MASK = ((1 << N_ALARMS) - 1),
        LEDS_MASK = ALARMS_MASK | (1 << LOAD_POWER_BIT),
        LOAD_POWER_CHAN = 0,
        CMPR_POWER_CHAN = 1,
        CHAN_TYPE_UNKNOWN = -1,
        CHAN_TYPE_TEMP    = 0,
        CHAN_TYPE_VOLTS   = 1,
        CHAN_TYPE_POWER   = 2,
        CHAN_TYPE_SWITCH  = 3;

    private final static long
        UPDATE_PERIOD = 1000;

    private final static float[]
        RTD_COEFFS = {100.0F, 0.003908F, -5.8019E-7F, -4.2735E-12F};

    private final static String[] DEV_NAMES = new String[N_MCC_DEVS];
    static {
        DEV_NAMES[MCC_DEV_TC_AI] = "USB_TC_AI";
        DEV_NAMES[MCC_DEV_TC]    = "USB_TC";
        DEV_NAMES[MCC_DEV_TEMP]  = "USB_TEMP";
    }

   /**
    ***************************************************************************
    **
    **  Private fields.
    **
    ***************************************************************************
    */
    private MccUsb[] mcc = new MccUsb[N_MCC_DEVS];
    private WattsUp wtu;
    private float[] power = {-1F, -1F};
    private boolean[] mccInited = new boolean[N_MCC_DEVS];
    private boolean wattsInited;
    private ChanHdw[] hdwData;


   /**
    ***************************************************************************
    **
    **  Performs additional structural configuration.
    **
    ***************************************************************************
    */
    private static final int
        PROP_TYPE    = 0,    // Don't change this value!!!
        PROP_SUBTYPE = 1,    // Or this one!!!
        PROP_HWCHAN  = 2,    // The rest must be sequential
        PROP_ALARM   = 3,
        PROP_OFFSET  = 4,
        PROP_SCALE   = 5,
        N_PROPS      = 6;

    private static final String[] propNames = new String[N_PROPS];
    static {
        propNames[PROP_TYPE]    = "types";
        propNames[PROP_SUBTYPE] = "subtypes";
        propNames[PROP_HWCHAN]  = "hwchans";
        propNames[PROP_ALARM]   = "alarms";
        propNames[PROP_OFFSET]  = "offsets";
        propNames[PROP_SCALE]   = "scales";
    }

    private final static HashMap<String,Integer> typeMap;
    static {
        typeMap = new HashMap<String,Integer>();
        typeMap.put("TEMP",   CHAN_TYPE_TEMP);
        typeMap.put("VOLTS",  CHAN_TYPE_VOLTS);
        typeMap.put("POWER",  CHAN_TYPE_POWER);
        typeMap.put("SWITCH", CHAN_TYPE_SWITCH);
    }

    private final static HashMap<String,Integer> tSubtypeMap;
    static {
        tSubtypeMap = new HashMap<String,Integer>();
        tSubtypeMap.put("TC",  MccUsb.STP_THERMOCOUPLE);
        tSubtypeMap.put("RTD", MccUsb.STP_RTD);
    }

    private final static HashMap<String,Integer> tcSubtypeMap;
    static {
        tcSubtypeMap = new HashMap<String,Integer>();
        tcSubtypeMap.put("J", MccUsb.TC_TYPE_J);
        tcSubtypeMap.put("K", MccUsb.TC_TYPE_K);
        tcSubtypeMap.put("T", MccUsb.TC_TYPE_T);
        tcSubtypeMap.put("E", MccUsb.TC_TYPE_E);
        tcSubtypeMap.put("R", MccUsb.TC_TYPE_R);
        tcSubtypeMap.put("S", MccUsb.TC_TYPE_S);
        tcSubtypeMap.put("B", MccUsb.TC_TYPE_B);
        tcSubtypeMap.put("N", MccUsb.TC_TYPE_N);
    }

    private final static HashMap<String,Integer> rtSubtypeMap;
    static {
        rtSubtypeMap = new HashMap<String,Integer>();
        rtSubtypeMap.put("2WIRE1", MccUsb.TCT_2WIRE_1SENSOR);
        rtSubtypeMap.put("2WIRE2", MccUsb.TCT_2WIRE_2SENSOR);
        rtSubtypeMap.put("3WIRE",  MccUsb.TCT_3WIRE);
        rtSubtypeMap.put("4WIRE",  MccUsb.TCT_4WIRE);
    }

    private final static HashMap<String,Integer> vSubtypeMap;
    static {
        vSubtypeMap = new HashMap<String,Integer>();
        vSubtypeMap.put("10V",   MccUsb.RANGE_10V);
        vSubtypeMap.put("5V",    MccUsb.RANGE_5V);
        vSubtypeMap.put("2.5V",  MccUsb.RANGE_2_5V);
        vSubtypeMap.put("1.25V", MccUsb.RANGE_1_25V);
    }

    @Override
    void addStructure(Properties struct)
    {
        hdwData = new ChanHdw[nChan];
        String[][] pValues = new String[N_PROPS][];
        for (int jp = 0; jp < N_PROPS; jp++) {
            String pValue = struct.getProperty(PREFIX + propNames[jp]);
            if (pValue != null) {
                pValues[jp] = pValue.split(",", -1);
            }
            else {
                log.error("Hardware structure is missing " + propNames[jp]
                          + " property");
                pValues[jp] = new String[nChan];
                for (int jc = 0; jc < nChan; jc++)
                    pValues[jp][jc] = "-1";
            }
        }

        Object[] values = new Object[N_PROPS];
        for (int jc = 0; jc < nChan; jc++) {
            int type = CHAN_TYPE_UNKNOWN, subtype = -1;
            for (int jp = 0; jp < N_PROPS; jp++) {
                try {
                    if (jp == PROP_TYPE) {
                        String cType =pValues[jp][jc].toUpperCase();
                        Integer typ = typeMap.get(cType);
                        if (typ == null)
                            throw new NumberFormatException();
                        type = typ;
                    }
                    else if (jp == PROP_SUBTYPE) {
                        Integer stp;
                        String csType = pValues[jp][jc].toUpperCase();
                        if (type == CHAN_TYPE_TEMP) {
                            String[] csTypes = csType.split(":", -1);
                            if (csTypes.length < 2)
                                throw new NumberFormatException();
                            stp = tSubtypeMap.get(csTypes[0]);
                            if (stp == null)
                                throw new NumberFormatException();
                            subtype = stp;
                            if (stp == MccUsb.STP_RTD)
                                stp = rtSubtypeMap.get(csTypes[1]);
                            else
                                stp = tcSubtypeMap.get(csTypes[1]);
                            if (stp == null)
                                throw new NumberFormatException();
                            subtype |= (stp << 8);
                        }
                        else if (type == CHAN_TYPE_VOLTS) {
                            stp = vSubtypeMap.get(csType);
                            if (stp == null)
                                throw new NumberFormatException();
                            subtype = stp;
                        }
                    }
                    else if (jp == PROP_HWCHAN) {
                        values[jp] = -1;
                        int value = Integer.decode(pValues[jp][jc]);
                        if (type == CHAN_TYPE_TEMP) {
                            int mask = (subtype & 0xff) == MccUsb.STP_RTD
                                         ? RT_CHAN_MASK : TC_CHAN_MASK;
                            if (((1 << value) & mask) == 0)
                                throw new NumberFormatException();
                        }
                        else if (type == CHAN_TYPE_VOLTS) {
                            if (((1 << value) & V_CHAN_MASK) == 0)
                                throw new NumberFormatException();
                        }
                        else if (type == CHAN_TYPE_SWITCH) {
                            if (value < 0 || value >= N_MCC_CHANS)
                                throw new NumberFormatException();
                        }
                        values[jp] = value;
                    }
                    else if (jp == PROP_ALARM) {
                        values[jp] = -1;
                        int value = Integer.decode(pValues[jp][jc]);
                        if (value >= N_ALARMS)
                            throw new NumberFormatException();
                        values[jp] = value;
                    }
                    else {
                        values[jp] = (jp == PROP_OFFSET) ? 0F : 1F;
                        values[jp] = Float.valueOf(pValues[jp][jc]);
                    }
                        
                }
                catch (ArrayIndexOutOfBoundsException e) {
                    log.error("Hardware structure is missing " + propNames[jp]
                              + " value for " + chanData[jc].name);
                }
                catch (NumberFormatException e) {
                    log.error("Hardware structure contains invalid "
                              + propNames[jp] + " value for "
                              + chanData[jc].name);
                }
            }
            hdwData[jc] = new ChanHdw(type, subtype,
                                      (Integer)values[PROP_HWCHAN],
                                      (Integer)values[PROP_ALARM],
                                      (Float)values[PROP_OFFSET],
                                      (Float)values[PROP_SCALE]);
        }
    }


   /**
    ***************************************************************************
    **
    **  Performs additional default configuration setup.
    **
    ***************************************************************************
    */
    @Override
    void addDefaultConfiguration(Properties dfltConfig)
    {
    }


   /**
    ***************************************************************************
    **
    **  Performs additional configuration.
    **
    ***************************************************************************
    */
    @Override
    void addConfiguration()
    {
    }


   /**
    ***************************************************************************
    **
    **  Initializes the refrigeration sensors.
    **
    ***************************************************************************
    */
    @Override
    void initSensors()
    {
        /*
        **  Perform basic configuration on the first MCC DAQ box (USB-TC-AI)
        */
        int dev = 0;
        try {
            dev = MCC_DEV_TC_AI;
            mcc[dev] = new MccUsb();
            mcc[dev].open(MccUsb.USB_TC_AI_DID, null, true);
            mcc[dev].blink();
            for (int j = 0; j < N_ALARMS; j++) {
                mcc[dev].configAlarm(j, 0, 0, 0f, 0f);
            }
            mcc[dev].dioConfig(0);
            mcc[dev].dioOut(0);
            mccInited[dev] = true;
        }
        catch (UsbException e) {
            log.error("Error initializing " + DEV_NAMES[dev] + " module: " + e);
        }

        /*
        **  Perform basic configuration on the second MCC DAQ box (USB-TC)
        */
        try {
            dev = MCC_DEV_TC;
            mcc[dev] = new MccUsb();
            mcc[dev].open(MccUsb.USB_TC_DID, null, true);
            mcc[dev].blink();
            mcc[dev].dioConfig(0);
            mcc[dev].dioOut(0);
            mccInited[dev] = true;
        }
        catch (UsbException e) {
            log.error("Error initializing " + DEV_NAMES[dev] + " module: " + e);
        }

        /*
        **  Perform basic configuration on the third MCC DAQ box (USB-TEMP)
        */
        try {
            dev = MCC_DEV_TEMP;
            mcc[dev] = new MccUsb();
            mcc[dev].open(MccUsb.USB_TEMP_DID, null, true);
            mcc[dev].blink();
            mcc[dev].dioConfig(0);
            mcc[dev].dioOut(0);
            for (int j = 0; j < N_MCC_DEV_CHANS; j += 2)
                mcc[dev].setSensorType(j, MccUsb.STP_DISABLED);
            mccInited[dev] = true;
        }
        catch (UsbException e) {
            log.error("Error initializing " + DEV_NAMES[dev] + " module: " + e);
        }

        /*
        **  Perform basic configuration of the WattsUp? meter
        */
        try {
            wtu = new WattsUp();
            wtu.addListener(this);
            wtu.open();
            wattsInited = true;
            power[LOAD_POWER_CHAN] = 0F;
        }
        catch (Exception e) {
            log.error("Cannot access WattsUp meter: " + e);
        }

        /*
        **  Configure the hardware channels
        */
        for (int j = 0; j < nChan; j++) {
            Channel ch = chanData[j];
            ChanHdw hw = hdwData[j];
            dev = hw.chan / N_MCC_DEV_CHANS;
            if (!mccInited[dev]) continue;
            int chan = hw.chan % N_MCC_DEV_CHANS;
            try {
                if (hw.type == CHAN_TYPE_TEMP) {
                    int sType = hw.subtype & 0xff, tType = hw.subtype >> 8;
                    mcc[dev].setSensorType(chan, sType);
                    if (sType == MccUsb.STP_RTD) {
                        mcc[dev].setThermConnType(chan, tType);
                        mcc[dev].setExcitation(chan, MccUsb.CEX_210UA);
                        mcc[dev].setGain(chan, MccUsb.GAIN_4X);
                        for (int k = 0; k < RTD_COEFFS.length; k++)
                            mcc[dev].setCoefficient(chan, k, RTD_COEFFS[k]);
                    }
                    else
                        mcc[dev].setTCType(chan, tType);
                }
                else if (hw.type == CHAN_TYPE_VOLTS) {
                    mcc[dev].setGain(chan, hw.subtype);
                    mcc[dev].setVoltageConnType(chan, MccUsb.VCT_DIFFERENTIAL);
                }
                else if (hw.type == CHAN_TYPE_SWITCH)
                    mcc[dev].dioConfigBit(chan, 1);
                configAlarm(j);
            }
            catch (UsbException e) {
                log.error("Error configuring " + DEV_NAMES[dev] + " module: "
                          + e);
                mccInited[dev] = false;
            }
        }

        /*
        **  Start the thread that receives remote power data
        */
        (new RemotePower()).start();

        /*
        **  Start the thread that updates the channel state periodically
        */
        (new Timer()).schedule(new UpdateState(), 0L, UPDATE_PERIOD);
    }


   /**
    ***************************************************************************
    **
    **  Reads the sensor data.
    **
    **  <p>
    **  Only channels with hardware alarms are read here, since all the other
    **  channels are read every second in a timer loop.
    **
    ***************************************************************************
    */
    @Override
    void readSensors()
    {
        for (int j = 0; j < nChan; j++) {
            Channel ch = chanData[j];
            ChanHdw hw = hdwData[j];
            int dev = hw.chan / N_MCC_DEV_CHANS;
            if ((hw.type != CHAN_TYPE_TEMP && hw.type != CHAN_TYPE_VOLTS)
                  || dev != MCC_DEV_TC_AI || hw.alarm < 0) continue;
            try {
                ch.value = mcc[dev].adcIn(hw.chan % N_MCC_DEV_CHANS, 0);
                ch.value = hw.offset + hw.scale * ch.value;
            }
            catch (UsbException e) {
                if (mccInited[dev])
                    log.error("Error reading ADC: " + e);
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Changes the WattsUp? meter powered state
    **
    **  <p>
    **  Performs meter setup upon power-on.
    **
    **  @param  on  Whether the meter is powered on or not
    **
    ***************************************************************************
    */
    @Override
    public void setPowered(boolean on)
    {
        if (on) {
            try {
                wtu.setLoggedFields(1 << WattsUp.FLD_WATTS);
                wtu.setExternalLogging(1);
            }
            catch (Exception e) {
                log.error("Cannot initialize WattsUp meter: " + e);
            }
        }
        else {
            power[LOAD_POWER_CHAN] = 0F;
        }
    }


   /**
    ***************************************************************************
    **
    **  Changes the WattsUp? meter open state
    **
    ***************************************************************************
    */
    @Override
    public void setClosed()
    {
        power[LOAD_POWER_CHAN] = -1F;
        wattsInited = false;
    }


   /**
    ***************************************************************************
    **
    **  Receives data periodically from the WattsUp? meter.
    **
    **  @param  data  The array of data from the meter.
    **
    ***************************************************************************
    */
    @Override
    public void processData(float[] data)
    {
        power[LOAD_POWER_CHAN] = data[WattsUp.FLD_WATTS];
    }


   /**
    ***************************************************************************
    **
    **  Configures the alarm for one channel using its limits.
    **
    **  @param  id  The channel ID (0 - 15)
    **
    ***************************************************************************
    */
    @Override
    void configAlarm(int id)
    {
        Channel ch = chanData[id];
        ChanHdw hw = hdwData[id];
        if ((hw.type != CHAN_TYPE_TEMP && hw.type != CHAN_TYPE_VOLTS)
              || hw.alarm < 0) return;
        int dev = hw.chan / N_MCC_DEV_CHANS;
        if (dev != MCC_DEV_TC_AI) return;
        int inOptn = hw.chan % N_MCC_DEV_CHANS, outOptn = 0;
        float value1 = 0, value2 = 0;
        if (ch.checkLo) {
            outOptn = 3;
            value1 = ch.limitLo;
            if (ch.checkHi) {
                inOptn |= 0x40;
                value2 = ch.limitHi;
            }
            else {
                inOptn |= 0x20;
            }
        }
        else if (ch.checkHi) {
            outOptn = 3;
            value1 = ch.limitHi;
        }
        value1 = (value1 - hw.offset) / hw.scale;
        value2 = (value2 - hw.offset) / hw.scale;
        try {
            mcc[dev].configAlarm(hw.alarm, inOptn, outOptn, value1, value2);
            if (outOptn == 0)
                mcc[dev].dioOutBit(hw.alarm, 0);
        }
        catch (UsbException e) {
            if (mccInited[dev])
                log.error("Error configuring alarm: " + e);
        }
    }


   /**
    ***************************************************************************
    **
    **  Turns the main power on or off.
    **
    ***************************************************************************
    */
    @Override
    void setMainPower()
    {
        try {
            mcc[MAIN_POWER_DEV].dioOutBit(MAIN_POWER_BIT, mainPower ? 1 : 0);
        }
        catch (UsbException e) {
            if (mccInited[MAIN_POWER_DEV])
                log.error("Error setting DIO line: " + e);
        }
    }
            

   /**
    ***************************************************************************
    **
    **  Turns the load power on or off.
    **
    ***************************************************************************
    */
    @Override
    void setLoadPower()
    {
        try {
            mcc[LOAD_POWER_DEV].dioOutBit(LOAD_POWER_BIT, loadPower ? 1 : 0);
        }
        catch (UsbException e) {
            if (mccInited[LOAD_POWER_DEV])
                log.error("Error setting DIO line: " + e);
        }
    }

}
