package org.lsst.ccs.subsystem.refrig;

import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Properties;
import org.lsst.ccs.bus.ValueNotification;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.subsystem.refrig.data.RefrigChannel;
import org.lsst.ccs.subsystem.refrig.data.RefrigFullState;
import org.lsst.ccs.subsystem.refrig.data.RefrigState;
import org.lsst.ccs.subsystem.refrig.status.RefrigStateStatus;

/**
 ***************************************************************************
 **
 **  Implements the refrigeration long lines test modular subsystem
 **
 **  <p>
 **  This is an abstract class implementing the interactions with the CCS
 **  buses, but deferring the hardware interaction, real or simulated, to
 **  sub-classes.
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public abstract class RefrigTest extends Module {

   /**
    ***************************************************************************
    **
    **  Container class for channel parameters
    **
    ***************************************************************************
    */
    static class Channel {

        String   name;          // Channel name
        String   description;   // Channel description
        String   units;         // Units name
        boolean  checkLo;       // Whether low limit is checked
        boolean  checkHi;       // Whether high limit is checked
        float    limitLo;       // Low limit value
        float    limitHi;       // High limit value
        String   limitLoName;   // Low limit name (derived)
        String   limitHiName;   // High limit name (derived)
        float    value;         // Current value read from channel

        public Channel(String name, String desc, String units,
                       boolean loCheck, boolean hiCheck)
        {
            this.name = name;
            this.description = desc;
            this.units = units;
            this.checkLo = loCheck;
            this.checkHi = hiCheck;
            this.limitLoName = name + "LoLim";
            this.limitHiName = name + "HiLim";
        }
    }

   /**
    ***************************************************************************
    **
    **  Package constants
    **
    ***************************************************************************
    */
    final static int
        MAX_CHANS = 16;

    final static String
        PREFIX        = "org.lsst.ccs.refrig.",
        HLT_CHAN_NAME = "LoadTmp";

   /**
    ***************************************************************************
    **
    **  Package fields
    **
    ***************************************************************************
    */
    String structFile = "structure.properties",
           configFile = "config.properties";
    Properties config;
    Channel[] chanData;
    int nChan, hltChan = -1, loLimChange = 0, hiLimChange = 0, chanState = 0;
    boolean running = false, mainPower = false, loadPower = false,
            loadTrip = false;


   /**
    ***************************************************************************
    **
    **  Initializes the refrigeration subsystem
    **
    ***************************************************************************
    */
    @Override
    public void initModule()
    {
        /*
        **  Initialize all configuration data
        */
        initStructure();
        initConfiguration();

        /*
        **  Initialize the hardware
        */
        initSensors();
        setLoadLine();
    }


    /**
    ***************************************************************************
    **
    **  Performs periodic sensor readout
    **
    ***************************************************************************
    */
    @Override
    public void tick()
    {
        /*
        **  Broadcast the state if first time
        */
        if (!running) {
            System.out.println("Refrigeration test system started");
            publishState();    // For any GUIs
            publishLimits();   // For GUIs and the trending database
            running = true;
        }

        /*
        **  Read the sensors
        */
        long timeStamp = System.currentTimeMillis();
        readSensors();

        /*
        **  Broadcast the trending data
        */
        ArrayList<ValueNotification> trendingValues
          = new ArrayList<ValueNotification>();
        for (int j = 0; j < nChan; j++) {
            Channel chan = chanData[j];
            trendingValues.add(new ValueNotification(chan.name, chan.value,
                                                     timeStamp));
        }
        trendingValues.add(new ValueNotification("state", getState()));
        publishData(trendingValues);

        /*
        **  Set the heat load state correctly
        */
        setLoadLine();
    }


   /**
    ***************************************************************************
    **
    **  Sets the structural configuration file name
    **
    **  @param  name  The name of the structural configuration file
    **
    ***************************************************************************
    */
    public void setStructFile(String name)
    {
        structFile = name;
    }


   /**
    ***************************************************************************
    **
    **  Sets the dynamic configuration file name
    **
    **  @param  name  The name of the dynamic configuration file
    **
    ***************************************************************************
    */
    public void setConfigFile(String name)
    {
        configFile = name;
    }


   /**
    ***************************************************************************
    **
    **  Sets the tick period
    **
    **  <p>
    **  Overrides the method in Module in order to publish a status update.
    **
    **  @param  value  The tick period (milliseconds) to set.
    **
    ***************************************************************************
    */
    @Override
    public void setTickMillis(int value)
    {
        super.setTickMillis(value);
        if (running) publishState();
    }


    /**
    ***************************************************************************
    **
    **  Sets the power enabled state on or off
    **
    **  @param  value  The enabled state value to set: 0 = off, ~0 = on.
    **
    ***************************************************************************
    */
    public void setPowerEnable(int value)
    {
        mainPower = (value != 0);
        setMainPower();
        setLoadLine();
        publishState();
    }


   /**
    ***************************************************************************
    **
    **  Sets the load trip enabled state on or off
    **
    **  @param  value  The load trip enabled state value to set: 0 = off;
    **                 ~0 = on.
    **
    ***************************************************************************
    */
    public void setLoadTripEnable(int value)
    {
        loadTrip = (value != 0);
        setLoadLine();
        publishState();
    }


   /**
    ***************************************************************************
    **
    **  Gets the operating state word
    **
    **  @return  The current value of the operating state
    **
    ***************************************************************************
    */
    public int getState()
    {
        int value = chanState;
        if (mainPower) value |= (1 << RefrigState.MAIN_POWER_STATE_BIT);
        if (loadPower) value |= (1 << RefrigState.LOAD_POWER_STATE_BIT);
        if (loadTrip) value |= (1 << RefrigState.LOAD_TRIP_STATE_BIT);

        return value;
    }


   /**
    ***************************************************************************
    **
    **  Sets the low limit for a channel
    **
    **  @param  id     The channel ID
    **
    **  @param  limit  The low limit value.
    **
    ***************************************************************************
    */
    public void setLowLimit(int id, double limit)
    {
        Channel chan = chanData[id];
        float fLimit = (float)limit;
        if (fLimit == chan.limitLo) return;
        chan.limitLo = fLimit;
        String sLimit = String.valueOf(fLimit);
        config.setProperty(PREFIX + chan.limitLoName, sLimit);
        configAlarm(id);
        loLimChange |= 1 << id;
        publishState();
        getSubsystem().publishMetaData(chan.name, "alarmLow", sLimit);
    }


   /**
    ***************************************************************************
    **
    **  Sets the high limit for a channel
    **
    **  @param  id     The channel ID
    **
    **  @param  limit  The high limit value.
    **
    ***************************************************************************
    */
    public void setHighLimit(int id, double limit)
    {
        Channel chan = chanData[id];
        float fLimit = (float)limit;
        if (fLimit == chan.limitHi) return;
        chan.limitHi = fLimit;
        String sLimit = String.valueOf(fLimit);
        config.setProperty(PREFIX + chan.limitHiName, sLimit);
        configAlarm(id);
        hiLimChange |= 1 << id;
        publishState();
        getSubsystem().publishMetaData(chan.name, "alarmHigh", sLimit);
    }


   /**
    ***************************************************************************
    **
    **  Saves the configuration data
    **
    ***************************************************************************
    */
    public void saveConfiguration()
    {
        FileWriter writer = null;
        try {
            writer = new FileWriter(configFile);
        }
        catch (IOException e) {
            log.error("Error opening configuration file: " + e);
        }
        if (writer != null) {
            try {
                config.store(writer,
                             "Refrigeration long lines test configuration");
                loLimChange = hiLimChange = 0;
            }
            catch (IOException e) {
                log.error("Error writing configuration: " + e);
            }
            try {
                writer.close();
            }
            catch (IOException e) {
                log.error("Error closing configuration file: " + e);
            }
        }
        publishState();   // Must always do this
    }


   /**
    ***************************************************************************
    **
    **  Gets the full state of the refrigeration module
    **
    **  <p>
    **  This is intended to be called by GUIs during initialization
    **
    ***************************************************************************
    */
    public RefrigFullState getFullState()
    {
        RefrigFullState status = new RefrigFullState();
        for ( int i = 0; i < nChan; i++) {
            Channel c = chanData[i];
            status.addChannel(new RefrigChannel(c.name, c.description, c.units,
                                                c.limitLo, c.limitHi, c.value));
        }
        status.setRefrigState(new RefrigState(getState(), loLimChange,
                                              hiLimChange, getTickMillis()));
        return status;
    }    


   /**
    ***************************************************************************
    **
    **  Publishes the state of the refrigeration module
    **
    **  <p>
    **  This is intended to be called whenever any element of the state is
    **  changed.
    **
    ***************************************************************************
    */
    public void publishState()
    {
        RefrigState refrigState = new RefrigState(getState(), loLimChange,
                                                  hiLimChange, getTickMillis());
        sendToStatus(new RefrigStateStatus(refrigState));
    }    


   /**
    ***************************************************************************
    **
    **  Publishes all the limit values of the refrigeration module
    **
    **  <p>
    **  This is intended to be called at startup time.
    **
    ***************************************************************************
    */
    public void publishLimits()
    {
        for ( int i = 0; i < nChan; i++) {
            Channel c = chanData[i];
            getSubsystem().publishMetaData(c.name, "alarmLow",
                                           String.valueOf(c.limitLo));
            getSubsystem().publishMetaData(c.name, "alarmHigh",
                                           String.valueOf(c.limitHi));
        }
    }    


   /**
    ***************************************************************************
    **
    **  Initializes the structural configuration data
    **
    ***************************************************************************
    */
    private static final int
        PROP_NAME     = 0,
        PROP_DESC     = 1,
        PROP_UNITS    = 2,
        PROP_LO_CHECK = 3,
        PROP_HI_CHECK = 4,
        N_PROPS       = 5;

    private void initStructure()
    {
        Properties struct = loadProperties(structFile, null);
        try {
            nChan = Integer.decode(struct.getProperty(PREFIX + "nchan"));
            if (nChan <= 0 || nChan > MAX_CHANS)
                throw new NumberFormatException();
        }
        catch (NullPointerException e) {
            log.error("General structure is missing nchan property");
            nChan = 0;
            return;
        }
        catch (NumberFormatException e) {
            log.error("Invalid number of channels specified");
            nChan = 0;
            return;
        }

        String[] names = new String[N_PROPS];
        names[PROP_NAME]     = "names";
        names[PROP_DESC]     = "descs";
        names[PROP_UNITS]    = "units";
        names[PROP_LO_CHECK] = "lolimchecks";
        names[PROP_HI_CHECK] = "hilimchecks";

        chanData = new Channel[nChan];
        String[][] pValues = new String[N_PROPS][];
        for (int jp = 0; jp < N_PROPS; jp++) {
            String pValue = struct.getProperty(PREFIX + names[jp]);
            if (pValue != null) {
                pValues[jp] = pValue.split(",", -1);
            }
            else {
                log.error("General structure is missing " + names[jp]
                          + " property");
                pValues[jp] = new String[nChan];
                for (int jc = 0; jc < nChan; jc++)
                    pValues[jp][jc]
                      = (jp == PROP_LO_CHECK || jp == PROP_HI_CHECK) ? "1" : "";
            }
        }

        String[] values = new String[N_PROPS];
        for (int jc = 0; jc < nChan; jc++) {
            boolean loCheck = true, hiCheck = true;
            for (int jp = 0; jp < N_PROPS; jp++) {
                if (jc < pValues[jp].length) {
                    values[jp] = pValues[jp][jc];
                    if (jp == PROP_LO_CHECK || jp == PROP_HI_CHECK) {
                        boolean value = true;
                        try {
                            value = (Integer.decode(values[jp]) != 0);
                        }
                        catch (NumberFormatException e) {
                            log.error("General structure contains invalid "
                                      + names[jp] + " value for channel " + jc);
                        }
                        if (jp == PROP_LO_CHECK) loCheck = value;
                        else hiCheck = value;
                    }
                }
                else {
                    log.error("General structure is missing " + names[jp]
                              + " value for channel " + jc);
                    values[jp] = "";
                }
            }
            chanData[jc] = new Channel(values[PROP_NAME], values[PROP_DESC],
                                       values[PROP_UNITS], loCheck, hiCheck);
            if (values[0].equals(HLT_CHAN_NAME))
                hltChan = jc;
        }

        addStructure(struct);
    }


   /**
    ***************************************************************************
    **
    **  Initializes the configuration data
    **
    ***************************************************************************
    */
    private void initConfiguration()
    {
        /*
        **  Generate a properties list containing default configuration values
        */
        Properties dfltConfig = new Properties();
        for (int j = 0; j < nChan; j++) {
            Channel chan = chanData[j];
            dfltConfig.setProperty(PREFIX + chan.limitLoName, "0.0");
            dfltConfig.setProperty(PREFIX + chan.limitHiName, "0.0");
        }
        addDefaultConfiguration(dfltConfig);

        /*
        **  Create the configuration properties list and load it
        */
        config = loadProperties(configFile, dfltConfig);

        /*
        **  Extract values from the properties list
        */
        for (int j = 0; j < nChan; j++) {
            Channel chan = chanData[j];
            chan.limitLo
              = Float.valueOf(config.getProperty(PREFIX + chan.limitLoName));
            chan.limitHi
              = Float.valueOf(config.getProperty(PREFIX + chan.limitHiName));
        }
        addConfiguration();
    }


   /**
    ***************************************************************************
    **
    **  Loads a properties file
    **
    ***************************************************************************
    */
    private Properties loadProperties(String file, Properties dflt)
    {
        /*
        **  Create the properties list
        */
        Properties propList;
        if (dflt == null)
            propList = new Properties();
        else
            propList = new Properties(dflt);

        /*
        **  Open the properties file
        */
        FileReader reader = null;
        try {
            reader = new FileReader(file);
        }
        catch (FileNotFoundException e) {
            log.error("Error opening properties file: " + e);
        }

        /*
        **  Load the properties list
        */
        try {
            propList.load(reader);
        }
        catch (IOException e) {
            log.error("Error reading properties file: " + e);
        }
        try {
            reader.close();
        }
        catch (IOException e) {
            log.error("Error closing properties file: " + e);
        }

        return propList;
    }


   /**
    ***************************************************************************
    **
    **  Sets the state of the load line
    **
    ***************************************************************************
    */
    private void setLoadLine()
    {
        loadPower = mainPower
                      & (loadTrip ? (chanState & (1 << hltChan)) != 0 : true);
        setLoadPower();
    }


   /**
    ***************************************************************************
    **
    **  Performs additional structural configuration.
    **
    ***************************************************************************
    */
    abstract void addStructure(Properties struct);


   /**
    ***************************************************************************
    **
    **  Performs additional default configuration setup.
    **
    ***************************************************************************
    */
    abstract void addDefaultConfiguration(Properties dfltConfig);


   /**
    ***************************************************************************
    **
    **  Performs additional configuration.
    **
    ***************************************************************************
    */
    abstract void addConfiguration();


   /**
    ***************************************************************************
    **
    **  Initializes the refrigeration sensors.
    **
    ***************************************************************************
    */
    abstract void initSensors();


   /**
    ***************************************************************************
    **
    **  Reads the sensor data.
    **
    ***************************************************************************
    */
    abstract void readSensors();


   /**
    ***************************************************************************
    **
    **  Configures the alarm for one channel using its limits.
    **
    **  @param  id  The channel ID (0 - 15)
    **
    ***************************************************************************
    */
    abstract void configAlarm(int id);


   /**
    ***************************************************************************
    **
    **  Turns the main power on or off, according to the set value.
    **
    ***************************************************************************
    */
    abstract void setMainPower();
            

   /**
    ***************************************************************************
    **
    **  Turns the load power on or off, according to the set value.
    **
    ***************************************************************************
    */
    abstract void setLoadPower();
    
}
