package org.lsst.ccs.subsystem.refrig;

import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintStream;
import java.net.Socket;
import java.net.UnknownHostException;
import java.util.Timer;
import java.util.TimerTask;
import org.lsst.ccs.drivers.wattsup.WattsUp;
import org.lsst.ccs.utilities.conv.Convert;

/**
 ***************************************************************************
 **
 **  Server program to send WattsUp power data to DAQ program
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class RefrigPower implements WattsUp.Listener {

   /**
    ***************************************************************************
    **
    **  Package and private constants
    **
    ***************************************************************************
    */
    final static int
        POWER_PORT = 7070,
        POWER_ID   = 13572468;

    final private int
        UPDATE_PERIOD = 1;

   /**
    ***************************************************************************
    **
    **  Private fields
    **
    ***************************************************************************
    */
    private static final PrintStream out = System.out;
    private OutputStream netOut;
    private WattsUp wtu;
    private Thread mainThread = Thread.currentThread();
    private float power = 0F;

   /**
    ***************************************************************************
    **
    **  Inner class to implement timer task to send the current power value
    **
    ***************************************************************************
    */
    private class SendPower extends TimerTask {

        @Override
        public void run()
        {
            sendData(power);
        }

    }


   /**
    ***************************************************************************
    **
    **  Main program
    **
    ***************************************************************************
    */
    public static void main(String[] args)
    {
        if (args.length < 1) {
            out.println("No DAQ IP address supplied");
            System.exit(0);
        }
        (new RefrigPower()).run(args[0]);
        System.exit(0);
    }


   /**
    ***************************************************************************
    **
    **  Runs the power meter reader
    **
    **  @param daqNode  The name of the DAQ node to receive the power data
    **
    ***************************************************************************
    */
    private void run(String daqNode)
    {
        /*
        **  Open meter connection: quit if it's there's a problem
        */
        try {
            wtu = new WattsUp();
            wtu.open();
            wtu.addListener(this);
        }
        catch (Exception e) {
            out.println("Cannot access WattsUp meter: " + e);
            return;
        }

        /*
        **  Start the timer task to regularly send the current power value
        */
        (new Timer()).schedule(new SendPower(), 0L, 1000 * UPDATE_PERIOD);

        /*
        **  Loop to connect to the DAQ process and check for disconnects
        */
        while (wtu != null) {
            try {
                netOut = (new Socket(daqNode, POWER_PORT)).getOutputStream();
                out.println("Connected to DAQ node (" + daqNode + ")");
            }
            catch (UnknownHostException e) {
                out.println(e);
                break;
            }
            catch (Exception e) {
            }
            while (true) {
                try {
                    Thread.sleep(10000);
                }
                catch (InterruptedException e) {
                }
                if (netOut == null) break;
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Receives notification of powered state change
    **
    **  @param  on  Whether power is on or not
    **
    ***************************************************************************
    */
    @Override
    public void setPowered(boolean on)
    {
        if (on) {
            try {
                wtu.setLoggedFields(1 << WattsUp.FLD_WATTS);
                wtu.setExternalLogging(UPDATE_PERIOD);
            }
            catch (Exception e) {
                out.println("Cannot initialize WattsUp meter: " + e);
            }
        }
        else {
            sendData(power = 0F);
        }
    }


   /**
    ***************************************************************************
    **
    **  Receives notification of device closure
    **
    ***************************************************************************
    */
    @Override
    public void setClosed()
    {
        out.println("WattsUp meter disconnected");
        wtu = null;
        sendData(power = -1F);
        closeNet();
    }


   /**
    ***************************************************************************
    **
    **  Receives data periodically from the WattsUp? meter.
    **
    **  @param  data  The array of data from the meter.
    **
    ***************************************************************************
    */
    @Override
    public void processData(float[] data)
    {
        power = data[WattsUp.FLD_WATTS];
    }


   /**
    ***************************************************************************
    **
    **  Sends data to the DAQ node
    **
    ***************************************************************************
    */
    private void sendData(float value)
    {
        if (netOut != null) {
            byte[] mesg = new byte[8];
            Convert.intToBytes(POWER_ID, mesg, 0);
            Convert.floatToBytes(value, mesg, 4);
            try {
                netOut.write(mesg);
            }
            catch (IOException e) {
                out.println("Lost connection to DAQ node");
                closeNet();
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Closes the connection to the DAQ node
    **
    ***************************************************************************
    */
    private void closeNet()
    {
        try {
            netOut.close();
        }
        catch (IOException e) {
        }
        netOut = null;
        mainThread.interrupt();
    }

}
