package org.lsst.ccs.subsystem.refrig;

import java.util.Properties;
import java.util.Timer;
import java.util.TimerTask;

/**
 ***************************************************************************
 **
 **  Implements the simulated refrigeration long lines test modular subsystem
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class RefrigTestSim extends RefrigTest {

   /**
    ***************************************************************************
    **
    **  Container class for channel simulation data.
    **
    ***************************************************************************
    */
    private static class ChanSim {

        float onValue;       // On (operating) value
        float onTime;        // Time (secs) to reach on value
        float offValue;      // Off (ambient) value
        float offTime;       // Time (secs) to reach off value
        float startVal;      // Starting value
        float endVal;        // Ending value
        float rate;          // Rate of change

        public ChanSim(float onValue, float onTime, float offValue,
                       float offTime)
        {
            this.onValue   = onValue;
            this.onTime    = onTime;
            this.offValue  = offValue;
            this.offTime   = offTime;
            this.startVal  = 0f;
            this.endVal    = 0f;
            this.rate      = 0f;
        }
    }

   /**
    ***************************************************************************
    **
    **  Inner class to implement a timer task to update the channel state
    **  in a timely manner.
    **
    ***************************************************************************
    */
    private class UpdateState extends TimerTask {

        @Override
        public void run()
        {
            if (!running) return;   // Do nothing if not yet running
            readSensors();
            if (chanState != lastChanState) {
                lastChanState = chanState;
                tick();
            }
        }
    }

                
   /**
    ***************************************************************************
    **
    **  Private constants.
    **
    ***************************************************************************
    */
    private final static long
        UPDATE_PERIOD = 1000;

   /**
    ***************************************************************************
    **
    **  Private fields.
    **
    ***************************************************************************
    */
    private int lastChanState = 0;
    private ChanSim[] simData;
    private long baseTime = System.currentTimeMillis();


   /**
    ***************************************************************************
    **
    **  Performs additional structural configuration.
    **
    ***************************************************************************
    */
    private static final int
        PROP_ON_VALUE  = 0,
        PROP_ON_TIME   = 1,
        PROP_OFF_VALUE = 2,
        PROP_OFF_TIME  = 3,
        N_PROPS        = 4;

    @Override
    void addStructure(Properties struct)
    {
        String[] names = new String[N_PROPS];
        names[PROP_ON_VALUE]  = "onvalues";
        names[PROP_ON_TIME]   = "ontimes";
        names[PROP_OFF_VALUE] = "offvalues";
        names[PROP_OFF_TIME]  = "offtimes";

        simData = new ChanSim[nChan];
        String[][] pValues = new String[N_PROPS][];
        for (int jp = 0; jp < N_PROPS; jp++) {
            String pValue = struct.getProperty(PREFIX + names[jp]);
            if (pValue != null) {
                pValues[jp] = pValue.split(",", -1);
            }
            else {
                log.error("Simulator structure is missing " + names[jp]
                          + " property");
                pValues[jp] = new String[nChan];
                for (int jc = 0; jc < nChan; jc++)
                    pValues[jp][jc] = "0";
            }
        }

        float[] values = new float[N_PROPS];
        for (int jc = 0; jc < nChan; jc++) {
            for (int jp = 0; jp < N_PROPS; jp++) {
                try {
                    values[jp] = Float.valueOf(pValues[jp][jc]);
                }
                catch (ArrayIndexOutOfBoundsException e) {
                    log.error("Simulator structure is missing " + names[jp]
                              + " value for " + chanData[jc].name);
                    values[jp] = 0F;
                }
                catch (NumberFormatException e) {
                    log.error("Simulator structure contains invalid "
                              + names[jp] + " value for " + chanData[jc].name);
                    values[jp] = 0F;
                }
            }
            simData[jc] = new ChanSim(values[PROP_ON_VALUE],
                                      values[PROP_ON_TIME],
                                      values[PROP_OFF_VALUE],
                                      values[PROP_OFF_TIME]);
        }

    }


   /**
    ***************************************************************************
    **
    **  Performs additional default configuration setup.
    **
    ***************************************************************************
    */
    @Override
    void addDefaultConfiguration(Properties dfltConfig)
    {
    }


   /**
    ***************************************************************************
    **
    **  Performs additional configuration.
    **
    ***************************************************************************
    */
    @Override
    void addConfiguration()
    {
    }


   /**
    ***************************************************************************
    **
    **  Initializes the simulated sensors.
    **
    ***************************************************************************
    */
    @Override
    void initSensors()
    {
        /*
        **  Set initial simulator values
        */
        for (int j = 0; j < nChan; j++) {
            ChanSim sim = simData[j];
            sim.startVal = sim.offValue;
            sim.endVal   = sim.offValue;
            chanData[j].value = sim.offValue;
        }

        /*
        **  Start the timer task which updates the channel state periodically
        */
        (new Timer()).schedule(new UpdateState(), 0L, UPDATE_PERIOD);
    }


   /**
    ***************************************************************************
    **
    **  Reads the simulated sensors.
    **
    ***************************************************************************
    */
    @Override
    void readSensors()
    {
        int cState = (1 << nChan) - 1;

        for (int j = 0; j < nChan; j++) {
            Channel ch = chanData[j];
            ChanSim sim = simData[j];
            float intvl = (float)((System.currentTimeMillis() - baseTime)
                                    / 1000.0);
            ch.value = sim.startVal + sim.rate * intvl;
            if (sim.rate < 0) {
                if (ch.value < sim.endVal) ch.value = sim.endVal;
            }
            else {
                if (ch.value > sim.endVal) ch.value = sim.endVal;
            }
            if ((ch.checkLo && ch.value < ch.limitLo)
                  || (ch.checkHi && ch.value > ch.limitHi))
                cState &= ~(1 << j);
        }

        chanState = cState;
    }


   /**
    ***************************************************************************
    **
    **  Configures the alarm for one channel using its limits
    **
    **  @param  id  The channel ID (0 - 7)
    **
    ***************************************************************************
    */
    @Override
    void configAlarm(int id)
    {
    }


   /**
    ***************************************************************************
    **
    **  Turns the main power on or off.
    **
    ***************************************************************************
    */
    @Override
    void setMainPower()
    {
        baseTime = System.currentTimeMillis();
        for (int j = 0; j < nChan; j++) {
            Channel ch = chanData[j];
            ChanSim sim = simData[j];
            sim.rate = 0F;
            if (mainPower) {
                sim.endVal = sim.onValue;
                if (sim.onTime != 0F)
                    sim.rate = (sim.onValue - sim.offValue) / sim.onTime;
            }
            else {
                sim.endVal = sim.offValue;
                if (sim.offTime != 0F)
                    sim.rate = (sim.offValue - sim.onValue) / sim.offTime;
            }
            sim.startVal = (sim.rate != 0F) ? ch.value : sim.endVal;
        }
    }
            

   /**
    ***************************************************************************
    **
    **  Turns the load power on or off.
    **
    ***************************************************************************
    */
    @Override
    void setLoadPower()
    {
    }

}
