package org.lsst.ccs.subsystem.refrig;

import java.io.IOException;
import java.io.InputStream;
import java.net.ServerSocket;
import java.util.Properties;
import java.util.Timer;
import java.util.TimerTask;
import javax.usb.UsbException;
import org.lsst.ccs.drivers.mcc.MccTc;
import org.lsst.ccs.drivers.mcc.MccTcAi;
import org.lsst.ccs.drivers.wattsup.WattsUp;
import org.lsst.ccs.utilities.conv.Convert;

/**
 ***************************************************************************
 **
 **  Implements the refrigeration long lines test modular subsystem
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class RefrigTestHdw extends RefrigTest implements WattsUp.Listener {

   /**
    ***************************************************************************
    **
    **  Container class for channel hardware data.
    **
    ***************************************************************************
    */
    private static class ChanHdw {

        int    type;       // Channel type (Temperature, voltage, power...)
        int    chan;       // Hardware channel for MCC channel
        int    alarm;      // Hardware alarm number
        float  offset;     // Value offset
        float  scale;      // Value scale

        public ChanHdw(int type, int chan, int alarm, float offset, float scale)
        {
            this.type   = type;
            this.chan   = chan;
            this.alarm  = alarm;
            this.offset = offset;
            this.scale  = scale;
        }
    }

   /**
    ***************************************************************************
    **
    **  Inner class implementing task to obtain remote WattsUp meter data
    **
    ***************************************************************************
    */
    private class RemotePower extends Thread {

        @Override
        public void run()
        {
            ServerSocket server;
            try {
                server = new ServerSocket(RefrigPower.POWER_PORT, 1);
            }
            catch (IOException e) {
                log.error("Error creating remote power listener: " + e);
                return;
            }

            byte[] data = new byte[8];
            InputStream netInp;

            while (true) {
                try {
                    netInp = server.accept().getInputStream();
                }
                catch (IOException e) {
                    log.error("Error creating remote power socket: " + e);
                    continue;
                }
                while (true) {
                    int nRead;
                    try {
                        nRead = netInp.read(data);
                    }
                    catch (IOException e) {
                        log.error("Error reading remote power data: " + e);
                        nRead = -2;
                    }
                    if (nRead < 0) {
                        try {
                            netInp.close();
                        }
                        catch (IOException e) {
                        }
                        power[CMPR_POWER_CHAN] = -1F;
                        break;
                    }
                    if (nRead < data.length)
                        continue;
                    if (Convert.bytesToInt(data, 0) != RefrigPower.POWER_ID)
                        continue;
                    power[CMPR_POWER_CHAN] = Convert.bytesToFloat(data, 4);
                }
            }
        }

    }

   /**
    ***************************************************************************
    **
    **  Inner class to implement timer task to update the channel state and
    **  the green state LEDs.
    **
    ***************************************************************************
    */
    private class UpdateState extends TimerTask {

        @Override
        public void run()
        {
            /*
            **  Read channels without hardware alarms and check them
            */
            int cState = 0;
            for (int j = 0; j < nChan; j++) {
                Channel ch = chanData[j];
                ChanHdw hw = hdwData[j];
                if (hw.alarm >= 0 && hw.chan >= 0 && hw.chan < N_MCC_CHAN0)
                    continue;
                if (hw.type == CHAN_TYPE_TEMP || hw.type == CHAN_TYPE_VOLTS) {
                    try {
                        if (hw.chan < N_MCC_CHAN0)
                            ch.value = tcai.adcIn(hw.chan, 0);
                        else
                            ch.value = tc.adcIn(hw.chan - N_MCC_CHAN0, 0);
                        ch.value = hw.offset + hw.scale * ch.value;
                    }
                    catch (UsbException e) {
                        if ((hw.chan < N_MCC_CHAN0 && mccTcAiInited)
                              || (hw.chan >= N_MCC_CHAN0 && mccTcInited))
                            log.error("Error reading ADC: " + e);
                    }
                }
                else if (hw.type == CHAN_TYPE_SWITCH) {
                    try {
                        if (hw.chan < N_MCC_CHAN0)
                            ch.value = tcai.dioInBit(hw.chan);
                        else
                            ch.value = tc.dioInBit(hw.chan - N_MCC_CHAN0);
                    }
                    catch (UsbException e) {
                        if ((hw.chan < N_MCC_CHAN0 && mccTcAiInited)
                              || (hw.chan >= N_MCC_CHAN0 && mccTcInited))
                            log.error("Error reading DIO line: " + e);
                    }
                }
                else {
                    ch.value = power[hw.chan];
                }
                boolean ok = (!ch.checkLo || ch.value >= ch.limitLo)
                               && (!ch.checkHi || ch.value <= ch.limitHi);
                if (ok) cState |= (1 << j);
                if (hw.alarm >= 0) {
                    try {
                        tcai.dioOutBit(hw.alarm, ok ? 0 : 1);
                    }
                    catch (UsbException e) {
                        if (mccTcAiInited)
                            log.error("Error setting DIO line: " + e);
                    }
                }
            }
                    
            /*
            **  Read LED/alarm lines and update other color
            */
            int lines = 0;
            try {
                lines = tcai.dioIn();
                try {
                    tc.dioOut((lines ^ LEDS_MASK) & LEDS_MASK);
                }
                catch (UsbException e) {
                    if (mccTcInited) log.error("Error writing DIO lines: " + e);
                }
            }
            catch (UsbException e) {
                if (mccTcAiInited) log.error("Error reading DIO lines: " + e);
            }

            /*
            **  Update state from alarm lines
            */
            for (int j = 0; j < nChan; j++) {
                int alarm = hdwData[j].alarm;
                if (alarm >= 0 && ((1 << alarm) & lines) == 0)
                    cState |= (1 << j);
            }

            /*
            **  Do full readout and report if state changed and system running
            */
            if (cState != chanState && running) {
                chanState = cState;
                tick();
            }
        }

    }

   /**
    ***************************************************************************
    **
    **  Private constants.
    **
    ***************************************************************************
    */
    private final static int
        N_MCC_CHANS = 16,
        N_MCC_CHAN0 = 8,
        MCC_V_CHAN0 = 4,
        V_CHAN_MASK = ((1 << N_MCC_CHAN0) - 1) ^ ((1 << MCC_V_CHAN0) - 1),
        N_ALARMS    = 6,
        MAIN_POWER_BIT = 7,
        LOAD_POWER_BIT = 6,
        ALARMS_MASK = ((1 << N_ALARMS) - 1),
        LEDS_MASK = ALARMS_MASK | (1 << LOAD_POWER_BIT),
        LOAD_POWER_CHAN = 0,
        CMPR_POWER_CHAN = 1,
        CHAN_TYPE_UNKNOWN = -1,
        CHAN_TYPE_TEMP    = 0,
        CHAN_TYPE_VOLTS   = 1,
        CHAN_TYPE_POWER   = 2,
        CHAN_TYPE_SWITCH  = 3;

    private final static long
        UPDATE_PERIOD = 1000;

   /**
    ***************************************************************************
    **
    **  Private fields.
    **
    ***************************************************************************
    */
    private MccTcAi tcai;
    private MccTc tc;
    private WattsUp wtu;
    private float[] power = {-1F, -1F};
    private boolean mccTcAiInited, mccTcInited, wattsInited;
    private ChanHdw[] hdwData;


   /**
    ***************************************************************************
    **
    **  Performs additional structural configuration.
    **
    ***************************************************************************
    */
    private static final int
        PROP_TYPE   = 0,
        PROP_HWCHAN = 1,
        PROP_ALARM  = 2,
        PROP_OFFSET = 3,
        PROP_SCALE  = 4,
        N_PROPS     = 5;

    @Override
    void addStructure(Properties struct)
    {
        String[] names = new String[N_PROPS];
        names[PROP_TYPE]   = "types";
        names[PROP_HWCHAN] = "hwchans";
        names[PROP_ALARM]  = "alarms";
        names[PROP_OFFSET] = "offsets";
        names[PROP_SCALE]  = "scales";

        hdwData = new ChanHdw[nChan];
        String[][] pValues = new String[N_PROPS][];
        for (int jp = 0; jp < N_PROPS; jp++) {
            String pValue = struct.getProperty(PREFIX + names[jp]);
            if (pValue != null) {
                pValues[jp] = pValue.split(",", -1);
            }
            else {
                log.error("Hardware structure is missing " + names[jp]
                          + " property");
                pValues[jp] = new String[nChan];
                for (int jc = 0; jc < nChan; jc++)
                    pValues[jp][jc] = "-1";
            }
        }

        Object[] values = new Object[N_PROPS];
        for (int jc = 0; jc < nChan; jc++) {
            int type = 0;
            for (int jp = 0; jp < N_PROPS; jp++) {
                try {
                    if (jp == PROP_TYPE) {
                        String cType = pValues[jp][jc].toUpperCase();
                        type = cType.equals("TEMP") ? CHAN_TYPE_TEMP
                               : cType.equals("VOLTS") ? CHAN_TYPE_VOLTS
                               : cType.equals("POWER") ? CHAN_TYPE_POWER
                               : cType.equals("SWITCH") ? CHAN_TYPE_SWITCH
                               : CHAN_TYPE_UNKNOWN;
                        if (type == CHAN_TYPE_UNKNOWN)
                            throw new NumberFormatException();
                    }
                    else if (jp == PROP_HWCHAN || jp == PROP_ALARM) {
                        values[jp] = -1;
                        int value = Integer.decode(pValues[jp][jc]);
                        if (jp == PROP_HWCHAN) {
                            if (type == CHAN_TYPE_TEMP) {
                                if (value < 0 || value >= N_MCC_CHANS
                                      || ((1 << value) & V_CHAN_MASK) != 0)
                                    throw new NumberFormatException();
                            }
                            else if (type == CHAN_TYPE_VOLTS) {
                                if (value < 0 || value >= N_MCC_CHANS
                                      || ((1 << value) & V_CHAN_MASK) == 0)
                                    throw new NumberFormatException();
                            }
                        }
                        else {
                            if (value >= N_ALARMS)
                                throw new NumberFormatException();
                        }
                        values[jp] = value;
                    }
                    else {
                        values[jp] = (jp == PROP_OFFSET) ? 0F : 1F;
                        values[jp] = Float.valueOf(pValues[jp][jc]);
                    }
                        
                }
                catch (ArrayIndexOutOfBoundsException e) {
                    log.error("Hardware structure is missing " + names[jp]
                              + " value for " + chanData[jc].name);
                }
                catch (NumberFormatException e) {
                    log.error("Hardware structure contains invalid "
                              + names[jp] + " value for " + chanData[jc].name);
                }
            }
            hdwData[jc] = new ChanHdw(type, (Integer)values[PROP_HWCHAN],
                                      (Integer)values[PROP_ALARM],
                                      (Float)values[PROP_OFFSET],
                                      (Float)values[PROP_SCALE]);
        }
    }


   /**
    ***************************************************************************
    **
    **  Performs additional default configuration setup.
    **
    ***************************************************************************
    */
    @Override
    void addDefaultConfiguration(Properties dfltConfig)
    {
    }


   /**
    ***************************************************************************
    **
    **  Performs additional configuration.
    **
    ***************************************************************************
    */
    @Override
    void addConfiguration()
    {
    }


   /**
    ***************************************************************************
    **
    **  Initializes the refrigeration sensors.
    **
    ***************************************************************************
    */
    @Override
    void initSensors()
    {
        /*
        **  Connect to and configure the first MCC DAQ box (USB-TC-AI)
        */
        try {
            tcai = new MccTcAi();
            tcai.open(null, true);
            tcai.blink();
            for (int j = 0; j < N_ALARMS; j++) {
                tcai.configAlarm(j, 0, 0, 0f, 0f);
            }
            tcai.dioConfig(0);
            tcai.dioOut(0);
            int dioCfg = 0;
            for (int j = 0; j < nChan; j++) {
                Channel ch = chanData[j];
                ChanHdw hw = hdwData[j];
                if (hw.chan >= N_MCC_CHAN0) continue;
                if (hw.type == CHAN_TYPE_TEMP)
                    tcai.setTCType(hw.chan, MccTcAi.TC_TYPE_T);
                else if (hw.type == CHAN_TYPE_VOLTS) {
                    tcai.setGain(hw.chan, MccTcAi.RANGE_1_25V);
                    tcai.setVoltageConnType(hw.chan,
                                            MccTcAi.VCT_DIFFERENTIAL);
                }
                else if (hw.type == CHAN_TYPE_SWITCH)
                    tcai.dioConfigBit(hw.chan, 1);
                configAlarm(j);
            }
            mccTcAiInited = true;
        }
        catch (UsbException e) {
            log.error("Error initializing USB-TC-AI module: " + e);
        }

        /*
        **  Connect to and configure the second MCC DAQ box (USB-TC)
        */
        try {
            tc = new MccTc();
            tc.open(null, true);
            tc.blink();
            tc.dioConfig(0);
            tc.dioOut(0);
            for (int j = 0; j < nChan; j++) {
                Channel ch = chanData[j];
                ChanHdw hw = hdwData[j];
                int hwChan = hw.chan - N_MCC_CHAN0;
                if (hwChan < 0) continue;
                if (hw.type == CHAN_TYPE_TEMP)
                    tc.setTCType(hwChan, MccTc.TC_TYPE_T);
                else if (hw.type == CHAN_TYPE_SWITCH)
                    tc.dioConfigBit(hwChan, 1);
            }
            mccTcInited = true;
        }
        catch (UsbException e) {
            log.error("Error initializing USB_TC module: " + e);
        }

        /*
        **  Connect to the WattsUp? meter
        */
        try {
            wtu = new WattsUp();
            wtu.addListener(this);
            wtu.open();
            wattsInited = true;
            power[LOAD_POWER_CHAN] = 0F;
        }
        catch (Exception e) {
            log.error("Cannot access WattsUp meter: " + e);
        }

        /*
        **  Start the thread that receives remote power data
        */
        (new RemotePower()).start();

        /*
        **  Start the thread that updates the channel state periodically
        */
        (new Timer()).schedule(new UpdateState(), 0L, UPDATE_PERIOD);
    }


   /**
    ***************************************************************************
    **
    **  Reads the sensor data.
    **
    **  <p>
    **  Only channels with hardware alarms are read here, since all the other
    **  channels are read every second in a timer loop.
    **
    ***************************************************************************
    */
    @Override
    void readSensors()
    {
        for (int j = 0; j < nChan; j++) {
            Channel ch = chanData[j];
            ChanHdw hw = hdwData[j];
            if ((hw.type == CHAN_TYPE_TEMP || hw.type == CHAN_TYPE_VOLTS)
                   && hw.alarm >= 0 && hw.chan < N_MCC_CHAN0) {
                try {
                    ch.value = tcai.adcIn(hw.chan, 0);
                    ch.value = hw.offset + hw.scale * ch.value;
                }
                catch (UsbException e) {
                    if (mccTcAiInited)
                        log.error("Error reading ADC: " + e);
                }
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Changes the WattsUp? meter powered state
    **
    **  <p>
    **  Performs meter setup upon power-on.
    **
    **  @param  on  Whether the meter is powered on or not
    **
    ***************************************************************************
    */
    @Override
    public void setPowered(boolean on)
    {
        if (on) {
            try {
                wtu.setLoggedFields(1 << WattsUp.FLD_WATTS);
                wtu.setExternalLogging(1);
            }
            catch (Exception e) {
                log.error("Cannot initialize WattsUp meter: " + e);
            }
        }
        else {
            power[LOAD_POWER_CHAN] = 0F;
        }
    }


   /**
    ***************************************************************************
    **
    **  Changes the WattsUp? meter open state
    **
    ***************************************************************************
    */
    @Override
    public void setClosed()
    {
        power[LOAD_POWER_CHAN] = -1F;
        wattsInited = false;
    }


   /**
    ***************************************************************************
    **
    **  Receives data periodically from the WattsUp? meter.
    **
    **  @param  data  The array of data from the meter.
    **
    ***************************************************************************
    */
    @Override
    public void processData(float[] data)
    {
        power[LOAD_POWER_CHAN] = data[WattsUp.FLD_WATTS];
    }


   /**
    ***************************************************************************
    **
    **  Configures the alarm for one channel using its limits.
    **
    **  @param  id  The channel ID (0 - 15)
    **
    ***************************************************************************
    */
    @Override
    void configAlarm(int id)
    {
        Channel ch = chanData[id];
        ChanHdw hw = hdwData[id];
        if (hw.type != CHAN_TYPE_TEMP && hw.type != CHAN_TYPE_VOLTS) return;
        if (hw.chan >= N_MCC_CHAN0 || hw.alarm < 0) return;
        int inOptn = hw.chan, outOptn = 0;
        float value1 = 0, value2 = 0;
        if (ch.checkLo) {
            outOptn = 3;
            value1 = ch.limitLo;
            if (ch.checkHi) {
                inOptn |= 0x40;
                value2 = ch.limitHi;
            }
            else {
                inOptn |= 0x20;
            }
        }
        else if (ch.checkHi) {
            outOptn = 3;
            value1 = ch.limitHi;
        }
        value1 = (value1 - hw.offset) / hw.scale;
        value2 = (value2 - hw.offset) / hw.scale;
        try {
            tcai.configAlarm(hw.alarm, inOptn, outOptn, value1, value2);
            if (outOptn == 0) tcai.dioOutBit(hw.alarm, 0);
        }
        catch (UsbException e) {
            if (mccTcAiInited) log.error("Error configuring alarm: " + e);
        }
    }


   /**
    ***************************************************************************
    **
    **  Turns the main power on or off.
    **
    ***************************************************************************
    */
    @Override
    void setMainPower()
    {
        try {
            tcai.dioOutBit(MAIN_POWER_BIT, mainPower ? 1 : 0);
        }
        catch (UsbException e) {
            if (mccTcAiInited) log.error("Error setting DIO line: " + e);
        }
    }
            

   /**
    ***************************************************************************
    **
    **  Turns the load power on or off.
    **
    ***************************************************************************
    */
    @Override
    void setLoadPower()
    {
        try {
            tcai.dioOutBit(LOAD_POWER_BIT, loadPower ? 1 : 0);
        }
        catch (UsbException e) {
            if (mccTcAiInited) log.error("Error setting DIO line: " + e);
        }
    }

}
