package org.lsst.ccs.subsystem.refrig.ui;

import java.awt.Color;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.util.HashMap;
import java.util.Map;
import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.SwingUtilities;
import org.lsst.ccs.subsystem.common.ui.jas.CommandSender;
import org.lsst.ccs.subsystem.common.ui.SystemStatusPanel;
import org.lsst.ccs.subsystem.common.ui.TextFieldX;
import org.lsst.ccs.subsystem.common.ui.UiConstants;
import org.lsst.ccs.subsystem.common.ui.UiUtilities;
import org.lsst.ccs.subsystem.refrig.constants.PcpHeaterState;
import org.lsst.ccs.subsystem.refrig.constants.PcpHeaters;
import org.lsst.ccs.subsystem.refrig.constants.ConditionState;
import org.lsst.ccs.subsystem.refrig.constants.LatchState;
import org.lsst.ccs.subsystem.refrig.constants.MonitorControl;
import org.lsst.ccs.subsystem.refrig.constants.PcpConditions;
import org.lsst.ccs.subsystem.refrig.constants.PcpLatches;
import org.lsst.ccs.subsystem.refrig.constants.PcpLimits;
import org.lsst.ccs.subsystem.refrig.constants.PcpSwitches;
import org.lsst.ccs.subsystem.refrig.constants.PLCState;
import org.lsst.ccs.subsystem.refrig.constants.PcpSwitchState;
import org.lsst.ccs.subsystem.refrig.constants.SwitchState;
import org.lsst.ccs.subsystem.refrig.data.PcpSysState;

/**
 *  Implements the PCP monitoring panel.
 *
 *  @author Owen Saxton
 */
public class PcpControlPanel extends JPanel implements UiUtilities.ActionHandler, CommandSender.ReplyHandler {

    private static final String CMND_GET_STATE = "getSystemState";
    private static final String[] switchNames = new String[PcpSwitches.NUM_SWITCHES];
    static {
        switchNames[PcpSwitches.SW_BLOCK_COLD_HEAT] = "Block Cold Heater";
        switchNames[PcpSwitches.SW_BLOCK_COLD_REFG] = "Block Cold Refrig";
    }
    private static final Map<PcpSwitchState, Color> switchColors = new HashMap<>();
    static {
        switchColors.put(PcpSwitchState.OFF,     Color.BLACK);
        switchColors.put(PcpSwitchState.ON,      UiConstants.GREEN);
        switchColors.put(PcpSwitchState.ALARM,   UiConstants.RED);
        switchColors.put(PcpSwitchState.OFFLINE, UiConstants.BLUE);
    }
    private static final String[] latchNames = new String[PcpLatches.NUM_LATCHES];
    static {
        latchNames[PcpLatches.LATCH_COLD_TEMP_HIGH] = "Cold Temp > 99C";
        latchNames[PcpLatches.LATCH_COLD_TEMP_LOW]  = "Cold Temp < -99C";
        latchNames[PcpLatches.LATCH_SMOKE_DETC]     = "Smoke Detc";
    }
    private static final String[] conditionNames = new String[PcpConditions.NUM_CONDITIONS];
    static {
        conditionNames[PcpConditions.COND_COLD_REFG] = "Refrig Permit";
        conditionNames[PcpConditions.COND_HEAT_PS1]  = "Heater PS1 Enable";
        conditionNames[PcpConditions.COND_HEAT_PS2]  = "Heater PS2 Enable";
        conditionNames[PcpConditions.COND_HEAT_PS3]  = "Heater PS3 Enable";
    }
    private static final String[] heaterNames = new String[PcpHeaters.NUM_HEATERS];
    static {
        heaterNames[PcpHeaters.HEATER_MINUS_Y] = "-Y";
        heaterNames[PcpHeaters.HEATER_PLUS_Y] = "+Y";
        heaterNames[PcpHeaters.HEATER_CENTER] = "Center";
    }
    private static final int SWITCH_LABEL_WIDTH = UiUtilities.maxLabelWidth(heaterNames, ":");
    private static final int SWITCH_STATE_WIDTH = UiUtilities.maxEnumLabelWidth(PcpHeaterState.class);
    private static final Map<PcpHeaterState, Color> stateColors = new HashMap<>();
    static {
        stateColors.put(PcpHeaterState.OFF, Color.BLACK);
        stateColors.put(PcpHeaterState.ON, UiConstants.GREEN);
        stateColors.put(PcpHeaterState.OFFLINE, UiConstants.BLUE);
        stateColors.put(PcpHeaterState.DISABLD, UiConstants.RED);
        stateColors.put(PcpHeaterState.VOLTERR, UiConstants.PURPLE);
        stateColors.put(PcpHeaterState.CANTSET, UiConstants.PURPLE);
        stateColors.put(PcpHeaterState.NOLOAD, UiConstants.PURPLE);
    }

    private SystemStatusPanel systemPanel;

    private JPanel plcPanel;
    private JLabel plcStatus;

    private JPanel switchPanel;
    private final JLabel[] switchStatus = new JLabel[PcpSwitches.NUM_SWITCHES];
    private final ButtonGroup[] switchBG = new ButtonGroup[PcpSwitches.NUM_SWITCHES];
    private final JRadioButton[] switchOffRB = new JRadioButton[PcpSwitches.NUM_SWITCHES];
    private final JRadioButton[] switchOnRB = new JRadioButton[PcpSwitches.NUM_SWITCHES];

    private JPanel latchPanel;
    private final JLabel[] latchLabel = new JLabel[PcpLatches.NUM_LATCHES];
    private final JLabel[] latchStatus = new JLabel[PcpLatches.NUM_LATCHES];
    private final JButton[] latchResetBtn = new JButton[PcpLatches.NUM_LATCHES];

    private JPanel conditionPanel;
    private final JLabel[] conditionStatus = new JLabel[PcpConditions.NUM_CONDITIONS];

    private JPanel heaterPanel;
    private final JLabel[] htrSwitchLabel = new JLabel[PcpHeaters.NUM_HEATERS];
    private final JLabel[] htrSwitchStatus = new JLabel[PcpHeaters.NUM_HEATERS];
    private final JLabel[] powerUnitsLabel = new JLabel[PcpHeaters.NUM_HEATERS];
    private final ButtonGroup[] htrSwitchBG = new ButtonGroup[PcpHeaters.NUM_HEATERS];
    private final JRadioButton[] htrSwitchOffRB = new JRadioButton[PcpHeaters.NUM_HEATERS];
    private final JRadioButton[] htrSwitchOnRB = new JRadioButton[PcpHeaters.NUM_HEATERS];
    private final TextFieldX[] powerTextField = new TextFieldX[PcpHeaters.NUM_HEATERS];

    private final CommandSender sender;
    private final UiUtilities uiUtils;
    private boolean gotTempLimits = false;

    public PcpControlPanel(String subsys) {
        sender = new CommandSender(subsys, this);
        uiUtils = new UiUtilities(this);
        initComponents();
        (new DisablePanel()).run();
    }

    public void initPanel() {
        sender.sendCommand(true, null, CMND_GET_STATE);
    }

    @Override
    public void onCommandReply(Object reply, String path, String command, Object[] args) {
        updatePanel((PcpSysState)reply);
    }

    @Override
    public void onCommandReject(String path, String command, Object[] args) {
        if (!command.equals(CMND_GET_STATE)) {
            sender.sendCommand(true, null, CMND_GET_STATE);
        }
    }

    public void updatePanel(PcpSysState rs) {
        SwingUtilities.invokeLater(new UpdatePcpStatus(rs));
    }

    public void disablePanel() {
        SwingUtilities.invokeLater(new DisablePanel());
    }

    private void initComponents() {

        // System status
        systemPanel = new SystemStatusPanel(sender, MonitorControl.NODE_NAME, true);

        // PLC state
        plcStatus = UiUtilities.newLabel("XXX", UiUtilities.maxEnumLabelWidth(PLCState.class));

        // PLC switch states
        for (int sw = 0; sw < PcpSwitches.NUM_SWITCHES; sw++) {
            switchStatus[sw] = UiUtilities.newLabel("XXX", UiUtilities.maxEnumLabelWidth(SwitchState.class));
            switchOffRB[sw] = uiUtils.newRadioButton("Off", "PF" + sw);
            switchOnRB[sw] = uiUtils.newRadioButton("On", "PT" + sw);
            ButtonGroup bg = switchBG[sw] = new ButtonGroup();
            bg.add(switchOffRB[sw]);
            bg.add(switchOnRB[sw]);
        }

        // PLC latched condition states
        for (int cond = 0; cond < PcpLatches.NUM_LATCHES; cond++) {
            latchLabel[cond] = UiUtilities.newLabel(latchNames[cond] + ":", 0);
            latchStatus[cond] = UiUtilities.newLabel("XXX", UiUtilities.maxEnumLabelWidth(LatchState.class));
            latchResetBtn[cond] = uiUtils.newButton("Reset", String.valueOf(cond), true);
        }

        // PLC condition states
        for (int cond = 0; cond < PcpConditions.NUM_CONDITIONS; cond++) {
            conditionStatus[cond] = UiUtilities.newLabel("XXX", UiUtilities.maxEnumLabelWidth(ConditionState.class));
        }

        // Heater states
        for (int htr = 0; htr < PcpHeaters.NUM_HEATERS; htr++) {
            htrSwitchLabel[htr] = UiUtilities.newLabel(heaterNames[htr] + ":", SWITCH_LABEL_WIDTH);
            htrSwitchStatus[htr] = UiUtilities.newLabel("OFF", SWITCH_STATE_WIDTH);
            htrSwitchOffRB[htr] = uiUtils.newRadioButton("Off", "HF" + htr);
            htrSwitchOnRB[htr] = uiUtils.newRadioButton("On", "HT" + htr);
            htrSwitchBG[htr] = new ButtonGroup();
            htrSwitchBG[htr].add(htrSwitchOffRB[htr]);
            htrSwitchBG[htr].add(htrSwitchOnRB[htr]);
            powerTextField[htr] = uiUtils.newTextFieldX("10000.0", String.valueOf(htr), TextFieldX.TYPE_DOUBLE, true);
            powerUnitsLabel[htr] = UiUtilities.newLabel("W", 0);
        }

        // Layout the PLC panel
        plcPanel = new JPanel();
        GridBagConstraints gbp = new GridBagConstraints();
        gbp.anchor = GridBagConstraints.WEST;
        gbp.gridx = 0;
        gbp.gridy = 0;
        gbp.insets = new Insets(0, 4, 0, 4);
        plcPanel.add(UiUtilities.newLabel("PLC state:", 0), gbp);
        gbp.gridx++;
        plcPanel.add(plcStatus, gbp);

        // Layout the switch panel
        switchPanel = UiUtilities.newBorderedPanel("PLC Switches");
        GridBagConstraints gbs = new GridBagConstraints();
        gbs.anchor = GridBagConstraints.WEST;
        gbs.insets = new Insets(0, 0, 4, 4);
        gbs.gridy = -1;
        int numColm = 2;
        int colm = 0;
        for (int sw = 0; sw < PcpSwitches.NUM_SWITCHES; sw++) {
            if (colm == 0) {
                gbs.gridx = 0;
                gbs.gridy++;
            }
            gbs.insets.top = gbs.gridy == 0 ? 4 : 0;
            gbs.insets.left = colm == 0 ? 4 : 40;
            switchPanel.add(UiUtilities.newLabel(switchNames[sw] + ":", 0), gbs);
            gbs.insets.left = 4;
            gbs.gridx++;
            switchPanel.add(switchStatus[sw], gbs);
            gbs.gridx++;
            switchPanel.add(switchOffRB[sw], gbs);
            gbs.gridx++;
            switchPanel.add(switchOnRB[sw], gbs);
            gbs.gridx++;
            colm = (colm + 1) % numColm;
        }

        // Layout the latched conditions panel
        latchPanel = UiUtilities.newBorderedPanel("PLC Error Conditions");
        GridBagConstraints gbl = new GridBagConstraints();
        gbl.anchor = GridBagConstraints.WEST;
        gbl.insets = new Insets(0, 0, 4, 4);
        gbl.insets.bottom = 4;
        gbl.gridy = -1;
        numColm = 2;
        colm = 0;
        for (int cond = 0; cond < PcpLatches.NUM_LATCHES; cond++) {
            if (colm == 0) {
                gbl.gridx = 0;
                gbl.gridy++;
            }
            gbl.insets.top = gbl.gridy == 0 ? 4 : 0;
            gbl.insets.left = colm == 0 ? 4 : 40;
            latchPanel.add(latchLabel[cond], gbl);
            gbl.insets.left = 4;
            gbl.gridx++;
            latchPanel.add(latchStatus[cond], gbl);
            gbl.gridx++;
            latchPanel.add(latchResetBtn[cond], gbl);
            gbl.gridx++;
            colm = (colm + 1) % numColm;
        }

        // Layout the conditions panel
        conditionPanel = UiUtilities.newBorderedPanel("PLC Running Conditions");
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.anchor = GridBagConstraints.WEST;
        gbc.insets = new Insets(0, 0, 4, 4);
        gbc.gridy = -1;
        numColm = 2;
        colm = 0;
        for (int cond = 0; cond < PcpConditions.NUM_CONDITIONS; cond++) {
            if (colm == 0) {
                gbc.gridx = 0;
                gbc.gridy++;
            }
            gbc.insets.top = gbc.gridy == 0 ? 4 : 0;
            gbc.insets.left = colm == 0 ? 4 : 35;
            conditionPanel.add( UiUtilities.newLabel(conditionNames[cond] + ":", 0), gbc);
            gbc.insets.left = 4;
            gbc.gridx++;
            conditionPanel.add(conditionStatus[cond], gbc);
            gbc.gridx++;
            colm = (colm + 1) % numColm;
        }

        // Lay out the heater panel
        heaterPanel = UiUtilities.newBorderedPanel("Heaters");
        GridBagConstraints gbh = new GridBagConstraints();
        gbh.anchor = GridBagConstraints.WEST;
        gbh.insets = new Insets(4, 4, 0, 4);
        gbh.gridy = 0;
        for (int htr = 0; htr < PcpHeaters.NUM_HEATERS; htr++) {
            gbh.insets.bottom = (htr == PcpHeaters.NUM_HEATERS - 1) ? 4 : 0;
            gbh.gridx = 0;
            heaterPanel.add(htrSwitchLabel[htr], gbh);
            gbh.gridx++;
            heaterPanel.add(htrSwitchStatus[htr], gbh);
            gbh.gridx++;
            heaterPanel.add(htrSwitchOffRB[htr], gbh);
            gbh.gridx++;
            gbh.insets.right = 0;
            heaterPanel.add(htrSwitchOnRB[htr], gbh);
            gbh.gridx++;
            heaterPanel.add(powerTextField[htr], gbh);
            gbh.gridx++;
            gbh.insets.right = 4;
            heaterPanel.add(powerUnitsLabel[htr], gbh);
            gbh.gridy++;
            gbh.insets.top = 0;
        }

        // Lay out the complete panel
        setLayout(new GridBagLayout());
        GridBagConstraints gbm = new GridBagConstraints();
        gbm.anchor = GridBagConstraints.NORTH;
        gbm.insets = new Insets(6, 0, 6, 0);
        gbm.gridx = 0;
        gbm.gridy = 0;
        add(systemPanel, gbm);
        gbm.gridy++;
        add(plcPanel, gbm);
        gbm.gridy++;
        add(switchPanel, gbm);
        gbm.gridy++;
        add(latchPanel, gbm);
        gbm.gridy++;
        add(conditionPanel, gbm);
        gbm.gridy++;
        add(heaterPanel, gbm);

    }

    @Override
    public void handleRadioButton(String name) {
        char type = name.charAt(0);
        char action = name.charAt(1);
        int sw = Integer.valueOf(name.substring(2));
        if (type == 'P') {
            switchOffRB[sw].setEnabled(false);
            switchOnRB[sw].setEnabled(false);
            sender.sendCommand(null, "setNamedSwitchOn", PcpSwitches.getName(sw), action == 'T');
        }
        else {
            htrSwitchOffRB[sw].setEnabled(false);
            htrSwitchOnRB[sw].setEnabled(false);
            sender.sendCommand(null, "setHeaterState", PcpHeaters.getName(sw), action == 'T');
        }
    }

    @Override
    public void handleButton(String name) {
        int latch = Integer.valueOf(name);
        latchResetBtn[latch].setEnabled(false);
        sender.sendCommand(null, "clearLatch", PcpLatches.getName(latch));
    }

    @Override
    public void handleTextFieldX(String name, Object value) {
        int htr = Integer.valueOf(name);
        powerTextField[htr].setDisabled();
        sender.sendCommand(null, "setHeaterPower", PcpHeaters.getName(htr), (Double)value);
    }

    class UpdatePcpStatus implements Runnable {

        private final PcpSysState ps;

        UpdatePcpStatus(PcpSysState ps) {
            this.ps = ps;
        }

        @Override
        public void run() {
            systemPanel.updatePanel(ps.getTickMillis());
            PLCState plcState = ps.getPlcState();
            plcStatus.setText(plcState.name());
            plcStatus.setForeground(plcState == PLCState.OFFLINE ? UiConstants.BLUE :
                                    plcState == PLCState.DEAD ? UiConstants.RED : UiConstants.GREEN);
            plcStatus.setEnabled(true);
            for (int sw = 0; sw < PcpSwitches.NUM_SWITCHES; sw++) {
                PcpSwitchState state = ps.getSwitchState(sw);
                switchStatus[sw].setText(state.name());
                switchStatus[sw].setForeground(switchColors.get(state));
                JRadioButton selButton = state == PcpSwitchState.ON ? switchOnRB[sw] : switchOffRB[sw];
                selButton.setSelected(true);
                switchStatus[sw].setEnabled(true);
                switchOffRB[sw].setEnabled(true);
                switchOnRB[sw].setEnabled(true);
            }
            for (int cond = 0; cond < PcpLatches.NUM_LATCHES; cond++) {
                LatchState state = ps.getLatch(cond);
                latchStatus[cond].setText(state.name());
                latchStatus[cond].setForeground(state == LatchState.OFFLINE ? UiConstants.BLUE :
                                                state == LatchState.CLEAR ? UiConstants.GREEN :
                                                state == LatchState.ACTIVE ? UiConstants.RED : UiConstants.PURPLE);
                latchStatus[cond].setEnabled(true);
                latchResetBtn[cond].setEnabled(true);
            }
            for (int cond = 0; cond < PcpConditions.NUM_CONDITIONS; cond++) {
                ConditionState state = ps.getCondition(cond);
                conditionStatus[cond].setText(state.name());
                conditionStatus[cond].setForeground(state == ConditionState.OFF ? UiConstants.BLUE :
                                                    state == ConditionState.NO ? Color.BLACK : UiConstants.GREEN);
                conditionStatus[cond].setEnabled(true);
            }
            for (int htr = 0; htr < PcpHeaters.NUM_HEATERS; htr++) {
                PcpHeaterState pState = ps.getHeaterState(htr);
                htrSwitchStatus[htr].setEnabled(true);
                htrSwitchStatus[htr].setText(pState.name());
                htrSwitchStatus[htr].setForeground(stateColors.get(pState));
                JRadioButton selButton = PcpHeaterState.isOnState(pState) ? htrSwitchOnRB[htr] : htrSwitchOffRB[htr];
                selButton.setSelected(true);
                htrSwitchOffRB[htr].setEnabled(true);
                htrSwitchOnRB[htr].setEnabled(true);
                powerTextField[htr].update(ps.getHeaterPower(htr), true);
            }
            if (!gotTempLimits && ps.getLimit(0) != Integer.MAX_VALUE) {
                latchLabel[PcpLatches.LATCH_COLD_TEMP_HIGH].setText("Cold Temp > " + ps.getLimit(PcpLimits.LIMIT_COLD_HIGH) + "C:");
                latchLabel[PcpLatches.LATCH_COLD_TEMP_LOW].setText("Cold Temp < " + ps.getLimit(PcpLimits.LIMIT_COLD_LOW) + "C:");
                gotTempLimits = true;
            }
            repaint();
        }

    }

    class DisablePanel implements Runnable {

        @Override
        public void run() {
            systemPanel.disablePanel();
            plcStatus.setEnabled(false);
            for (int sw = 0; sw < PcpSwitches.NUM_SWITCHES; sw++) {
                switchStatus[sw].setEnabled(false);
                switchOffRB[sw].setEnabled(false);
                switchOnRB[sw].setEnabled(false);
            }
            for (int cond = 0; cond < PcpLatches.NUM_LATCHES; cond++) {
                latchStatus[cond].setEnabled(false);
                latchResetBtn[cond].setEnabled(false);
            }
            for (int cond = 0; cond < PcpConditions.NUM_CONDITIONS; cond++) {
                conditionStatus[cond].setEnabled(false);
            }
            for (int htr = 0; htr < PcpHeaters.NUM_HEATERS; htr++) {
                htrSwitchStatus[htr].setEnabled(false);
                htrSwitchOffRB[htr].setEnabled(false);
                htrSwitchOnRB[htr].setEnabled(false);
                powerTextField[htr].setDisabled();
            }
            repaint();
        } 
    }

    private static final long serialVersionUID = 1L;
}
