package org.lsst.ccs.subsystem.refrig.ui;

import java.awt.Color;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import javax.swing.ButtonGroup;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.SwingUtilities;
import org.lsst.ccs.subsystem.common.ui.UiConstants;
import org.lsst.ccs.subsystem.common.ui.UiUtilities;
import org.lsst.ccs.subsystem.common.ui.jas.CommandSender;
import org.lsst.ccs.subsystem.refrig.constants.SwitchState;
import org.lsst.ccs.subsystem.refrig.data.ChillerPlcTestState;

/**
 *  Implements the chiller PLC test control panel.
 *
 *  @author Owen Saxton
 */
public class PlcTestControlPanel extends JPanel implements CommandSender.ReplyHandler, UiUtilities.ActionHandler {

    private static final String CMND_GET_STATE = "getControlState";
    private static final int NUM_SWITCHES = 10;
    private final CommandSender sender;
    private final UiUtilities uiUtilities = new UiUtilities(this);
    private JPanel switchPanel;
    private final JLabel swStatus[] = new JLabel[NUM_SWITCHES];
    private final JRadioButton swOnButton[] = new JRadioButton[NUM_SWITCHES];
    private final JRadioButton swOffButton[] = new JRadioButton[NUM_SWITCHES];
    private final ButtonGroup swButtonGroup[] = new ButtonGroup[NUM_SWITCHES];
    private ChillerPlcControlPanel plcControlPanel;

    public PlcTestControlPanel(String agent) {
        sender = new CommandSender(agent, this);
        initComponents();
        (new DisablePanel()).run();
    }

    public void initPanel() {
        sender.sendCommand(true, null, CMND_GET_STATE);
    }

    @Override
    public void onCommandReply(Object reply, String path, String command, Object[] args) {
        ChillerPlcTestState ps = (ChillerPlcTestState)reply;
        updatePanel(ps);
    }

    @Override
    public void onCommandReject(String path, String command, Object[] args) {
        if (!command.equals(CMND_GET_STATE)) {
            sender.sendCommand(true, null, CMND_GET_STATE);
        }
    }

     @Override
    public void onCommandError(String path, String command, Object[] args) {
        if (!command.equals(CMND_GET_STATE)) {
            sender.sendCommand(true, null, CMND_GET_STATE);
        }
    }

    public void updatePanel(ChillerPlcTestState ps) {
        SwingUtilities.invokeLater(new UpdateRefrigState(ps));
    }

    public void disablePanel() {
        SwingUtilities.invokeLater(new DisablePanel());
    }

    private void initComponents() {

        // Test switches control panel
        for (int sw = 0; sw < NUM_SWITCHES; sw++) {
            swStatus[sw] = UiUtilities.newLabel("XXX", UiUtilities.maxEnumLabelWidth(SwitchState.class));
            swOnButton[sw] = uiUtilities.newRadioButton("On", "T" + sw);
            swOffButton[sw] = uiUtilities.newRadioButton("Off", "F" + sw);
            swButtonGroup[sw] = new ButtonGroup();
            swButtonGroup[sw].add(swOnButton[sw]);
            swButtonGroup[sw].add(swOffButton[sw]);
        }
        switchPanel = UiUtilities.newBorderedPanel("Maq20 Switches");
        GridBagConstraints gbs = new GridBagConstraints();
        gbs.anchor = GridBagConstraints.WEST;
        gbs.insets = new Insets(0, 0, 4, 4);
        gbs.gridy = -1;
        int numColm = 2;
        int colm = 0;
        for (int sw = 0; sw < NUM_SWITCHES; sw++) {
            if (colm == 0) {
                gbs.gridx = 0;
                gbs.gridy++;
            }
            gbs.insets.top = gbs.gridy == 0 ? 4 : 0;
            gbs.insets.left = colm == 0 ? 4 : 40;
            switchPanel.add(UiUtilities.newLabel("Maq20 line " + sw + ":", 0), gbs);
            gbs.insets.left = 4;
            gbs.gridx++;
            switchPanel.add(swStatus[sw], gbs);
            gbs.gridx++;
            switchPanel.add(swOffButton[sw], gbs);
            gbs.gridx++;
            switchPanel.add(swOnButton[sw], gbs);
            gbs.gridx++;
            colm = (colm + 1) % numColm;
        }
        
        // PLC control panel
        plcControlPanel = new ChillerPlcControlPanel(sender);

        // Lay out the initial complete panel
        setLayout(new GridBagLayout());
        GridBagConstraints gbm = new GridBagConstraints();
        gbm.insets = new Insets(4, 0, 4, 0);
        gbm.anchor = GridBagConstraints.NORTH;
        gbm.gridx = 0;
        gbm.gridy = 0;
        add(switchPanel, gbm);
        gbm.gridy++;
        add(plcControlPanel, gbm);
    }

    @Override
    public void handleRadioButton(String name) {
        char action = name.charAt(0);
        int sw = Integer.valueOf(name.substring(1));
        swOffButton[sw].setEnabled(false);
        swOnButton[sw].setEnabled(false);
        sender.sendCommand(null, "setPlcLatch", sw, action == 'T');
    }

    class UpdateRefrigState implements Runnable {

        private final ChillerPlcTestState ps;

        UpdateRefrigState(ChillerPlcTestState ps) {
            this.ps = ps;
        }

        @Override
        public void run() {
            for (int sw = 0; sw < NUM_SWITCHES; sw++) {
                SwitchState state = ps.getSwitchState(sw);
                swStatus[sw].setText(state.name());
                Color color = state == SwitchState.OFF ? Color.black : state == SwitchState.ON ? UiConstants.GREEN : UiConstants.BLUE;
                swStatus[sw].setForeground(color);
                JRadioButton selButton = state == SwitchState.ON ? swOnButton[sw] : swOffButton[sw];
                selButton.setSelected(true);
                swStatus[sw].setEnabled(true);
                swOffButton[sw].setEnabled(true);
                swOnButton[sw].setEnabled(true);
            }
            plcControlPanel.updatePanel(ps.getPlcState());
            repaint();
        }

    }

    class DisablePanel implements Runnable {

        @Override
        public void run() {
            for (int sw = 0; sw < NUM_SWITCHES; sw++) {
                swStatus[sw].setEnabled(false);
                swOffButton[sw].setEnabled(false);
                swOnButton[sw].setEnabled(false);
            }
            plcControlPanel.disablePanel();
            repaint();
        } 
    }

    private static final long serialVersionUID = 1L;
}
