package org.lsst.ccs.subsystem.refrig.ui;

import java.awt.Color;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.util.LinkedHashMap;
import java.util.Map;
import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.SwingUtilities;
import org.lsst.ccs.subsystem.common.ui.jas.CommandSender;
import org.lsst.ccs.subsystem.common.ui.UiConstants;
import org.lsst.ccs.subsystem.common.ui.UiUtilities;
import org.lsst.ccs.subsystem.refrig.constants.ChillerConditions;
import org.lsst.ccs.subsystem.refrig.constants.ChillerLatches;
import org.lsst.ccs.subsystem.refrig.constants.ChillerPLCState;
import org.lsst.ccs.subsystem.refrig.constants.ChillerSwitches;
import org.lsst.ccs.subsystem.refrig.constants.ConditionState;
import org.lsst.ccs.subsystem.refrig.constants.LatchState;
import org.lsst.ccs.subsystem.refrig.constants.SwitchState;
import org.lsst.ccs.subsystem.refrig.data.ChillerPlcState;

/**
 *  Implements the chiller PLC control panel.
 *
 *  @author Owen Saxton
 */
public class ChillerPlcControlPanel extends JPanel implements UiUtilities.ActionHandler {

    private static final Map<Integer, String> latches = new LinkedHashMap<>();
    static {
        latches.put(ChillerLatches.LATCH_PERMIT, "External Permit");
        latches.put(ChillerLatches.LATCH_EXT_EMO, "Chiller EMO Button");
        latches.put(ChillerLatches.LATCH_SMOKE_DETC, "Smoke Detector");
        latches.put(ChillerLatches.LATCH_LEAK_DETC, "Leak Detector");
        latches.put(ChillerLatches.LATCH_LEAK_FAULT, "Leak Detc Fault");
        latches.put(ChillerLatches.LATCH_BD_SUPPLY_P, "Burst Disc Supp > 5");
        latches.put(ChillerLatches.LATCH_BD_RETURN_P, "Burst Disc Retn > 5");
    }
    private static final Map<Integer, String> conditions = new LinkedHashMap<>();
    static {
        conditions.put(ChillerConditions.COND_CLP_REF_PERM, "Enable switch on");
        conditions.put(ChillerConditions.COND_KEY_SWITCH, "Keyswitch on");
        conditions.put(ChillerConditions.COND_CHILLER_EMO, "Chiller enabled");
        conditions.put(ChillerConditions.COND_MASTER_RESET, "Master reset on");
        conditions.put(ChillerConditions.COND_BD_SUPP_NO_WRN, "Burst Disc Supp < 4.5");
        conditions.put(ChillerConditions.COND_BD_RETN_NO_WRN, "Burst Disc Retn < 4.5");
    }

    private JPanel plcPanel;
    private JLabel plcStatus;
    private JLabel switchStatus;
    private ButtonGroup switchBG;
    private JRadioButton switchOffRB;
    private JRadioButton switchOnRB;

    private JPanel latchPanel;
    private final JLabel[] latchLabel = new JLabel[ChillerLatches.NUM_LATCHES];
    private final JLabel[] latchStatus = new JLabel[ChillerLatches.NUM_LATCHES];
    private final JButton[] latchResetBtn = new JButton[ChillerLatches.NUM_LATCHES];

    private JPanel conditionPanel;
    private final JLabel[] conditionStatus = new JLabel[ChillerConditions.NUM_CONDITIONS];

    private final CommandSender sender;
    private final UiUtilities uiUtils;

    public ChillerPlcControlPanel(CommandSender sender) {
        this.sender = sender;
        uiUtils = new UiUtilities(this);
        initComponents();
        (new DisablePanel()).run();
    }

    public void updatePanel(ChillerPlcState ps) {
        SwingUtilities.invokeLater(new UpdatePlcStatus(ps));
    }

    public void disablePanel() {
        SwingUtilities.invokeLater(new DisablePanel());
    }

    private void initComponents() {

        // PLC state
        plcStatus = UiUtilities.newLabel("XXX", UiUtilities.maxEnumLabelWidth(ChillerPLCState.class));

        // Switch state
        switchStatus = UiUtilities.newLabel("XXX", UiUtilities.maxEnumLabelWidth(SwitchState.class));
        switchOffRB = uiUtils.newRadioButton("Off", "F");
        switchOnRB = uiUtils.newRadioButton("On", "T");
        ButtonGroup bg = switchBG = new ButtonGroup();
        bg.add(switchOffRB);
        bg.add(switchOnRB);

        // Latched condition states
        for (int cond : latches.keySet()) {
            latchLabel[cond] = UiUtilities.newLabel(latches.get(cond) + ":", 0);
            latchStatus[cond] = UiUtilities.newLabel("XXX", UiUtilities.maxEnumLabelWidth(LatchState.class));
            latchResetBtn[cond] = uiUtils.newButton("Reset", String.valueOf(cond), true);
        }

        // Condition states
        for (int cond : conditions.keySet()) {
            conditionStatus[cond] = UiUtilities.newLabel("XXX", UiUtilities.maxEnumLabelWidth(ConditionState.class));
        }

        // Layout the PLC panel
        plcPanel = UiUtilities.newBorderedPanel("PLC State");
        GridBagConstraints gbp = new GridBagConstraints();
        gbp.anchor = GridBagConstraints.WEST;
        gbp.gridx = 0;
        gbp.gridy = 0;
        gbp.insets = new Insets(4, 4, 4, 4);
        plcPanel.add(UiUtilities.newLabel("PLC state:", 0), gbp);
        gbp.gridx++;
        gbp.insets.left = 4;
        plcPanel.add(plcStatus, gbp);
        gbp.gridx++;
        gbp.insets.left = 35;
        plcPanel.add(UiUtilities.newLabel("Enable chiller:", 0), gbp);
        gbp.insets.left = 4;
        gbp.gridx++;
        plcPanel.add(switchStatus, gbp);
        gbp.gridx++;
        plcPanel.add(switchOffRB, gbp);
        gbp.gridx++;
        plcPanel.add(switchOnRB, gbp);

        // Layout the latched conditions panel
        latchPanel = UiUtilities.newBorderedPanel("PLC Error Conditions");
        GridBagConstraints gbl = new GridBagConstraints();
        gbl.anchor = GridBagConstraints.WEST;
        gbl.insets = new Insets(0, 0, 4, 4);
        gbl.insets.bottom = 4;
        gbl.gridy = -1;
        int numColm = 1;
        int colm = 0;
        for (int cond : latches.keySet()) {
            if (colm == 0) {
                gbl.gridx = 0;
                gbl.gridy++;
            }
            gbl.insets.top = gbl.gridy == 0 ? 4 : 0;
            gbl.insets.left = colm == 0 ? 4 : 40;
            latchPanel.add(latchLabel[cond], gbl);
            gbl.insets.left = 4;
            gbl.gridx++;
            latchPanel.add(latchStatus[cond], gbl);
            gbl.gridx++;
            latchPanel.add(latchResetBtn[cond], gbl);
            gbl.gridx++;
            colm = (colm + 1) % numColm;
        }

        // Layout the conditions panel
        conditionPanel = UiUtilities.newBorderedPanel("PLC Running Conditions");
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.anchor = GridBagConstraints.WEST;
        gbc.insets = new Insets(0, 0, 4, 4);
        gbc.gridy = -1;
        numColm = 1;
        colm = 0;
        for (int cond : conditions.keySet()) {
            if (colm == 0) {
                gbc.gridx = 0;
                gbc.gridy++;
            }
            gbc.insets.top = gbc.gridy == 0 ? 4 : 0;
            gbc.insets.left = colm == 0 ? 4 : 35;
            conditionPanel.add(UiUtilities.newLabel(conditions.get(cond) + ":", 0), gbc);
            gbc.insets.left = 4;
            gbc.gridx++;
            conditionPanel.add(conditionStatus[cond], gbc);
            gbc.gridx++;
            colm = (colm + 1) % numColm;
        }

        // Lay out the complete panel
        setLayout(new GridBagLayout());
        GridBagConstraints gbm = new GridBagConstraints();
        gbm.anchor = GridBagConstraints.NORTH;
        gbm.insets = new Insets(6, 4, 6, 4);
        gbm.gridwidth = 2;
        gbm.gridx = 0;
        gbm.gridy = 0;
        add(plcPanel, gbm);
        gbm.gridwidth = 1;
        gbm.gridy++;
        add(latchPanel, gbm);
        gbm.gridx++;
        add(conditionPanel, gbm);

    }

    @Override
    public void handleRadioButton(String name) {
        char action = name.charAt(0);
        switchOffRB.setEnabled(false);
        switchOnRB.setEnabled(false);
        sender.sendCommand(null, "enableChiller", action == 'T');
    }

    @Override
    public void handleButton(String name) {
        int latch = Integer.valueOf(name);
        latchResetBtn[latch].setEnabled(false);
        sender.sendCommand(null, "clearPlcLatch", ChillerLatches.getName(latch));
    }

    class UpdatePlcStatus implements Runnable {

        private final ChillerPlcState ps;

        UpdatePlcStatus(ChillerPlcState ps) {
            this.ps = ps;
        }

        @Override
        public void run() {
            ChillerPLCState plcState = ps.getPlcState();
            plcStatus.setText(plcState.name());
            plcStatus.setForeground(plcState == ChillerPLCState.OFFLINE ? UiConstants.BLUE :
                                    plcState == ChillerPLCState.PLC_DEAD ? UiConstants.RED :
                                    plcState == ChillerPLCState.RUNNING ? UiConstants.GREEN : Color.BLACK);
            plcStatus.setEnabled(true);
            SwitchState swState = ps.getSwitchState(ChillerSwitches.SW_ENABLE_CHILLER);
            switchStatus.setText(swState.name());
            Color color = swState == SwitchState.OFF ? Color.black : swState == SwitchState.ON ? UiConstants.GREEN : UiConstants.BLUE;
            switchStatus.setForeground(color);

            JRadioButton selButton = swState == SwitchState.ON ? switchOnRB : switchOffRB;
            selButton.setSelected(true);
            switchStatus.setEnabled(true);
            switchOffRB.setEnabled(true);
            switchOnRB.setEnabled(true);
            for (int cond : latches.keySet()) {
                LatchState state = ps.getLatch(cond);
                latchStatus[cond].setText(state.name());
                latchStatus[cond].setForeground(state == LatchState.OFFLINE ? UiConstants.BLUE :
                                                state == LatchState.CLEAR ? UiConstants.GREEN :
                                                state == LatchState.ACTIVE ? UiConstants.RED : UiConstants.PURPLE);
                latchStatus[cond].setEnabled(true);
                latchResetBtn[cond].setEnabled(true);
            }
            for (int cond : conditions.keySet()) {
                ConditionState state = ps.getCondition(cond);
                conditionStatus[cond].setText(state.name());
                conditionStatus[cond].setForeground(state == ConditionState.OFF ? UiConstants.BLUE :
                                                    state == ConditionState.NO ? Color.BLACK : UiConstants.GREEN);
                conditionStatus[cond].setEnabled(true);
            }
            repaint();
        }

    }

    class DisablePanel implements Runnable {

        @Override
        public void run() {
            plcStatus.setEnabled(false);
            switchStatus.setEnabled(false);
            switchOffRB.setEnabled(false);
            switchOnRB.setEnabled(false);
            for (int cond : latches.keySet()) {
                latchStatus[cond].setEnabled(false);
                latchResetBtn[cond].setEnabled(false);
            }
            for (int cond : conditions.keySet()) {
                conditionStatus[cond].setEnabled(false);
            }
            repaint();
        } 
    }

    private static final long serialVersionUID = 1L;
}
