package org.lsst.ccs.subsystem.refrig.ui;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dialog;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.Window;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.SwingUtilities;
import org.lsst.ccs.subsystem.common.ui.UiConstants;
import org.lsst.ccs.subsystem.common.ui.UiUtilities;
import org.lsst.ccs.subsystem.common.ui.jas.CommandSender;
import org.lsst.ccs.subsystem.refrig.constants.CompSwitches;
import org.lsst.ccs.subsystem.refrig.constants.CompressorState;
import org.lsst.ccs.subsystem.refrig.constants.SwitchState;
import org.lsst.ccs.subsystem.refrig.data.CompState;

/**
 *  Implements the compressor summary line sub-panel.
 *
 *  @author Owen Saxton
 */
public class CompressorSummaryLine extends JPanel implements UiUtilities.ActionHandler {

    private static final int STATE_WIDTH = UiUtilities.maxEnumLabelWidth(CompressorState.class);
    private static final int HORZ_INSET = UiUtilities.getLabelSize("M").width / 2;

    private String comp;
    private final UiUtilities uiUtils;
    private final CommandSender sender;
    private final CompressorControlPanel control;
    private JDialog dialog = null;
    private JLabel nameLabel, nameValue, stateLabel, stateValue, switchLabel;
    private ButtonGroup switchBG;
    private JRadioButton switchOffRB, switchOnRB;
    private JButton controlButton;

    public CompressorSummaryLine(int type, CommandSender sender) {
        this.sender = sender;
        uiUtils = new UiUtilities(this);
        initComponents();
        control = new CompressorControlPanel(type, sender);
        disableSystem();
    }

    public void updateControlPanel(CompState rs) {
        comp = rs.getName();
        nameValue.setText(comp);
        CompressorState cs = rs.getCompressorState();
        Color cColor = cs == CompressorState.OFFLINE ? UiConstants.BLUE :
                       cs == CompressorState.STOPPED ? Color.BLACK :
                       cs == CompressorState.RUNNING ? UiConstants.GREEN :
                       cs == CompressorState.WAITING ? UiConstants.YELLOW : 
                       cs == CompressorState.STOP_WRN || cs == CompressorState.WAIT_WRN || cs == CompressorState.RUN_WRN ? UiConstants.PURPLE : UiConstants.RED;
        stateValue.setText(cs.name());
        stateValue.setForeground(cColor);
        SwitchState state = rs.getSwitchState(CompSwitches.SW_ENABLE);
        JRadioButton selButton = state == SwitchState.ON ? switchOnRB : switchOffRB;
        selButton.setSelected(true);
        nameValue.setEnabled(true);
        stateValue.setEnabled(true);
        switchOffRB.setEnabled(true);
        switchOnRB.setEnabled(true);
        //controlButton.setEnabled(true);
        control.updateControlPanel(rs);
        repaint();
    }

    public final void disableSystem() {
        nameValue.setEnabled(false);
        stateValue.setEnabled(false);
        switchOffRB.setEnabled(false);
        switchOnRB.setEnabled(false);
        //controlButton.setEnabled(false);
        control.disableSystem();
        repaint();
    }

    private void initComponents() {

        nameLabel = UiUtilities.newLabel("Comp:  ", 0);
        nameValue = UiUtilities.newLabel("Cryo6", 0);

        stateLabel = UiUtilities.newLabel("State:  ", 0);
        stateValue = UiUtilities.newLabel("X", STATE_WIDTH);

        switchLabel = UiUtilities.newLabel("Enable:  ", 0);
        switchOffRB = uiUtils.newRadioButton("Off", "F");
        switchOnRB = uiUtils.newRadioButton("On", "T");
        switchBG = new ButtonGroup();
        switchBG.add(switchOffRB);
        switchBG.add(switchOnRB);

        controlButton = uiUtils.newButton("Full control...", "", true);

        // Lay out this panel
        setLayout(new GridBagLayout());
        GridBagConstraints gbm = new GridBagConstraints();
        gbm.insets = new Insets(0, HORZ_INSET, 0, 0);
        gbm.anchor = GridBagConstraints.WEST;
        gbm.gridx = 0;
        gbm.gridy = 0;
        add(nameLabel, gbm);
        gbm.gridx++;
        gbm.insets.left = 0;
        add(nameValue, gbm);
        gbm.gridx++;
        gbm.insets.left = 25;
        add(stateLabel, gbm);
        gbm.gridx++;
        gbm.insets.left = 0;
        add(stateValue, gbm);
        gbm.gridx++;
        gbm.insets.left = 25;
        add(switchLabel, gbm);
        gbm.gridx++;
        gbm.insets.left = 0;
        add(switchOffRB, gbm);
        gbm.gridx++;
        gbm.insets.left = HORZ_INSET;
        add(switchOnRB, gbm);
        gbm.gridx++;
        gbm.insets.left = 25;
        gbm.insets.right = HORZ_INSET;
        add(controlButton, gbm);
    }

    @Override
    public void handleRadioButton(String name) {
        sender.sendCommand(comp, "setSwitchOn", CompSwitches.ID_MAP.get(CompSwitches.SW_ENABLE), name.charAt(0) == 'T');
    }

    @Override
    public void handleButton(String name) {
        if (dialog != null) {
            dialog.setVisible(false);
            dialog.dispose();
            dialog = null;
            return;
        }
        Window window = SwingUtilities.getWindowAncestor(this);
        dialog = new JDialog(window, comp + " Control", Dialog.ModalityType.MODELESS);
        dialog.add(control, BorderLayout.CENTER);
        dialog.pack();
        dialog.setLocationRelativeTo(window);
        dialog.addWindowListener(new WindowAdapter() {
            @Override
            public void windowClosing(WindowEvent e) {
                dialog.setVisible(false);
                dialog.dispose();
                dialog = null;
            }
        });
        dialog.setVisible(true);
    }

    private static final long serialVersionUID = 1L;
}
