package org.lsst.ccs.subsystem.refrig.ui;

import java.awt.Color;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.util.HashMap;
import java.util.Map;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;
import org.lsst.ccs.subsystem.common.ui.SystemStatusPanel;
import org.lsst.ccs.subsystem.common.ui.TextFieldX;
import org.lsst.ccs.subsystem.common.ui.UiConstants;
import org.lsst.ccs.subsystem.common.ui.UiUtilities;
import org.lsst.ccs.subsystem.common.ui.jas.CommandSender;
import org.lsst.ccs.subsystem.refrig.constants.ChillerState;
import org.lsst.ccs.subsystem.refrig.constants.RefrigConstants;
import org.lsst.ccs.subsystem.refrig.data.ChillerControlState;

/**
 *  Implements the chiller control panel.
 *
 *  @author Owen Saxton
 */
public class ChillerControlPanelOld extends JPanel implements CommandSender.ReplyHandler, UiUtilities.ActionHandler  {

    private static final String CMND_GET_STATE = "getControlState";
    private static final Map<ChillerState, Color> stateColors = new HashMap<>();
    static {
        stateColors.put(ChillerState.OFFLINE, UiConstants.BLUE);
        stateColors.put(ChillerState.SETPOINT, UiConstants.GREEN);
        stateColors.put(ChillerState.CONTROLLING, UiConstants.YELLOW);
    }

    private final CommandSender sender;
    private final UiUtilities uiUtils;

    private SystemStatusPanel statusPanel;
    private JPanel mainPanel;
    private JPanel chillerStatePnl;
    private JLabel chillerStateLbl;
    private TextFieldX flowRateTf;
    private JButton setFlowRateBtn;
    private TextFieldX tankPressTf;
    private JButton setTankPressBtn;
    private TextFieldX tempSetPointTf;
    private JLabel defaultRampLbl;
    private JButton setTempBtn;
    private TextFieldX tempWithRampTf;
    private TextFieldX rampRateTf;
    private JButton setTempWithRampBtn;
    private JButton quitControlBtn;
    private double flowRate, tankPress, tempSetPoint, rampRate;

    public ChillerControlPanelOld(String agent) {
        sender = new CommandSender(agent, this);
        uiUtils = new UiUtilities(this);
        initComponents();
        (new DisablePanel()).run();
    }

    public void initPanel() {
        sender.sendCommand(true, null, CMND_GET_STATE);
    }

    @Override
    public void onCommandReply(Object reply, String path, String command, Object[] args) {
        updatePanel((ChillerControlState)reply);
    }

    @Override
    public void onCommandReject(String path, String command, Object[] args) {
        if (!command.equals(CMND_GET_STATE)) {
            sender.sendCommand(true, null, CMND_GET_STATE);
        }
    }

    @Override
    public void onCommandError(String path, String command, Object[] args) {
        if (!command.equals(CMND_GET_STATE)) {
            sender.sendCommand(true, null, CMND_GET_STATE);
        }
    }

    public void updatePanel(ChillerControlState cs) {
        SwingUtilities.invokeLater(new UpdateChillerState(cs));
    }

    public void disablePanel() {
        SwingUtilities.invokeLater(new DisablePanel());
    }

    private void initComponents() {

        // Individual graphic elements
        chillerStateLbl = UiUtilities.newLabel("XXXXX", UiUtilities.maxEnumLabelWidth(ChillerState.class));
        flowRateTf = uiUtils.newTextFieldX("999999.9", "F", TextFieldX.TYPE_DOUBLE);
        flowRateTf.update(flowRate, true);
        setFlowRateBtn = uiUtils.newButton("Set", "F", false);
        tankPressTf = uiUtils.newTextFieldX("999999.9", "P", TextFieldX.TYPE_DOUBLE);
        tankPressTf.update(tankPress, true);
        setTankPressBtn = uiUtils.newButton("Set", "P", false);
        tempSetPointTf = uiUtils.newTextFieldX("999999.9", "T", TextFieldX.TYPE_DOUBLE);
        tempSetPointTf.update(tempSetPoint, true);
        defaultRampLbl = UiUtilities.newLabel("99999.9", 0);
        defaultRampLbl.setHorizontalAlignment(SwingConstants.CENTER);
        defaultRampLbl.setText("0.0");
        setTempBtn = uiUtils.newButton("Set", "T", false);
        tempWithRampTf = uiUtils.newTextFieldX("999999.9", "U", TextFieldX.TYPE_DOUBLE);
        tempWithRampTf.update(tempSetPoint, true);
        rampRateTf = uiUtils.newTextFieldX("99999.9", "R", TextFieldX.TYPE_DOUBLE);
        rampRateTf.update(rampRate, true);
        setTempWithRampBtn = uiUtils.newButton("Set", "R", false);
        quitControlBtn = uiUtils.newButton("Quit", "Q", false);

        // System status line panel
        statusPanel = new SystemStatusPanel(sender, RefrigConstants.MONITOR_CONTROL, true);

        // Chiller status line panel
        chillerStatePnl = new JPanel();
        chillerStatePnl.setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.anchor = GridBagConstraints.WEST;
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.insets = new Insets(0, 4, 0, 4);
        chillerStatePnl.add(UiUtilities.newLabel("Chiller state:", 0), gbc);
        gbc.gridx++;
        chillerStatePnl.add(chillerStateLbl, gbc);

        // Main panel
        mainPanel = new JPanel();
        mainPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbm = new GridBagConstraints();
        gbm.anchor = GridBagConstraints.WEST;
        gbm.insets = new Insets(0, 0, 4, 5);

        gbm.gridx = 0;
        gbm.gridy = 0;
        mainPanel.add(UiUtilities.newLabel("Flow rate:", 0), gbm);
        gbm.gridx++;
        mainPanel.add(flowRateTf, gbm);
        gbm.gridx++;
        gbm.gridwidth = 2;
        mainPanel.add(UiUtilities.newLabel("gal/min", 0), gbm);
        gbm.gridx += 4;
        gbm.gridwidth = 1;
        mainPanel.add(setFlowRateBtn, gbm);

        gbm.gridx = 0;
        gbm.gridy++;
        mainPanel.add(UiUtilities.newLabel("Tank press:", 0), gbm);
        gbm.gridx++;
        mainPanel.add(tankPressTf, gbm);
        gbm.gridx++;
        gbm.gridwidth = 2;
        mainPanel.add(UiUtilities.newLabel("psig", 0), gbm);
        gbm.gridx += 4;
        gbm.gridwidth = 1;
        mainPanel.add(setTankPressBtn, gbm);

        gbm.gridx = 0;
        gbm.gridy++;
        mainPanel.add(UiUtilities.newLabel("Temperature:", 0), gbm);
        gbm.gridx++;
        mainPanel.add(tempSetPointTf, gbm);
        gbm.gridx++;
        gbm.insets.right = 20;
        mainPanel.add(UiUtilities.newLabel("C", 0), gbm);
        gbm.gridx ++;
        gbm.insets.right = 5;
        mainPanel.add(UiUtilities.newLabel("Default ramp:", 0), gbm);
        gbm.gridx++;
        mainPanel.add(defaultRampLbl, gbm);
        gbm.gridx ++;
        gbm.insets.right = 20;
        mainPanel.add(UiUtilities.newLabel("C/min", 0), gbm);
        gbm.gridx ++;
        gbm.insets.right = 5;
        mainPanel.add(setTempBtn, gbm);

        gbm.gridx = 0;
        gbm.gridy++;
        mainPanel.add(UiUtilities.newLabel("Temp with ramp:", 0), gbm);
        gbm.gridx++;
        mainPanel.add(tempWithRampTf, gbm);
        gbm.gridx ++;
        gbm.insets.right = 20;
        mainPanel.add(UiUtilities.newLabel("C", 0), gbm);
        gbm.gridx ++;
        gbm.insets.right = 5;
        mainPanel.add(UiUtilities.newLabel("Ramp:", 0), gbm);
        gbm.gridx++;
        mainPanel.add(rampRateTf, gbm);
        gbm.gridx ++;
        gbm.insets.right = 20;
        mainPanel.add(UiUtilities.newLabel("C/min", 0), gbm);
        gbm.gridx ++;
        gbm.insets.right = 5;
        mainPanel.add(setTempWithRampBtn, gbm);

        gbm.gridx = 0;
        gbm.gridy++;
        mainPanel.add(UiUtilities.newLabel("Quit controlling:", 0), gbm);
        gbm.gridx += 6;
        mainPanel.add(quitControlBtn, gbm);

        // Lay out the complete panel
        setLayout(new GridBagLayout());
        GridBagConstraints gbp = new GridBagConstraints();
        gbp.insets = new Insets(4, 4, 0, 0);
        gbp.anchor = GridBagConstraints.NORTH;
        gbp.gridx = 0;
        gbp.gridy = 0;
        add(statusPanel, gbp);
        gbp.gridy++;
        gbp.insets.bottom = 8;
        add(chillerStatePnl, gbp);
        gbp.gridy++;
        gbp.insets.bottom = 4;
        add(mainPanel, gbp);
    }

    @Override
    public void handleTextFieldX(String name, Object value) {
        switch (name) {
        case "F":
            flowRate = (double)value;
            flowRateTf.update(value, true);
            break;
        case "P":
            tankPress = (double)value;
            tankPressTf.update(value, true);
            break;
        case "T":
            tempSetPoint = (double)value;
            tempSetPointTf.update(value, true);
            break;
        case "U":
            tempSetPoint = (double)value;
            tempWithRampTf.update(value, true);
            break;
        case "R":
            rampRate = (double)value;
            rampRateTf.update(value, true);
            break;
        }
    }

    @Override
    public void handleButton(String name) {
        switch (name) {
        case "F":
            sender.sendCommand(null, "setFlow", flowRate);
            break;
        case "P":
            sender.sendCommand(null, "setTankPressure", tankPress);
            break;
        case "T":
            sender.sendCommand(null, "setTemperature", tempSetPoint);
            break;
        case "R":
            sender.sendCommand(null, "setTemperatureWithRamp", tempSetPoint, rampRate);
            break;
        case "Q":
            sender.sendCommand(null, "quitControllingTemperature");
            break;
        }
    }

    class UpdateChillerState implements Runnable {

        private final ChillerControlState cs;

        UpdateChillerState(ChillerControlState cs) {
            this.cs = cs;
        }

        @Override
        public void run() {
            statusPanel.updatePanel(cs.getFastPeriod());
            chillerStateLbl.setText(cs.getChillerState().name());
            chillerStateLbl.setForeground(stateColors.getOrDefault(cs.getChillerState(), Color.BLACK));
            if (!Double.isNaN(cs.getFlowSet())) {
                flowRate = cs.getFlowSet();
            }
            flowRateTf.update(flowRate, true);
            setFlowRateBtn.setEnabled(true);
            if (!Double.isNaN(cs.getSetPoint())) {
                tempSetPoint = cs.getSetPoint();
            }
            tempSetPointTf.update(tempSetPoint, true);
            if (!Double.isNaN(cs.getDefaultRamp())) {
                defaultRampLbl.setText(String.valueOf(cs.getDefaultRamp()));
            }
            setTempBtn.setEnabled(true);
            tempWithRampTf.update(tempSetPoint, true);
            rampRateTf. update(rampRate, true);
            setTempWithRampBtn.setEnabled(true);
            quitControlBtn.setEnabled(true);
            repaint();
        }

    }

    class DisablePanel implements Runnable {

        @Override
        public void run() {
            statusPanel.disablePanel();
            flowRateTf.setDisabled();
            setFlowRateBtn.setEnabled(false);
            tempSetPointTf.setDisabled();
            setTempBtn.setEnabled(false);
            tempWithRampTf.setDisabled();
            rampRateTf.setDisabled();
            setTempWithRampBtn.setEnabled(false);
            quitControlBtn.setEnabled(false);
            repaint();
        } 
    }

    private static final long serialVersionUID = 1L;
}
