package org.lsst.ccs.subsystem.refrig.ui;

import java.awt.Color;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import javax.swing.ButtonGroup;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.SwingUtilities;
import org.lsst.ccs.subsystem.common.ui.MonitorTaskPanel;
import org.lsst.ccs.subsystem.common.ui.SystemStatusPanel;
import org.lsst.ccs.subsystem.common.ui.UiConstants;
import org.lsst.ccs.subsystem.common.ui.UiUtilities;
import org.lsst.ccs.subsystem.common.ui.jas.CommandSender;
import org.lsst.ccs.subsystem.refrig.constants.CompTypes;
import org.lsst.ccs.subsystem.refrig.constants.RefrigConstants;
import org.lsst.ccs.subsystem.refrig.constants.SwitchState;
import org.lsst.ccs.subsystem.refrig.data.CompState;
import org.lsst.ccs.subsystem.refrig.data.RefrigState;

/**
 *  Implements the refrigeration control panel.
 *
 *  @author Owen Saxton
 */
public class RefrigControlPanel extends JPanel implements CommandSender.ReplyHandler, UiUtilities.ActionHandler {

    private static final String SOLID_BAR = "\u2588";
    private static final String CMND_GET_STATE = "getSystemState";
    private final CommandSender sender;
    private boolean cmndError = false;
    private final JPanel compressorPanel = UiUtilities.newBorderedPanel("Compressor Control");
    private final JPanel lightsPanel = UiUtilities.newBorderedPanel("Cabinet Lights Control");
    private final JRadioButton lightsOnRB[] = new JRadioButton[RefrigState.NUM_CRYO_CABINETS];
    private final JRadioButton lightsOffRB[] = new JRadioButton[RefrigState.NUM_CRYO_CABINETS];
    private final ButtonGroup lightsBG[] = new ButtonGroup[RefrigState.NUM_CRYO_CABINETS];
    private final JLabel[][] lightsState = new JLabel[RefrigState.NUM_CRYO_COMPS_PER_CAB][RefrigState.NUM_CRYO_CABINETS];
    private SystemStatusPanel statusPanel;
    private MonitorTaskPanel monTaskPanel;
    private int numColdPanels = 0, numCryoPanels = 0, numMonTasks = 0;
    private final CompressorSummaryLine[] coldLines = new CompressorSummaryLine[RefrigState.MAX_COLD_COMPRESSORS];
    private final CompressorSummaryLine[] cryoLines = new CompressorSummaryLine[RefrigState.MAX_CRYO_COMPRESSORS];
    private final UiUtilities uiUtils;

    public RefrigControlPanel(String agent) {
        sender = new CommandSender(agent, this);
        uiUtils = new UiUtilities(this);
        initComponents();
        (new DisablePanel()).run();
    }

    public void initPanel() {
        sender.sendCommand(true, null, CMND_GET_STATE);
    }

    @Override
    public void onCommandReply(Object reply, String path, String command, Object[] args) {
        RefrigState rs = (RefrigState)reply;
        if (!cmndError) {
            if (numMonTasks > 0) {
                remove(monTaskPanel);
            }
            monTaskPanel.initPanel(rs.getMonitorTasks());
            numMonTasks = rs.getNumMonTasks();
            if (numMonTasks > 0) {
                GridBagConstraints gbm = new GridBagConstraints();
                gbm.insets = new Insets(5, 0, 0, 0);
                gbm.anchor = GridBagConstraints.NORTH;
                gbm.gridx = 0;
                gbm.gridy = 1;
                add(monTaskPanel, gbm);
            }
            layoutCompressorPanel(rs.getNumColdComps(), rs.getNumCryoComps());
        }
        cmndError = false;
        updatePanel(rs);
    }

    @Override
    public void onCommandReject(String path, String command, Object[] args) {
        if (!command.equals(CMND_GET_STATE)) {
            cmndError = true;
            sender.sendCommand(true, null, CMND_GET_STATE);
        }
    }

    public void updatePanel(RefrigState rs) {
        SwingUtilities.invokeLater(new UpdateRefrigState(rs));
    }

    public void disablePanel() {
        SwingUtilities.invokeLater(new DisablePanel());
    }

    private void initComponents() {

        // Status line panel
        statusPanel = new SystemStatusPanel(sender, RefrigConstants.MONITOR_CONTROL, true);

        // Monitor task control panel
        monTaskPanel = new MonitorTaskPanel("Monitor Task Control", sender, RefrigConstants.MONITOR_CONTROL);

        // Set up the compressor summary lines and control panels
        for (int comp = 0; comp < coldLines.length; comp++) {
            coldLines[comp] = new CompressorSummaryLine(CompTypes.TYPE_COLD, sender);
        }
        for (int comp = 0; comp < cryoLines.length; comp++) {
            cryoLines[comp] = new CompressorSummaryLine(CompTypes.TYPE_CRYO, sender);
        }

        // Cabinet lights control panel
        for (int cab = 0; cab < RefrigState.NUM_CRYO_CABINETS; cab++) {
            for (int cmp = 0; cmp < RefrigState.NUM_CRYO_COMPS_PER_CAB; cmp++) {
                lightsState[cmp][cab] = UiUtilities.newLabel(SOLID_BAR, 0);
                lightsState[cmp][cab].setForeground(Color.red);
            }
        }
        lightsPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbl = new GridBagConstraints();
        gbl.insets = new Insets(4, 4, 4, 4);
        gbl.anchor = GridBagConstraints.WEST;
        gbl.gridx = 0;
        gbl.gridy = 0;
        for (int cab = 0; cab < RefrigState.NUM_CRYO_CABINETS; cab++) {
            lightsOffRB[cab] = uiUtils.newRadioButton("Off", "F" + cab);
            lightsOnRB[cab] = uiUtils.newRadioButton("On", "N" + cab);
            lightsBG[cab] = new ButtonGroup();
            lightsBG[cab].add(lightsOffRB[cab]);
            lightsBG[cab].add(lightsOnRB[cab]);
            lightsPanel.add(UiUtilities.newLabel("Cabinet " + cab + ":", 0), gbl);
            gbl.gridx++;
            gbl.insets.left = 0;
            lightsPanel.add(lightsOffRB[cab], gbl);
            gbl.gridx++;
            lightsPanel.add(lightsOnRB[cab], gbl);
            gbl.gridx++;
            gbl.insets.left = 4;
            for (int cmp = 0; cmp < RefrigState.NUM_CRYO_COMPS_PER_CAB; cmp++) {
                gbl.insets.right = (cmp < RefrigState.NUM_CRYO_COMPS_PER_CAB - 1) ? 0 : 4;
                lightsPanel.add(lightsState[cmp][cab], gbl);
                gbl.gridx++;
                gbl.insets.left = 0;
            }
            gbl.insets.left = 30;
        }

        // Lay out the initial complete panel
        setLayout(new GridBagLayout());
        GridBagConstraints gbm = new GridBagConstraints();
        gbm.insets = new Insets(4, 0, 4, 0);
        gbm.anchor = GridBagConstraints.NORTH;
        gbm.gridx = 0;
        gbm.gridy = 0;
        add(statusPanel, gbm);
        gbm.gridy++;
        add(compressorPanel, gbm);
        gbm.gridy++;
        add(lightsPanel, gbm);
    }

    private void layoutCompressorPanel(int numColdComps, int numCryoComps) {
        for (int j = 0; j < numColdPanels; j++) {
            compressorPanel.remove(coldLines[j]);
        }
        for (int j = 0; j < numCryoPanels; j++) {
            compressorPanel.remove(cryoLines[j]);
        }
        numColdPanels = numColdComps;
        numCryoPanels = numCryoComps;
        compressorPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbm = new GridBagConstraints();
        gbm.insets = new Insets(4, 0, 0, 0);
        gbm.anchor = GridBagConstraints.NORTH;
        gbm.gridx = 0;
        gbm.gridy = 0;
        for (int j = 0; j < numColdPanels; j++) {
            gbm.insets.bottom = (j == numColdPanels + numCryoPanels - 1) ? 4 : 0;
            compressorPanel.add(coldLines[j], gbm);
            gbm.gridy++;
        }
        for (int j = 0; j < numCryoPanels; j++) {
            gbm.insets.bottom = (j == numCryoPanels - 1) ? 4 : 0;
            compressorPanel.add(cryoLines[j], gbm);
            gbm.gridy++;
        }
    }

    @Override
    public void handleRadioButton(String name) {
        char action = name.charAt(0);
        int cab = Integer.valueOf(name.substring(1));
        sender.sendCommand(null, "switchLights", cab, action == 'N');
    }

    class UpdateRefrigState implements Runnable {

        private final RefrigState rs;

        UpdateRefrigState(RefrigState rs) {
            this.rs = rs;
        }

        @Override
        public void run() {
            statusPanel.updatePanel(rs.getTickMillis());
            monTaskPanel.updatePanel(rs.getMonitorTasks());
            for (CompState cs : rs.getColdStates()) {
                coldLines[cs.getIndex()].updateControlPanel(cs);
            }
            for (CompState cs : rs.getCryoStates()) {
                cryoLines[cs.getIndex()].updateControlPanel(cs);
            }
            for (int cab = 0; cab < RefrigState.NUM_CRYO_CABINETS; cab++) {
                boolean on = false;
                for (int cmp = 0; cmp < RefrigState.NUM_CRYO_COMPS_PER_CAB; cmp++) {
                    SwitchState state = rs.getLightState(cab, cmp);
                    lightsState[cmp][cab].setEnabled(true);
                    lightsState[cmp][cab].setForeground(state == SwitchState.OFF ? Color.BLACK : state == SwitchState.ON ? UiConstants.GREEN : UiConstants.BLUE);
                    on |= (state == SwitchState.ON);
                }
                JRadioButton selButton = on ? lightsOnRB[cab] : lightsOffRB[cab];
                selButton.setSelected(true);
                lightsOffRB[cab].setEnabled(true);
                lightsOnRB[cab].setEnabled(true);
            }
            repaint();
        }

    }

    class DisablePanel implements Runnable {

        @Override
        public void run() {
            statusPanel.disablePanel();
            monTaskPanel.disablePanel();
            for (CompressorSummaryLine comp : coldLines) {
                comp.disableSystem();
            }
            for (CompressorSummaryLine comp : cryoLines) {
                comp.disableSystem();
            }
            for (int cab = 0; cab < RefrigState.NUM_CRYO_CABINETS; cab++) {
                lightsOffRB[cab].setEnabled(false);
                lightsOnRB[cab].setEnabled(false);
                for (int cmp = 0; cmp < RefrigState.NUM_CRYO_COMPS_PER_CAB; cmp++) {
                    lightsState[cmp][cab].setEnabled(false);
                }
            }
            repaint();
        } 
    }

    private static final long serialVersionUID = 1L;
}
