package org.lsst.ccs.subsystem.refrig.ui;

import java.awt.Color;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.util.Collection;
import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import org.lsst.ccs.subsystem.common.ui.TextFieldX;
import org.lsst.ccs.subsystem.common.ui.UiConstants;
import org.lsst.ccs.subsystem.common.ui.UiUtilities;
import org.lsst.ccs.subsystem.common.ui.jas.CommandSender;
import org.lsst.ccs.subsystem.refrig.constants.ColdCompValves;
import org.lsst.ccs.subsystem.refrig.constants.CompConditions;
import org.lsst.ccs.subsystem.refrig.constants.CompLatches;
import org.lsst.ccs.subsystem.refrig.constants.CompSwConds;
import org.lsst.ccs.subsystem.refrig.constants.CompSwitches;
import org.lsst.ccs.subsystem.refrig.constants.CompTypes;
import org.lsst.ccs.subsystem.refrig.constants.CompressorState;
import org.lsst.ccs.subsystem.refrig.constants.ConditionState;
import org.lsst.ccs.subsystem.refrig.constants.LatchState;
import org.lsst.ccs.subsystem.refrig.constants.SwCondState;
import org.lsst.ccs.subsystem.refrig.constants.SwitchState;
import org.lsst.ccs.subsystem.refrig.constants.VfdState;
import org.lsst.ccs.subsystem.refrig.data.ColdCompState;
import org.lsst.ccs.subsystem.refrig.data.CompState;
import org.lsst.ccs.subsystem.refrig.data.CryoCompState;

/**
 *  Implements the compressor control panel.
 *
 *  @author Owen Saxton
 */
public class CompressorControlPanel extends JPanel implements UiUtilities.ActionHandler {

    private static final int
        SW_TYPE_INVISIBLE = 0,
        SW_TYPE_POWER     = 1,
        SW_TYPE_VALVE_NC  = 2,
        SW_TYPE_VALVE_NO  = 3;

    private static final int[] switchTypes = new int[CompSwitches.NUM_SWITCHES];
    static {
        switchTypes[CompSwitches.SW_ENABLE]        = SW_TYPE_POWER;
        switchTypes[CompSwitches.SW_LIGHTS]        = SW_TYPE_POWER;
        switchTypes[CompSwitches.SW_HEATER]        = SW_TYPE_POWER;
        switchTypes[CompSwitches.SW_ORIFICE_VALVE] = SW_TYPE_VALVE_NC;
        switchTypes[CompSwitches.SW_COOLANT_VALVE] = SW_TYPE_VALVE_NO;
        switchTypes[CompSwitches.SW_BYPASS_VALVE]  = SW_TYPE_VALVE_NC;
        switchTypes[CompSwitches.SW_SURGE_HEATER]  = SW_TYPE_POWER;
        switchTypes[CompSwitches.SW_ENABLE_ALERTS] = SW_TYPE_POWER;
        switchTypes[CompSwitches.SW_CABINET_LIGHT] = SW_TYPE_INVISIBLE;
    }
    private static final int COMP_STATE_WIDTH = UiUtilities.maxEnumLabelWidth(CompressorState.class);
    private static final int SWITCH_STATE_WIDTH = Math.max(UiUtilities.maxEnumLabelWidth(SwitchState.class),
                                                           UiUtilities.maxLabelWidth(new String[]{"OPEN", "SHUT"}, ""));
    private static final int LATCH_STATE_WIDTH = UiUtilities.maxEnumLabelWidth(LatchState.class);
    private static final int CONDITION_STATE_WIDTH = UiUtilities.maxEnumLabelWidth(ConditionState.class);
    private static final int SW_COND_STATE_WIDTH = UiUtilities.maxEnumLabelWidth(SwCondState.class);
    private static final int HORZ_INSET = UiUtilities.getLabelSize("M").width / 2;
    private static final int VERT_INSET = UiUtilities.getLabelSize("M").height / 4;

    private String comp;
    private final CommandSender sender;
    private final int type;
    private final Collection<Integer> validSwitches, validLatches, validConditions, validSwConds;

    private final JPanel headPanel = new JPanel();
    private JLabel compName, compState;

    private final JPanel switchPanel = UiUtilities.newBorderedPanel("Switches");
    private final JLabel[] switchStatus = new JLabel[CompSwitches.NUM_SWITCHES];
    private final ButtonGroup[] switchBG = new ButtonGroup[CompSwitches.NUM_SWITCHES];
    private final JRadioButton[] switchOffRB = new JRadioButton[CompSwitches.NUM_SWITCHES];
    private final JRadioButton[] switchOnRB = new JRadioButton[CompSwitches.NUM_SWITCHES];

    private final JPanel valvePanel = UiUtilities.newBorderedPanel("Cold Valve Control");
    private final TextFieldX[] valvePosition = new TextFieldX[ColdCompValves.NUM_VALVES];

    private final JPanel vfdPanel = UiUtilities.newBorderedPanel("VFD Control");
    private JLabel vfdState;
    private JButton vfdReset;
    private TextFieldX vfdFrequency;

    private final JPanel latchPanel = UiUtilities.newBorderedPanel("PLC Error Conditions");
    private final JLabel[] latchStatus = new JLabel[CompLatches.NUM_LATCHES];
    private JButton plcReset;

    private final JPanel conditionPanel = UiUtilities.newBorderedPanel("PLC Running Conditions");
    private final JLabel[] conditionStatus = new JLabel[CompConditions.NUM_CONDITIONS];

    private final JPanel swCondPanel = UiUtilities.newBorderedPanel("CCS Error Conditions");
    private final JLabel[] swCondStatus = new JLabel[CompSwConds.NUM_SW_CONDITIONS];

    private final UiUtilities uiUtils;

    public CompressorControlPanel(int type, CommandSender cons) {
        this.type = type;
        sender = cons;
        validSwitches = type == CompTypes.TYPE_COLD ? ColdCompState.SWITCHES : CryoCompState.SWITCHES;
        validLatches = type == CompTypes.TYPE_COLD ? ColdCompState.LATCHES : CryoCompState.LATCHES;
        validConditions = type == CompTypes.TYPE_COLD ? ColdCompState.CONDITIONS : CryoCompState.CONDITIONS;
        validSwConds = type == CompTypes.TYPE_COLD ? ColdCompState.SW_CONDITIONS : CryoCompState.SW_CONDITIONS;
        uiUtils = new UiUtilities(this);
        initComponents();
        disableSystem();
    }

    public void updateControlPanel(CompState rs) {
        comp = rs.getName();
        compName.setText(comp);
        compName.setEnabled(true);
        CompressorState cState = rs.getCompressorState();
        Color cColor = cState == CompressorState.OFFLINE ? UiConstants.BLUE :
                       cState == CompressorState.STOPPED ? Color.BLACK :
                       cState == CompressorState.RUNNING ? UiConstants.GREEN :
                       cState == CompressorState.WAITING ? UiConstants.YELLOW : UiConstants.RED;
        compState.setText(cState.name());
        compState.setForeground(cColor);
        compState.setEnabled(true);
        for (int sw : rs.getValidSwitches()) {
            SwitchState state = rs.getSwitchState(sw);
            String text = state.name();
            int swType = switchTypes[sw];
            if (swType == SW_TYPE_INVISIBLE) continue;
            if (swType != SW_TYPE_POWER) {
                text = state == SwitchState.ON ? swType == SW_TYPE_VALVE_NC ? "OPEN" : "SHUT" :
                       state == SwitchState.OFF ? swType == SW_TYPE_VALVE_NC ? "SHUT" : "OPEN" : text;
            }
            Color color = state == SwitchState.OFF ? swType == SW_TYPE_VALVE_NO ? UiConstants.GREEN : Color.black :
                          state == SwitchState.ON ? swType == SW_TYPE_VALVE_NO ? Color.black : UiConstants.GREEN : UiConstants.BLUE;
            switchStatus[sw].setText(text);
            switchStatus[sw].setForeground(color);
            switchStatus[sw].setEnabled(true);
            JRadioButton selButton = state == SwitchState.ON ? switchOnRB[sw] : switchOffRB[sw];
            selButton.setSelected(true);
            switchOffRB[sw].setEnabled(true);
            switchOnRB[sw].setEnabled(true);
        }
        if (type == CompTypes.TYPE_COLD) {
            ColdCompState crs = (ColdCompState)rs;
            for (int valve = 0; valve < ColdCompValves.NUM_VALVES; valve++) {
                valvePosition[valve].update(String.format("%.1f", 100.0 * crs.getValvePosition(valve)), true);
            }
            VfdState vState = crs.getVfdState();
            cColor = vState == VfdState.FAULT ? UiConstants.RED :
                     vState == VfdState.ALARM ? UiConstants.YELLOW :
                     vState == VfdState.RUNNING || vState == VfdState.REVERSE ? UiConstants.GREEN :
                     vState == VfdState.OFFLINE ? UiConstants.BLUE : Color.BLACK;
            vfdState.setText(vState.name());
            vfdState.setForeground(cColor);
            vfdState.setEnabled(true);
            vfdFrequency.update(crs.getVfdFrequency(), true);
        }
        for (int cond : rs.getValidLatches()) {
            LatchState state = rs.getLatchState(cond);
            Color color = state == LatchState.OFFLINE ? UiConstants.BLUE :
                          state == LatchState.CLEAR ? UiConstants.GREEN :
                          state == LatchState.WARNING ? UiConstants.YELLOW :
                          state == LatchState.ACTIVE ? UiConstants.RED : UiConstants.PURPLE;
            latchStatus[cond].setText(state.name());
            latchStatus[cond].setForeground(color);
            latchStatus[cond].setEnabled(true);
        }
        plcReset.setEnabled(true);
        for (int cond : rs.getValidConditions()) {
            ConditionState state = rs.getConditionState(cond);
            Color color = state == ConditionState.OFF ? UiConstants.BLUE :
                          state == ConditionState.NO ? Color.BLACK : UiConstants.GREEN;
            conditionStatus[cond].setText(state.name());
            conditionStatus[cond].setForeground(color);
            conditionStatus[cond].setEnabled(true);
        }
        for (int cond : rs.getValidSwConditions()) {
            SwCondState state = rs.getSwConditionState(cond);
            Color color = state == SwCondState.OFFLINE ? UiConstants.BLUE :
                          state == SwCondState.CLEAR ? UiConstants.GREEN :
                          state == SwCondState.DLYPEND ? UiConstants.PURPLE : UiConstants.RED;
            swCondStatus[cond].setText(state.name());
            swCondStatus[cond].setForeground(color);
            swCondStatus[cond].setEnabled(true);
        }
        repaint();
    }

    public final void disableSystem() {
        compName.setEnabled(false);
        compState.setEnabled(false);
        for (int sw : validSwitches) {
            if (switchTypes[sw] == SW_TYPE_INVISIBLE) continue;
            switchStatus[sw].setEnabled(false);
            switchOffRB[sw].setEnabled(false);
            switchOnRB[sw].setEnabled(false);
        }
        if (type == CompTypes.TYPE_COLD) {
            for (TextFieldX vPosition : valvePosition) {
                vPosition.setDisabled();
            }
            vfdFrequency.setDisabled();
            vfdState.setEnabled(false);
        }
        for (int cond : validLatches) {
            latchStatus[cond].setEnabled(false);
        }
        plcReset.setEnabled(false);
        for (int cond : validConditions) {
            conditionStatus[cond].setEnabled(false);
        }
        for (int cond : validSwConds) {
            swCondStatus[cond].setEnabled(false);
        }
        repaint();
    }

    private void initComponents() {

        // Heading fields
        compName = UiUtilities.newLabel("MM99", 0);
        compState = UiUtilities.newLabel("X", COMP_STATE_WIDTH);

        // Switch states
        for (int sw : validSwitches) {
            int swType = switchTypes[sw];
            if (swType == SW_TYPE_INVISIBLE) continue;
            switchStatus[sw] = UiUtilities.newLabel("XXX", SWITCH_STATE_WIDTH);
            switchOffRB[sw] = uiUtils.newRadioButton(swType == SW_TYPE_POWER ? "Off" : swType == SW_TYPE_VALVE_NC ? "Shut" : "Open", "F" + sw);
            switchOnRB[sw] = uiUtils.newRadioButton(swType == SW_TYPE_POWER ? "On" : swType == SW_TYPE_VALVE_NC ? "Open" : "Shut", "T" + sw);
            ButtonGroup bg = switchBG[sw] = new ButtonGroup();
            bg.add(switchOffRB[sw]);
            bg.add(switchOnRB[sw]);
        }

        // Cold VFD & valve control
        if (type == CompTypes.TYPE_COLD) {
            vfdState = UiUtilities.newLabel("XXX", UiUtilities.maxEnumLabelWidth(VfdState.class));
            vfdReset = uiUtils.newButton("Reset Fault", "V", true);
            vfdFrequency = uiUtils.newTextFieldX("00.0000", "F", TextFieldX.TYPE_DOUBLE, true);
            for (int valve = 0; valve < ColdCompValves.NUM_VALVES; valve++) {
                valvePosition[valve] = uiUtils.newTextFieldX("0000.00", "V" + valve, TextFieldX.TYPE_DOUBLE, true);
            }
        }

        // Latched condition states
        for (int cond : validLatches) {
            latchStatus[cond] = UiUtilities.newLabel("XXX", LATCH_STATE_WIDTH);
        }
        plcReset = uiUtils.newButton("Reset", "P", true);

        // Condition states
        for (int cond : validConditions) {
            conditionStatus[cond] = UiUtilities.newLabel("XXX", CONDITION_STATE_WIDTH);
        }

        // Software condition states
        for (int cond : validSwConds) {
            swCondStatus[cond] = UiUtilities.newLabel("XXX", SW_COND_STATE_WIDTH);
        }

        // Lay out the head panel
        headPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbh = new GridBagConstraints();
        gbh.anchor = GridBagConstraints.WEST;
        gbh.gridx = 0;
        gbh.gridy = 0;
        headPanel.add(UiUtilities.newLabel("Compressor:  ", 0), gbh);
        gbh.gridx++;
        headPanel.add(compName, gbh);
        gbh.gridx++;
        gbh.insets.left = 30;
        headPanel.add(UiUtilities.newLabel("State:  ", 0), gbh);
        gbh.gridx++;
        gbh.insets.left = 0;
        headPanel.add(compState, gbh);
        
        // Lay out the switch panel
        GridBagConstraints gbs = new GridBagConstraints();
        gbs.anchor = GridBagConstraints.WEST;
        gbs.insets = new Insets(0, 0, VERT_INSET, HORZ_INSET);
        gbs.gridy = -1;
        int ncols = 2, colm = 0;
        for (int sw : validSwitches) {
            int swType = switchTypes[sw];
            if (swType == SW_TYPE_INVISIBLE) continue;
            if (colm == 0) {
                gbs.gridx = 0;
                gbs.gridy++;
            }
            gbs.insets.top = gbs.gridy == 0 ? VERT_INSET : 0;
            gbs.insets.left = colm == 0 ? HORZ_INSET : 40;
            switchPanel.add(UiUtilities.newLabel(CompSwitches.DESCS[sw] + ":", 0), gbs);
            gbs.insets.left = HORZ_INSET;
            gbs.gridx++;
            switchPanel.add(switchStatus[sw], gbs);
            gbs.gridx++;
            switchPanel.add(swType == SW_TYPE_VALVE_NO ? switchOnRB[sw] : switchOffRB[sw], gbs);
            gbs.gridx++;
            gbs.insets.left = 0;
            switchPanel.add(swType == SW_TYPE_VALVE_NO ? switchOffRB[sw] : switchOnRB[sw], gbs);
            gbs.gridx++;
            colm = (colm + 1) % ncols;
        }

        // Lay out the cold valve control panel
        if (type == CompTypes.TYPE_COLD) {
            GridBagConstraints gbd = new GridBagConstraints();
            gbd.anchor = GridBagConstraints.WEST;
            gbd.insets = new Insets(VERT_INSET, HORZ_INSET, VERT_INSET, HORZ_INSET);
            gbd.gridy = -1;
            ncols = 3;
            colm = 0;
            for (int valve = 0; valve < ColdCompValves.NUM_VALVES; valve++) {
                if (colm == 0) {
                    gbd.gridx = 0;
                    gbd.gridy++;
                }
                gbd.insets.top = gbd.gridy == 0 ? VERT_INSET : -VERT_INSET;
                gbd.insets.left = colm == 0 ? HORZ_INSET : 30;
                valvePanel.add(UiUtilities.newLabel(ColdCompValves.DESCS[valve] + ":", 0), gbd);
                gbd.gridx++;
                gbd.insets.left = HORZ_INSET;
                valvePanel.add(valvePosition[valve], gbd);
                gbd.gridx++;
                gbd.insets.left = 0;
                valvePanel.add(UiUtilities.newLabel("%", 0), gbd);
                gbd.gridx++;
                colm = (colm + 1) % ncols;
           }
        }

        // Lay out the cold VFD control panel
        if (type == CompTypes.TYPE_COLD) {
            GridBagConstraints gbf = new GridBagConstraints();
            gbf.anchor = GridBagConstraints.WEST;
            gbf.insets = new Insets(VERT_INSET, HORZ_INSET, VERT_INSET, HORZ_INSET);
            gbf.gridx = 0;
            gbf.gridy = 0;
            vfdPanel.add(UiUtilities.newLabel("State:", 0), gbf);
            gbf.gridx++;
            vfdPanel.add(vfdState, gbf);
            gbf.gridx++;
            gbf.insets.left = 30;
            vfdPanel.add(vfdReset, gbf);
            gbf.gridx++;
            vfdPanel.add(UiUtilities.newLabel("Frequency:", 0), gbf);
            gbf.gridx++;
            gbf.insets.left = HORZ_INSET;
            vfdPanel.add(vfdFrequency, gbf);
            gbf.gridx++;
            gbf.insets.left = 0;
            vfdPanel.add(UiUtilities.newLabel("Hz", 0), gbf);
        }

        // Lay out the latched conditions panel
        GridBagConstraints gbl = new GridBagConstraints();
        gbl.anchor = GridBagConstraints.WEST;
        gbl.insets = new Insets(0, 0, VERT_INSET, HORZ_INSET);
        gbl.gridy = -1;
        ncols = 1;
        colm = 0;
        for (int cond : validLatches) {
            if (colm == 0) {
                gbl.gridx = 0;
                gbl.gridy++;
            }
            gbl.insets.top = gbl.gridy == 0 ? VERT_INSET : 0;
            gbl.insets.left = colm == 0 ? HORZ_INSET : 30;
            latchPanel.add(UiUtilities.newLabel(CompLatches.DESCS[cond] + ":", 0), gbl);
            gbl.insets.left = HORZ_INSET;
            gbl.gridx++;
            latchPanel.add(latchStatus[cond], gbl);
            gbl.gridx++;
            colm = (colm + 1) % ncols;
        }
        gbl.anchor = GridBagConstraints.NORTH;
        gbl.gridwidth = 2 * ncols;
        gbl.gridx = 0;
        gbl.gridy++;
        gbl.insets.top = VERT_INSET;
        latchPanel.add(plcReset, gbl);

        // Lay out the conditions panel
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.anchor = GridBagConstraints.WEST;
        gbc.insets = new Insets(0, 0, VERT_INSET, HORZ_INSET);
        gbc.gridy = -1;
        ncols = 2;
        colm = 0;
        for (int cond : validConditions) {
            if (colm == 0) {
                gbc.gridx = 0;
                gbc.gridy++;
            }
            gbc.insets.top = gbc.gridy == 0 ? VERT_INSET : 0;
            gbc.insets.left = colm == 0 ? HORZ_INSET : 30;
            conditionPanel.add(UiUtilities.newLabel(CompConditions.DESCS[cond] + ":", 0), gbc);
            gbc.insets.left = HORZ_INSET; //0
            gbc.gridx++;
            conditionPanel.add(conditionStatus[cond], gbc);
            gbc.gridx++;
            colm = (colm + 1) % ncols;
        }

        // Lay out the software conditions panel
        GridBagConstraints gbw = new GridBagConstraints();
        gbw.anchor = GridBagConstraints.WEST;
        gbw.insets = new Insets(0, 0, VERT_INSET, HORZ_INSET);
        gbw.gridy = -1;
        ncols = 2;
        colm = 0;
        for (int cond : validSwConds) {
            if (colm == 0) {
                gbw.gridx = 0;
                gbw.gridy++;
            }
            gbw.insets.top = gbw.gridy == 0 ? VERT_INSET : 0;
            gbw.insets.left = colm == 0 ? HORZ_INSET : 30;
            swCondPanel.add(UiUtilities.newLabel(CompSwConds.DESCS[cond] + ":", 0), gbw);
            gbw.insets.left = HORZ_INSET; //0
            gbw.gridx++;
            swCondPanel.add(swCondStatus[cond], gbw);
            gbw.gridx++;
            colm = (colm + 1) % ncols;
        }

        // Lay out this panel
        setLayout(new GridBagLayout());
        GridBagConstraints gbm = new GridBagConstraints();
        gbm.insets = new Insets(12, 6, 6, 6);
        gbm.anchor = GridBagConstraints.NORTH;
        gbm.gridx = 0;
        gbm.gridy = 0;
        gbm.gridwidth = 2;
        add(headPanel, gbm);
        gbm.gridy++;
        add(switchPanel, gbm);
        gbm.gridy++;
        gbm.insets.top = 6;
        if (type == CompTypes.TYPE_COLD) {
            add(valvePanel, gbm);
            gbm.gridy++;
            add(vfdPanel, gbm);
            gbm.gridy++;
        }
        gbm.gridwidth = 1;
        gbm.gridheight = 2;
        add(latchPanel, gbm);
        gbm.gridx++;
        gbm.gridheight = 1;
        add(conditionPanel, gbm);
        gbm.gridy++;
        add(swCondPanel, gbm);
    }

    @Override
    public void handleRadioButton(String name) {
        char action = name.charAt(0);
        int sw = Integer.valueOf(name.substring(1));
        sender.sendCommand(comp, "setSwitchOn", CompSwitches.ID_MAP.get(sw), action == 'T');
    }

    @Override
    public void handleTextFieldX(String name, Object value) {
        char action = name.charAt(0);
        if (action == 'V') {
            int valve = Integer.valueOf(name.substring(1));
            sender.sendCommand(comp, "setValvePosition", ColdCompValves.ID_MAP.get(valve), (Double)value / 100.0);
        }
        else if (action == 'F') {
            sender.sendCommand(comp, "setVfdFrequency", value);
        }
    }

    @Override
    public void handleButton(String name) {
        char action = name.charAt(0);
        if (action == 'P') {
            sender.sendCommand(comp, "resetLatches");
        }
        else if (action == 'V') {
            sender.sendCommand(comp, "resetVfdFault");
        }
    }

    private static final long serialVersionUID = 1L;
}
