package org.lsst.ccs.subsystem.refrig.ui;

import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import javax.swing.JPanel;
import javax.swing.SwingUtilities;
import org.lsst.ccs.subsystem.common.ui.MonitorTaskPanel;
import org.lsst.ccs.subsystem.common.ui.SystemStatusPanel;
import org.lsst.ccs.subsystem.common.ui.jas.CommandSender;
import org.lsst.ccs.subsystem.refrig.constants.RefrigConstants;
import org.lsst.ccs.subsystem.refrig.data.HexState;

/**
 *  Implements the heat exchanger control panel.
 *
 *  @author Owen Saxton
 */
public class HexControlPanel extends JPanel implements CommandSender.ReplyHandler {

    private final CommandSender sender;

    private SystemStatusPanel statusPanel;
    private MonitorTaskPanel monTaskPanel;
    private int numMonTasks = 0;

    public HexControlPanel(String agent) {
        sender = new CommandSender(agent, this);
        initComponents();
        (new DisablePanel()).run();
    }

    public void initPanel() {
        sender.sendCommand(true, null, "getSystemState");
    }

    @Override
    public void onCommandReply(Object reply, String path, String command, Object[] args) {
        HexState hs = (HexState)reply;
        if (numMonTasks > 0) {
            remove(monTaskPanel);
        }
        monTaskPanel.initPanel(hs.getMonitorTasks());
        numMonTasks = hs.getNumMonTasks();
        if (numMonTasks > 0) {
            GridBagConstraints gbm = new GridBagConstraints();
            gbm.insets = new Insets(5, 0, 0, 0);
            gbm.anchor = GridBagConstraints.NORTH;
            gbm.gridx = 0;
            gbm.gridy = 1;
            add(monTaskPanel, gbm);
        }
        updatePanel(hs);
    }

    public void updatePanel(HexState hs) {
        SwingUtilities.invokeLater(new UpdateHexState(hs));
    }

    public void disablePanel() {
        SwingUtilities.invokeLater(new DisablePanel());
    }

    private void initComponents() {

        // Status line panel
        statusPanel = new SystemStatusPanel(sender, RefrigConstants.MONITOR_CONTROL, true);

        // Monitor task control panel
        monTaskPanel = new MonitorTaskPanel("Monitor Task Control", sender, RefrigConstants.MONITOR_CONTROL);

        // Lay out the initial complete panel
        setLayout(new GridBagLayout());
        GridBagConstraints gbm = new GridBagConstraints();
        gbm.insets = new Insets(4, 0, 0, 0);
        gbm.anchor = GridBagConstraints.NORTH;
        gbm.gridx = 0;
        gbm.gridy = 0;
        add(statusPanel, gbm);
    }

    class UpdateHexState implements Runnable {

        private final HexState hs;

        UpdateHexState(HexState hs) {
            this.hs = hs;
        }

        @Override
        public void run() {
            statusPanel.updatePanel(hs.getTickMillis());
            monTaskPanel.updatePanel(hs.getMonitorTasks());
            repaint();
        }

    }

    class DisablePanel implements Runnable {

        @Override
        public void run() {
            statusPanel.disablePanel();
            monTaskPanel.disablePanel();
            repaint();
        } 
    }

    private static final long serialVersionUID = 1L;
}
