package org.lsst.ccs.subsystem.refrig.ui;

import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import javax.swing.JPanel;
import javax.swing.SwingUtilities;
import org.lsst.ccs.subsystem.common.ui.MonitorTaskPanel;
import org.lsst.ccs.subsystem.common.ui.SystemStatusPanel;
import org.lsst.ccs.subsystem.common.ui.UiUtilities;
import org.lsst.ccs.subsystem.common.ui.jas.CommandSender;
import org.lsst.ccs.subsystem.refrig.constants.CompTypes;
import org.lsst.ccs.subsystem.refrig.constants.RefrigConstants;
import org.lsst.ccs.subsystem.refrig.data.CompState;
import org.lsst.ccs.subsystem.refrig.data.RefrigState;

/**
 *  Implements the refrigeration control panel.
 *
 *  @author Owen Saxton
 */
public class RefrigControlPanel extends JPanel implements CommandSender.ReplyHandler {

    private final CommandSender sender;

    private final JPanel compressorPanel = UiUtilities.newBorderedPanel("Compressor Control");
    private SystemStatusPanel statusPanel;
    private MonitorTaskPanel monTaskPanel;
    private int numColdPanels = 0, numCryoPanels = 0, numMonTasks = 0;
    private final CompressorSummaryLine[] coldLines = new CompressorSummaryLine[RefrigState.MAX_COLD_COMPRESSORS];
    private final CompressorSummaryLine[] cryoLines = new CompressorSummaryLine[RefrigState.MAX_CRYO_COMPRESSORS];

    public RefrigControlPanel(String agent) {
        sender = new CommandSender(agent, this);
        initComponents();
        (new DisablePanel()).run();
    }

    public void initPanel() {
        sender.sendCommand(true, null, "getSystemState");
    }

    @Override
    public void onCommandReply(Object reply, String path, String command, Object[] args) {
        RefrigState rs = (RefrigState)reply;
        if (numMonTasks > 0) {
            remove(monTaskPanel);
        }
        monTaskPanel.initPanel(rs.getMonitorTasks());
        numMonTasks = rs.getNumMonTasks();
        if (numMonTasks > 0) {
            GridBagConstraints gbm = new GridBagConstraints();
            gbm.insets = new Insets(5, 0, 0, 0);
            gbm.anchor = GridBagConstraints.NORTH;
            gbm.gridx = 0;
            gbm.gridy = 1;
            add(monTaskPanel, gbm);
        }
        layoutCompressorPanel(rs.getNumColdComps(), rs.getNumCryoComps());
        updatePanel(rs);
    }

    public void updatePanel(RefrigState rs) {
        SwingUtilities.invokeLater(new UpdateRefrigState(rs));
    }

    public void disablePanel() {
        SwingUtilities.invokeLater(new DisablePanel());
    }

    private void initComponents() {

        // Status line panel
        statusPanel = new SystemStatusPanel(sender, RefrigConstants.MONITOR_CONTROL);

        // Monitor task control panel
        monTaskPanel = new MonitorTaskPanel("Monitor Task Control", sender, RefrigConstants.MONITOR_CONTROL);

        // Set up the compressor summary lines and control panels
        for (int comp = 0; comp < coldLines.length; comp++) {
            coldLines[comp] = new CompressorSummaryLine(CompTypes.TYPE_COLD, sender);
        }
        for (int comp = 0; comp < cryoLines.length; comp++) {
            cryoLines[comp] = new CompressorSummaryLine(CompTypes.TYPE_CRYO, sender);
        }

        // Lay out the initial complete panel
        setLayout(new GridBagLayout());
        GridBagConstraints gbm = new GridBagConstraints();
        gbm.insets = new Insets(4, 0, 4, 0);
        gbm.anchor = GridBagConstraints.NORTH;
        gbm.gridx = 0;
        gbm.gridy = 0;
        add(statusPanel, gbm);
        gbm.gridy = 2;
        add(compressorPanel, gbm);
    }

    private void layoutCompressorPanel(int numColdComps, int numCryoComps) {
        for (int j = 0; j < numColdPanels; j++) {
            compressorPanel.remove(coldLines[j]);
        }
        for (int j = 0; j < numCryoPanels; j++) {
            compressorPanel.remove(cryoLines[j]);
        }
        numColdPanels = numColdComps;
        numCryoPanels = numCryoComps;
        compressorPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbm = new GridBagConstraints();
        gbm.insets = new Insets(4, 0, 0, 0);
        gbm.anchor = GridBagConstraints.NORTH;
        gbm.gridx = 0;
        gbm.gridy = 0;
        for (int j = 0; j < numColdPanels; j++) {
            gbm.insets.bottom = (j == numColdPanels + numCryoPanels - 1) ? 4 : 0;
            compressorPanel.add(coldLines[j], gbm);
            gbm.gridy++;
        }
        for (int j = 0; j < numCryoPanels; j++) {
            gbm.insets.bottom = (j == numCryoPanels - 1) ? 4 : 0;
            compressorPanel.add(cryoLines[j], gbm);
            gbm.gridy++;
        }
    }

    class UpdateRefrigState implements Runnable {

        private final RefrigState rs;

        UpdateRefrigState(RefrigState rs) {
            this.rs = rs;
        }

        @Override
        public void run() {
            statusPanel.updatePanel(rs.getTickMillis());
            monTaskPanel.updatePanel(rs.getMonitorTasks());
            for (CompState cs : rs.getColdStates()) {
                coldLines[cs.getIndex()].updateControlPanel(cs);
            }
            for (CompState cs : rs.getCryoStates()) {
                cryoLines[cs.getIndex()].updateControlPanel(cs);
            }
            repaint();
        }

    }

    class DisablePanel implements Runnable {

        @Override
        public void run() {
            statusPanel.disablePanel();
            monTaskPanel.disablePanel();
            for (CompressorSummaryLine comp : coldLines) {
                comp.disableSystem();
            }
            for (CompressorSummaryLine comp : cryoLines) {
                comp.disableSystem();
            }
            repaint();
        } 
    }

    private static final long serialVersionUID = 1L;
}
