package org.lsst.ccs.subsystem.refrig.ui;

import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import javax.swing.JPanel;
import javax.swing.SwingUtilities;
import org.lsst.ccs.subsystem.common.ui.SystemStatusPanel;
import org.lsst.ccs.subsystem.common.ui.UiUtilities;
import org.lsst.ccs.subsystem.common.ui.jas.CommandSender;
import org.lsst.ccs.subsystem.refrig.data.ThermalState;

/**
 *  Implements the thermal control panel.
 *
 *  @author Owen Saxton
 */
public class ThermalControlPanel extends JPanel implements CommandSender.ReplyHandler, UiUtilities.ActionHandler {

    private final CommandSender sender;
    private SystemStatusPanel statusPanel;
    private final TrimHeaterPanel[] trimHeaterPanels = new TrimHeaterPanel[ThermalState.NUM_TRIM_HEATERS];
    private final AuxHeaterPanel[] auxHeaterPanels = new AuxHeaterPanel[ThermalState.NUM_AUX_HEATERS];
    private JPanel trimPanel;
    private JPanel auxPanel;
    private final JPanel mainPanel = new JPanel();
    private Integer[] trimHeaters = new Integer[0];
    private Integer[] auxHeaters = new Integer[0];

    public ThermalControlPanel(String agent) {
        sender = new CommandSender(agent, this);
        initComponents();
        (new DisablePanel()).run();
    }

    public void initPanel() {
        sender.sendCommand(true, null, "getSystemState");
    }

    @Override
    public void onCommandReply(Object reply, String path, String command, Object[] args) {
        updatePanel((ThermalState)reply);
    }

    public void updatePanel(ThermalState rs) {
        SwingUtilities.invokeLater(new UpdateThermalState(rs));
    }

    public void disablePanel() {
        SwingUtilities.invokeLater(new DisablePanel());
    }

    private void initComponents() {

        // Create status line panel
        statusPanel = new SystemStatusPanel(sender);

        // Create trim heater line panels
        for (int htr = 0; htr < ThermalState.NUM_TRIM_HEATERS; htr++) {
            trimHeaterPanels[htr] = new TrimHeaterPanel(htr, sender);
        }

        // Create trim heater control panel
        trimPanel = UiUtilities.newBorderedPanel("Trim heater control");

        // Create auxiliary heater line control panels
        for (int htr = 0; htr < ThermalState.NUM_AUX_HEATERS; htr++) {
            auxHeaterPanels[htr] = new AuxHeaterPanel(htr, sender);
        }

        // Create auxiliary heater control panel
        auxPanel = UiUtilities.newBorderedPanel("Auxiliary heater control");
        
        // Add panels to main panel
        mainPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbm = new GridBagConstraints();
        gbm.insets = new Insets(5, 0, 0, 0);
        gbm.gridx = 0;
        gbm.gridy = 0;
        gbm.anchor = GridBagConstraints.NORTH;
        mainPanel.add(statusPanel, gbm);
        gbm.gridy++;
        gbm.insets.bottom = 5;
        mainPanel.add(trimPanel, gbm);

        // Lay out the complete panel
        add(mainPanel);
    }

    private void updateTrimPanel(Integer[] newTrimHeaters) {
        for (int htr : trimHeaters) {
            trimPanel.remove(trimHeaterPanels[htr]);
        }
        trimHeaters = newTrimHeaters;
        GridBagConstraints gbt = new GridBagConstraints();
        gbt.insets = new Insets(5, 0, 0, 0);
        gbt.gridx = 0;
        gbt.gridy = 0;
        gbt.anchor = GridBagConstraints.NORTH;
        for (int htr : trimHeaters) {
            gbt.insets.bottom = gbt.gridy == trimHeaters.length - 1 ? 5 : 0;
            trimPanel.add(trimHeaterPanels[htr], gbt);
            gbt.insets.top = 0;
            gbt.gridy++;
        }
    }

    private void updateAuxPanel(Integer[] newAuxHeaters) {
        if (auxHeaters.length > 0) {
            for (int htr : auxHeaters) {
                auxPanel.remove(auxHeaterPanels[htr]);
            }
            mainPanel.remove(auxPanel);
        }
        auxHeaters = newAuxHeaters;
        if (auxHeaters.length > 0) {
            GridBagConstraints gba = new GridBagConstraints();
            gba.insets = new Insets(5, 0, 0, 0);
            gba.gridx = 0;
            gba.gridy = 0;
            gba.anchor = GridBagConstraints.NORTH;
            for (int htr : auxHeaters) {
                gba.insets.bottom = gba.gridy == auxHeaters.length - 1 ? 5 : 0;
                auxPanel.add(auxHeaterPanels[htr], gba);
                gba.insets.top = 0;
                gba.gridy++;
            }
            GridBagConstraints gbm = new GridBagConstraints();
            gbm.insets = new Insets(5, 0, 0, 0);
            gbm.gridx = 0;
            gbm.gridy = 2;
            gbm.anchor = GridBagConstraints.NORTH;
            mainPanel.add(auxPanel, gbm);
        }
    }

    public ThermalControlPanel() {
        this.sender = null;
    }

    class UpdateThermalState implements Runnable {

        private final ThermalState rs;

        UpdateThermalState(ThermalState rs) {
            this.rs = rs;
        }

        @Override
        public void run() {
            statusPanel.updatePanel(rs.getTickMillis());
            if (rs.getTrimHeaters().length != trimHeaters.length) {
                updateTrimPanel(rs.getTrimHeaters());
            }
            for (int htr : trimHeaters) {
                trimHeaterPanels[htr].updatePanel(rs);
            }
            if (rs.getAuxHeaters().length != auxHeaters.length) {
                updateAuxPanel(rs.getAuxHeaters());
            }
            for (int htr : auxHeaters) {
                auxHeaterPanels[htr].updatePanel(rs);
            }
            repaint();
        }

    }

    class DisablePanel implements Runnable {

        @Override
        public void run() {
            statusPanel.disablePanel();
            for (int htr : trimHeaters) {
                trimHeaterPanels[htr].disablePanel();
            }
            for (int htr : auxHeaters) {
                auxHeaterPanels[htr].disablePanel();
            }
            repaint();
        }
 
    }

    private static final long serialVersionUID = 1L;
}
