package org.lsst.ccs.subsystem.refrig.ui;

import java.awt.Color;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.util.Collection;
import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import org.lsst.ccs.subsystem.common.ui.TextFieldX;
import org.lsst.ccs.subsystem.common.ui.UiConstants;
import org.lsst.ccs.subsystem.common.ui.UiUtilities;
import org.lsst.ccs.subsystem.common.ui.jas.CommandSender;
import org.lsst.ccs.subsystem.refrig.constants.CompConditions;
import org.lsst.ccs.subsystem.refrig.constants.CompLatches;
import org.lsst.ccs.subsystem.refrig.constants.CompSwConds;
import org.lsst.ccs.subsystem.refrig.constants.CompSwitches;
import org.lsst.ccs.subsystem.refrig.constants.CompTypes;
import org.lsst.ccs.subsystem.refrig.constants.CompressorState;
import org.lsst.ccs.subsystem.refrig.constants.ConditionState;
import org.lsst.ccs.subsystem.refrig.constants.LatchState;
import org.lsst.ccs.subsystem.refrig.constants.SwCondState;
import org.lsst.ccs.subsystem.refrig.constants.SwitchState;
import org.lsst.ccs.subsystem.refrig.data.ColdCompState;
import org.lsst.ccs.subsystem.refrig.data.CompState;
import org.lsst.ccs.subsystem.refrig.data.CryoCompState;

/**
 *  Implements the compressor control panel.
 *
 *  @author Owen Saxton
 */
public class CompressorControlPanel extends JPanel implements UiUtilities.ActionHandler {

    private static final int
        SW_TYPE_POWER = 0,
        SW_TYPE_VALVE_NC = 1,
        SW_TYPE_VALVE_NO = 2;

    private static final int[] switchTypes = new int[CompSwitches.NUM_SWITCHES];
    static {
        switchTypes[CompSwitches.SW_ENABLE]        = SW_TYPE_POWER;
        switchTypes[CompSwitches.SW_LIGHTS]        = SW_TYPE_POWER;
        switchTypes[CompSwitches.SW_HEATER]        = SW_TYPE_POWER;
        switchTypes[CompSwitches.SW_ORIFICE_VALVE] = SW_TYPE_VALVE_NC;
        switchTypes[CompSwitches.SW_COOLANT_VALVE] = SW_TYPE_VALVE_NO;
        switchTypes[CompSwitches.SW_BYPASS_VALVE]  = SW_TYPE_VALVE_NC;
        switchTypes[CompSwitches.SW_SURGE_HEATER]  = SW_TYPE_POWER;
        switchTypes[CompSwitches.SW_ENABLE_ALERTS] = SW_TYPE_POWER;
    }
    private static final int compStateWidth = UiUtilities.maxEnumLabelWidth(CompressorState.class);
    private static final int switchStateWidth = Math.max(UiUtilities.maxEnumLabelWidth(SwitchState.class),
                                                         UiUtilities.maxLabelWidth(new String[]{"OPEN", "SHUT"}, ""));
    private static final int latchStateWidth = UiUtilities.maxEnumLabelWidth(LatchState.class);
    private static final int conditionStateWidth = UiUtilities.maxEnumLabelWidth(ConditionState.class);
    private static final int swCondStateWidth = UiUtilities.maxEnumLabelWidth(SwCondState.class);
    private static final int horzInset = UiUtilities.getLabelSize("M").width / 2;
    private static final int vertInset = UiUtilities.getLabelSize("M").height / 4;

    private String comp;
    private final CommandSender sender;
    private final int type;
    private final Collection<Integer> validSwitches, validLatches, validConditions, validSwConds;

    private final JPanel headPanel = new JPanel();
    private JLabel compName, compState;

    private final JPanel switchPanel = UiUtilities.newBorderedPanel("Switches");
    private final JLabel[] switchStatus = new JLabel[CompSwitches.NUM_SWITCHES];
    private final ButtonGroup[] switchBG = new ButtonGroup[CompSwitches.NUM_SWITCHES];
    private final JRadioButton[] switchOffRB = new JRadioButton[CompSwitches.NUM_SWITCHES];
    private final JRadioButton[] switchOnRB = new JRadioButton[CompSwitches.NUM_SWITCHES];

    private final JPanel capacityPanel = UiUtilities.newBorderedPanel("Cold Capacity Control");
    private TextFieldX bypassValve;
    private TextFieldX vfdFrequency;

    private final JPanel coolantPanel = UiUtilities.newBorderedPanel("Coolant Control");
    private TextFieldX coolantValve;

    private final JPanel latchPanel = UiUtilities.newBorderedPanel("PLC Error Conditions");
    private final JLabel[] latchStatus = new JLabel[CompLatches.NUM_LATCHES];
    private JButton resetButton;

    private final JPanel conditionPanel = UiUtilities.newBorderedPanel("PLC Running Conditions");
    private final JLabel[] conditionStatus = new JLabel[CompConditions.NUM_CONDITIONS];

    private final JPanel swCondPanel = UiUtilities.newBorderedPanel("CCS Error Conditions");
    private final JLabel[] swCondStatus = new JLabel[CompSwConds.NUM_SW_CONDITIONS];

    private final UiUtilities uiUtils;

    public CompressorControlPanel(int type, CommandSender cons) {
        this.type = type;
        sender = cons;
        validSwitches = type == CompTypes.TYPE_COLD ? ColdCompState.SWITCHES : CryoCompState.SWITCHES;
        validLatches = type == CompTypes.TYPE_COLD ? ColdCompState.LATCHES : CryoCompState.LATCHES;
        validConditions = type == CompTypes.TYPE_COLD ? ColdCompState.CONDITIONS : CryoCompState.CONDITIONS;
        validSwConds = type == CompTypes.TYPE_COLD ? ColdCompState.SW_CONDITIONS : CryoCompState.SW_CONDITIONS;
        uiUtils = new UiUtilities(this);
        initComponents();
        disableSystem();
    }

    public void updateControlPanel(CompState rs) {
        comp = rs.getName();
        compName.setText(comp);
        compName.setEnabled(true);
        CompressorState cState = rs.getCompressorState();
        Color cColor = cState == CompressorState.OFFLINE ? UiConstants.BLUE :
                       cState == CompressorState.STOPPED ? Color.BLACK :
                       cState == CompressorState.RUNNING ? UiConstants.GREEN :
                       cState == CompressorState.WAITING ? UiConstants.YELLOW : UiConstants.RED;
        compState.setText(cState.name());
        compState.setForeground(cColor);
        compState.setEnabled(true);
        for (int sw : rs.getValidSwitches()) {
            SwitchState state = rs.getSwitchState(sw);
            String text = state.name();
            int swType = switchTypes[sw];
            if (swType != SW_TYPE_POWER) {
                text = state == SwitchState.ON ? swType == SW_TYPE_VALVE_NC ? "OPEN" : "SHUT" :
                       state == SwitchState.OFF ? swType == SW_TYPE_VALVE_NC ? "SHUT" : "OPEN" : text;
            }
            Color color = state == SwitchState.OFF ? swType == SW_TYPE_VALVE_NO ? UiConstants.GREEN : Color.black :
                          state == SwitchState.ON ? swType == SW_TYPE_VALVE_NO ? Color.black : UiConstants.GREEN : UiConstants.BLUE;
            switchStatus[sw].setText(text);
            switchStatus[sw].setForeground(color);
            switchStatus[sw].setEnabled(true);
            JRadioButton selButton = state == SwitchState.ON ? switchOnRB[sw] : switchOffRB[sw];
            selButton.setSelected(true);
            switchOffRB[sw].setEnabled(true);
            switchOnRB[sw].setEnabled(true);
        }
        if (type == CompTypes.TYPE_COLD) {
            bypassValve.update(String.format("%.1f", 100.0 * rs.getBypassValve()), true);
            vfdFrequency.update(rs.getVfdFrequency(), true);
            coolantValve.update(String.format("%.1f", 100.0 * rs.getCoolantValve()), true);
        }
        for (int cond : rs.getValidLatches()) {
            LatchState state = rs.getLatchState(cond);
            Color color = state == LatchState.OFFLINE ? UiConstants.BLUE :
                          state == LatchState.CLEAR ? UiConstants.GREEN :
                          state == LatchState.WARNING ? UiConstants.YELLOW :
                          state == LatchState.ACTIVE ? UiConstants.RED : UiConstants.PURPLE;
            latchStatus[cond].setText(state.name());
            latchStatus[cond].setForeground(color);
            latchStatus[cond].setEnabled(true);
        }
        resetButton.setEnabled(true);
        for (int cond : rs.getValidConditions()) {
            ConditionState state = rs.getConditionState(cond);
            Color color = state == ConditionState.OFF ? UiConstants.BLUE :
                          state == ConditionState.NO ? Color.BLACK : UiConstants.GREEN;
            conditionStatus[cond].setText(state.name());
            conditionStatus[cond].setForeground(color);
            conditionStatus[cond].setEnabled(true);
        }
        for (int cond : rs.getValidSwConditions()) {
            SwCondState state = rs.getSwConditionState(cond);
            Color color = state == SwCondState.OFFLINE ? UiConstants.BLUE :
                          state == SwCondState.CLEAR ? UiConstants.GREEN :
                          state == SwCondState.DLYPEND ? UiConstants.PURPLE : UiConstants.RED;
            if (swCondStatus[cond] == null) {
                System.out.println("Can't handle sw condition " + cond + " for compressor " + rs.getName());
                continue;
            }
            swCondStatus[cond].setText(state.name());
            swCondStatus[cond].setForeground(color);
            swCondStatus[cond].setEnabled(true);
        }
        repaint();
    }

    public final void disableSystem() {
        compName.setEnabled(false);
        compState.setEnabled(false);
        for (int sw : validSwitches) {
            switchStatus[sw].setEnabled(false);
            switchOffRB[sw].setEnabled(false);
            switchOnRB[sw].setEnabled(false);
        }
        if (type == CompTypes.TYPE_COLD) {
            bypassValve.setDisabled();
            vfdFrequency.setDisabled();
            coolantValve.setDisabled();
        }
        for (int cond : validLatches) {
            latchStatus[cond].setEnabled(false);
        }
        resetButton.setEnabled(false);
        for (int cond : validConditions) {
            conditionStatus[cond].setEnabled(false);
        }
        for (int cond : validSwConds) {
            swCondStatus[cond].setEnabled(false);
        }
        repaint();
    }

    private void initComponents() {

        // Heading fields
        compName = UiUtilities.newLabel("MM99", 0);
        compState = UiUtilities.newLabel("X", compStateWidth);

        // Switch states
        for (int sw : validSwitches) {
            switchStatus[sw] = UiUtilities.newLabel("XXX", switchStateWidth);
            int swType = switchTypes[sw];
            switchOffRB[sw] = uiUtils.newRadioButton(swType == SW_TYPE_POWER ? "Off" : swType == SW_TYPE_VALVE_NC ? "Shut" : "Open", "F" + sw);
            switchOnRB[sw] = uiUtils.newRadioButton(swType == SW_TYPE_POWER ? "On" : swType == SW_TYPE_VALVE_NC ? "Open" : "Shut", "T" + sw);
            ButtonGroup bg = switchBG[sw] = new ButtonGroup();
            bg.add(switchOffRB[sw]);
            bg.add(switchOnRB[sw]);
        }

        // Cold capacity & coolant control
        if (type == CompTypes.TYPE_COLD) {
            bypassValve = uiUtils.newTextFieldX("0000.00", "B", TextFieldX.TYPE_DOUBLE, true);
            vfdFrequency = uiUtils.newTextFieldX("00.0000", "F", TextFieldX.TYPE_DOUBLE, true);
            coolantValve = uiUtils.newTextFieldX("0000.00", "C", TextFieldX.TYPE_DOUBLE, true);
        }

        // Latched condition states
        for (int cond : validLatches) {
            latchStatus[cond] = UiUtilities.newLabel("XXX", latchStateWidth);
        }
        resetButton = uiUtils.newButton("Reset", "", true);

        // Condition states
        for (int cond : validConditions) {
            conditionStatus[cond] = UiUtilities.newLabel("XXX", conditionStateWidth);
        }

        // Software condition states
        for (int cond : validSwConds) {
            swCondStatus[cond] = UiUtilities.newLabel("XXX", swCondStateWidth);
        }

        // Lay out the head panel
        headPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbh = new GridBagConstraints();
        gbh.anchor = GridBagConstraints.WEST;
        gbh.gridx = 0;
        gbh.gridy = 0;
        headPanel.add(UiUtilities.newLabel("Compressor:  ", 0), gbh);
        gbh.gridx++;
        headPanel.add(compName, gbh);
        gbh.gridx++;
        gbh.insets.left = 30;
        headPanel.add(UiUtilities.newLabel("State:  ", 0), gbh);
        gbh.gridx++;
        gbh.insets.left = 0;
        headPanel.add(compState, gbh);
        
        // Lay out the switch panel
        GridBagConstraints gbs = new GridBagConstraints();
        gbs.anchor = GridBagConstraints.WEST;
        gbs.insets = new Insets(0, 0, vertInset, horzInset);
        gbs.gridy = -1;
        int ncols = 2, colm = 0;
        for (int sw : validSwitches) {
            int swType = switchTypes[sw];
            if (colm == 0) {
                gbs.gridx = 0;
                gbs.gridy++;
            }
            gbs.insets.top = gbs.gridy == 0 ? vertInset : 0;
            gbs.insets.left = colm == 0 ? horzInset : 40;
            switchPanel.add(UiUtilities.newLabel(CompSwitches.DESCS[sw] + ":", 0), gbs);
            gbs.insets.left = horzInset;
            gbs.gridx++;
            switchPanel.add(switchStatus[sw], gbs);
            gbs.gridx++;
            switchPanel.add(swType == SW_TYPE_VALVE_NO ? switchOnRB[sw] : switchOffRB[sw], gbs);
            gbs.gridx++;
            gbs.insets.left = 0;
            switchPanel.add(swType == SW_TYPE_VALVE_NO ? switchOffRB[sw] : switchOnRB[sw], gbs);
            gbs.gridx++;
            colm = (colm + 1) % ncols;
        }

        // Lay out the cold capacity control panel
        if (type == CompTypes.TYPE_COLD) {
            GridBagConstraints gbd = new GridBagConstraints();
            gbd.anchor = GridBagConstraints.WEST;
            gbd.insets = new Insets(vertInset, horzInset, vertInset, horzInset);
            gbd.gridx = 0;
            gbd.gridy = 0;
            capacityPanel.add(UiUtilities.newLabel("Bypass valve:", 0), gbd);
            gbd.gridx++;
            capacityPanel.add(bypassValve, gbd);
            gbd.gridx++;
            gbd.insets.left = 0;
            capacityPanel.add(UiUtilities.newLabel("%", 0), gbd);
            gbd.gridx++;
            gbd.insets.left = 30;
            capacityPanel.add(UiUtilities.newLabel("VFD frequency:", 0), gbd);
            gbd.gridx++;
            gbd.insets.left = horzInset;
            capacityPanel.add(vfdFrequency, gbd);
            gbd.gridx++;
            gbd.insets.left = 0;
            capacityPanel.add(UiUtilities.newLabel("Hz", 0), gbd);
        }

        // Lay out the cold coolant control panel
        if (type == CompTypes.TYPE_COLD) {
            GridBagConstraints gbf = new GridBagConstraints();
            gbf.anchor = GridBagConstraints.WEST;
            gbf.insets = new Insets(vertInset, horzInset, vertInset, horzInset);
            gbf.gridx = 0;
            gbf.gridy = 0;
            coolantPanel.add(UiUtilities.newLabel("Coolant valve:", 0), gbf);
            gbf.gridx++;
            coolantPanel.add(coolantValve, gbf);
            gbf.gridx++;
            gbf.insets.left = 0;
            coolantPanel.add(UiUtilities.newLabel("%", 0), gbf);
        }

        // Lay out the latched conditions panel
        GridBagConstraints gbl = new GridBagConstraints();
        gbl.anchor = GridBagConstraints.WEST;
        gbl.insets = new Insets(0, 0, vertInset, horzInset);
        gbl.gridy = -1;
        ncols = 1;
        colm = 0;
        for (int cond : validLatches) {
            if (colm == 0) {
                gbl.gridx = 0;
                gbl.gridy++;
            }
            gbl.insets.top = gbl.gridy == 0 ? vertInset : 0;
            gbl.insets.left = colm == 0 ? horzInset : 30;
            latchPanel.add(UiUtilities.newLabel(CompLatches.DESCS[cond] + ":", 0), gbl);
            gbl.insets.left = horzInset;
            gbl.gridx++;
            latchPanel.add(latchStatus[cond], gbl);
            gbl.gridx++;
            colm = (colm + 1) % ncols;
        }
        gbl.anchor = GridBagConstraints.NORTH;
        gbl.gridwidth = 2 * ncols;
        gbl.gridx = 0;
        gbl.gridy++;
        gbl.insets.top = vertInset;
        latchPanel.add(resetButton, gbl);

        // Lay out the conditions panel
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.anchor = GridBagConstraints.WEST;
        gbc.insets = new Insets(0, 0, vertInset, horzInset);
        gbc.gridy = -1;
        ncols = 2;
        colm = 0;
        for (int cond : validConditions) {
            if (colm == 0) {
                gbc.gridx = 0;
                gbc.gridy++;
            }
            gbc.insets.top = gbc.gridy == 0 ? vertInset : 0;
            gbc.insets.left = colm == 0 ? horzInset : 30;
            conditionPanel.add(UiUtilities.newLabel(CompConditions.DESCS[cond] + ":", 0), gbc);
            gbc.insets.left = horzInset; //0
            gbc.gridx++;
            conditionPanel.add(conditionStatus[cond], gbc);
            gbc.gridx++;
            colm = (colm + 1) % ncols;
        }

        // Lay out the software conditions panel
        GridBagConstraints gbw = new GridBagConstraints();
        gbw.anchor = GridBagConstraints.WEST;
        gbw.insets = new Insets(0, 0, vertInset, horzInset);
        gbw.gridy = -1;
        ncols = 2;
        colm = 0;
        for (int cond : validSwConds) {
            if (colm == 0) {
                gbw.gridx = 0;
                gbw.gridy++;
            }
            gbw.insets.top = gbw.gridy == 0 ? vertInset : 0;
            gbw.insets.left = colm == 0 ? horzInset : 30;
            swCondPanel.add(UiUtilities.newLabel(CompSwConds.DESCS[cond] + ":", 0), gbw);
            gbw.insets.left = horzInset; //0
            gbw.gridx++;
            swCondPanel.add(swCondStatus[cond], gbw);
            gbw.gridx++;
            colm = (colm + 1) % ncols;
        }

        // Lay out this panel
        setLayout(new GridBagLayout());
        GridBagConstraints gbm = new GridBagConstraints();
        gbm.insets = new Insets(12, 6, 6, 6);
        gbm.anchor = GridBagConstraints.NORTH;
        gbm.gridx = 0;
        gbm.gridy = 0;
        gbm.gridwidth = 2;
        add(headPanel, gbm);
        gbm.gridy++;
        add(switchPanel, gbm);
        gbm.gridy++;
        gbm.insets.top = 6;
        if (type == CompTypes.TYPE_COLD) {
            add(capacityPanel, gbm);
            gbm.gridy++;
            add(coolantPanel, gbm);
            gbm.gridy++;
        }
        gbm.gridwidth = 1;
        gbm.gridheight = 2;
        add(latchPanel, gbm);
        gbm.gridx++;
        gbm.gridheight = 1;
        add(conditionPanel, gbm);
        gbm.gridy++;
        add(swCondPanel, gbm);
    }

    @Override
    public void handleRadioButton(String name) {
        char action = name.charAt(0);
        int sw = Integer.valueOf(name.substring(1));
        sender.sendCommand(comp, "setSwitchOn", sw, action == 'T');
    }

    @Override
    public void handleTextFieldX(String name, Object value) {
        char action = name.charAt(0);
        if (action == 'B') {
            sender.sendCommand(comp, "setBypassValve", (Double)value / 100.0);
        }
        else if (action == 'F') {
            sender.sendCommand(comp, "setVfdFrequency", value);
        }
        else if (action == 'C') {
            sender.sendCommand(comp, "setCoolantValve", (Double)value / 100.0);
        }
    }

    @Override
    public void handleButton(String name) {
        sender.sendCommand(comp, "resetLatches");
    }

    private static final long serialVersionUID = 1L;
}
