package org.lsst.ccs.subsystem.refrig.ui;

import java.awt.Color;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.util.HashMap;
import java.util.Map;
import javax.swing.ButtonGroup;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.SwingUtilities;
import org.lsst.ccs.subsystem.common.ui.UiConstants;
import org.lsst.ccs.subsystem.common.ui.UiUtilities;
import org.lsst.ccs.subsystem.common.ui.jas.CommandSender;
import org.lsst.ccs.subsystem.refrig.constants.CmprSetState;
import org.lsst.ccs.subsystem.refrig.constants.HexCmprState;
import org.lsst.ccs.subsystem.refrig.constants.HexValves;
import org.lsst.ccs.subsystem.refrig.constants.ValvePosnState;
import org.lsst.ccs.subsystem.refrig.constants.ValveSetState;
import org.lsst.ccs.subsystem.refrig.data.HexValveState;

/**
 *  Implements the heat exchanger valve control panel.
 *
 *  @author Owen Saxton
 */
public class HexValveControlPanel extends JPanel implements CommandSender.ReplyHandler, UiUtilities.ActionHandler {

    private static final Map<ValveSetState, Color> setStateColors = new HashMap<>();
    static {
        setStateColors.put(ValveSetState.OPEN, UiConstants.GREEN);
        setStateColors.put(ValveSetState.SHUT, Color.BLACK);
        setStateColors.put(ValveSetState.OFFLINE, UiConstants.BLUE);
    }
    private static final Map<ValvePosnState, Color> posnStateColors = new HashMap<>();
    static {
        posnStateColors.put(ValvePosnState.OPEN, UiConstants.GREEN);
        posnStateColors.put(ValvePosnState.SHUT, Color.BLACK);
        posnStateColors.put(ValvePosnState.BOTH, UiConstants.RED);
        posnStateColors.put(ValvePosnState.NONE, UiConstants.YELLOW);
        posnStateColors.put(ValvePosnState.OFFLINE, UiConstants.BLUE);
    }
    private static final Map<HexCmprState, Color> hexStateColors = new HashMap<>();
    static {
        hexStateColors.put(HexCmprState.NORMAL, UiConstants.GREEN);
        hexStateColors.put(HexCmprState.SETERROR, UiConstants.RED);
        hexStateColors.put(HexCmprState.POSNERROR, UiConstants.PURPLE);
        hexStateColors.put(HexCmprState.OFFLINE, UiConstants.BLUE);
    }
    private static final String BLOCK = "\u2588";
    private static final int horzInset = UiUtilities.getLabelSize("M").width / 2;
    private static final int vertInset = UiUtilities.getLabelSize("M").height / 4;

    private final CommandSender sender;
    private final UiUtilities uiUtils;

    private JLabel sysStatus;
    private final JLabel[] valveSetStatus = new JLabel[HexValves.NUM_VALVES];
    private final JLabel[] valvePosnStatus = new JLabel[HexValves.NUM_VALVES];
    private final JRadioButton[] cmprButton = new JRadioButton[HexValves.NUM_CMPRS];
    private final ButtonGroup[] hexBG = new ButtonGroup[HexValves.NUM_HEXS];
    private final JLabel[] hexStatus = new JLabel[HexValves.NUM_HEXS];

    public HexValveControlPanel(String agent) {
        sender = new CommandSender(agent, this);
        uiUtils = new UiUtilities(this);
        initComponents();
        (new DisablePanel()).run();
    }

    public void initPanel() {
        sender.sendCommand(true, null, "getSystemState");
    }

    @Override
    public void onCommandReply(Object reply, String path, String command, Object[] args) {
        updatePanel((HexValveState)reply);
    }

    public void updatePanel(HexValveState vs) {
        SwingUtilities.invokeLater(new UpdateValveState(vs));
    }

    public void disablePanel() {
        SwingUtilities.invokeLater(new DisablePanel());
    }

    private void initComponents() {

        sysStatus = UiUtilities.newLabel("XXX", UiUtilities.maxLabelWidth(new String[]{"STOPPED", "RUNNING"}, ""));

        for (int hNum = 0; hNum < HexValves.NUM_HEXS; hNum++) {
            hexBG[hNum] = new ButtonGroup();
            for (int cNum : HexValves.hxCmprSets[hNum]) {
                cmprButton[cNum] = uiUtils.newRadioButton(HexValves.cmprNumberMap.get(cNum), hNum + "" + cNum);
                hexBG[hNum].add(cmprButton[cNum]);
            }
            hexStatus[hNum] = UiUtilities.newLabel("XXX", UiUtilities.maxEnumLabelWidth(HexCmprState.class));
        }
        for (int vNum = 0; vNum < HexValves.NUM_VALVES; vNum++) {
            valveSetStatus[vNum] = UiUtilities.newLabel(BLOCK, 0);
            valvePosnStatus[vNum] = UiUtilities.newLabel(BLOCK, 0);
        }

        JPanel mainPanel = UiUtilities.newBorderedPanel(null);
        GridBagConstraints gbm = new GridBagConstraints();
        gbm.insets = new Insets(vertInset, 2 * horzInset, vertInset, 2 * horzInset);
        gbm.gridx = 0;
        gbm.gridy = 0;
        gbm.anchor = GridBagConstraints.NORTH;
        gbm.gridwidth = 4;
        mainPanel.add(newSysPanel(), gbm);
        gbm.anchor = GridBagConstraints.NORTHWEST;
        gbm.gridwidth = 1;
        gbm.gridy++;
        gbm.insets.bottom = 2 * vertInset;
        for (int hNum = 0; hNum < HexValves.NUM_HEXS; hNum++) {
            mainPanel.add(newHexPanel(hNum), gbm);
            gbm.gridx++;
            for (int cNum : HexValves.hxCmprSets[hNum]) {
                mainPanel.add(newCmprPanel(cNum), gbm);
                gbm.gridx++;
            }
            gbm.gridy++;
            gbm.gridx = 0;
        }
        add(mainPanel);
    }

    private JPanel newSysPanel() {
        JPanel panel = new JPanel();
        panel.add(UiUtilities.newLabel("System State:  ", 0));
        panel.add(sysStatus);
        return panel;
    }

    private JPanel newHexPanel(int hNum) {
        JPanel panel = new JPanel();
        panel.setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = new Insets(0, 0, vertInset, 0);
        gbc.anchor = GridBagConstraints.NORTH;
        gbc.gridx = 0;
        gbc.gridy = 0;
        panel.add(UiUtilities.newLabel(HexValves.hxNumberMap.get(hNum) + ":", 0), gbc);
        gbc.gridy++;
        panel.add(hexStatus[hNum], gbc);
        return panel;
    }

    private JPanel newCmprPanel(int cNum) {
        JPanel panel = new JPanel();
        panel.setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = new Insets(0, 0, vertInset, 0);
        gbc.anchor = GridBagConstraints.NORTH;
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.gridwidth = 6;
        panel.add(cmprButton[cNum], gbc);
        gbc.gridy++;
        gbc.gridwidth = 2;
        panel.add(UiUtilities.newLabel("Liq", 0), gbc);
        gbc.gridx += 2;
        panel.add(UiUtilities.newLabel("Vap", 0), gbc);
        gbc.gridx += 2;
        panel.add(UiUtilities.newLabel("Rtn", 0), gbc);
        gbc.gridy++;
        gbc.gridx = 0;
        gbc.gridwidth = 1;
        for (int vNum : HexValves.cmprValveSets[cNum]) {
            panel.add(valveSetStatus[vNum], gbc);
            gbc.gridx++;
            gbc.insets.left = 0;
            panel.add(valvePosnStatus[vNum], gbc);
            gbc.gridx++;
            gbc.insets.left = horzInset;
        }
        return panel;
    }

    @Override
    public void handleRadioButton(String name) {
        int hNum = Integer.valueOf(name.substring(0, 1));
        int cNum = Integer.valueOf(name.substring(1, 2));
        sender.sendCommand(null, "connectHex", HexValves.hxNumberMap.get(hNum), HexValves.cmprNumberMap.get(cNum));
    }

    class UpdateValveState implements Runnable {

        private final HexValveState vs;

        UpdateValveState(HexValveState vs) {
            this.vs = vs;
        }

        @Override
        public void run() {
            sysStatus.setText("RUNNING");
            sysStatus.setForeground(UiConstants.GREEN);
            for (int hNum = 0; hNum < HexValves.NUM_HEXS; hNum++) {
                hexBG[hNum].clearSelection();
                for (int cNum : HexValves.hxCmprSets[hNum]) {
                    if (vs.getCmprState(cNum) == CmprSetState.OPEN) {
                        cmprButton[cNum].setSelected(true);
                    }
                    cmprButton[cNum].setEnabled(true);
                }
                HexCmprState hState = vs.getHexState(hNum);
                hexStatus[hNum].setText(hState.toString());
                hexStatus[hNum].setForeground(hexStateColors.get(hState));
                hexStatus[hNum].setEnabled(true);
            }
            for (int vNum = 0; vNum < HexValves.NUM_VALVES; vNum++) {
                valveSetStatus[vNum].setForeground(setStateColors.get(vs.getSetState(vNum)));
                valveSetStatus[vNum].setEnabled(true);
                valvePosnStatus[vNum].setForeground(posnStateColors.get(vs.getPosnState(vNum)));
                valvePosnStatus[vNum].setEnabled(true);
            } 
            repaint();
        }

    }

    class DisablePanel implements Runnable {

        @Override
        public void run() {
            sysStatus.setText("STOPPED");
            sysStatus.setForeground(UiConstants.RED);
            for (int hNum = 0; hNum < HexValves.NUM_HEXS; hNum++) {
                hexStatus[hNum].setEnabled(false);
            }
            for (int cNum = 0; cNum < HexValves.NUM_CMPRS; cNum++) {
                cmprButton[cNum].setEnabled(false);
            }
            for (int vNum = 0; vNum < HexValves.NUM_VALVES; vNum++) {
                valveSetStatus[vNum].setEnabled(false);
                valvePosnStatus[vNum].setEnabled(false);
            } 
            repaint();
        }
 
    }

    private static final long serialVersionUID = 1L;
}
