package org.lsst.ccs.subsystem.refrig.ui;

import java.awt.Color;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.util.HashMap;
import java.util.Map;
import javax.swing.ButtonGroup;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import org.lsst.ccs.subsystem.common.ui.TextFieldX;
import org.lsst.ccs.subsystem.common.ui.UiConstants;
import org.lsst.ccs.subsystem.common.ui.UiUtilities;
import org.lsst.ccs.subsystem.common.ui.jas.CommandSender;
import org.lsst.ccs.subsystem.refrig.constants.ThermalConstants;
import org.lsst.ccs.subsystem.refrig.constants.TrimHeaterState;
import org.lsst.ccs.subsystem.refrig.data.ThermalState;

/**
 *  Implements a heater sub-panel.
 *
 *  @author saxton
 */
public class TrimHeaterPanel extends JPanel implements UiUtilities.ActionHandler {

    private static final String[] plateNames = new String[ThermalState.NUM_TRIM_HEATERS];
    static {
        plateNames[ThermalState.TRIM_HEATER_COLD] = "Cold";
        plateNames[ThermalState.TRIM_HEATER_CRYO] = "Cryo";
    }
    private static final int labelWidth = UiUtilities.maxLabelWidth(plateNames, ":");
    private static final int stateWidth = UiUtilities.maxEnumLabelWidth(TrimHeaterState.class);
    private static final Map<TrimHeaterState, Color> stateColors = new HashMap<>();
    static {
        stateColors.put(TrimHeaterState.OFF, Color.BLACK);
        stateColors.put(TrimHeaterState.POWER, UiConstants.YELLOW);
        stateColors.put(TrimHeaterState.TEMP, UiConstants.GREEN);
        stateColors.put(TrimHeaterState.OFFLINE, UiConstants.BLUE);
        stateColors.put(TrimHeaterState.DISABLD, UiConstants.RED);
        stateColors.put(TrimHeaterState.VOLTERR, UiConstants.PURPLE);
        stateColors.put(TrimHeaterState.NOLOAD, UiConstants.PURPLE);
        stateColors.put(TrimHeaterState.VOLTERR_, UiConstants.PURPLE);
        stateColors.put(TrimHeaterState.NOLOAD_, UiConstants.PURPLE);
    }
    private static final int horzInset = UiUtilities.getLabelSize("M").width / 2;

    private final int id;
    private final CommandSender sender;
    private final UiUtilities uiUtils;

    private JLabel controlLabel, controlState, powerUnitsLabel, tempUnitsLabel;
    private ButtonGroup controlBG;
    private JRadioButton controlOffRB, controlPowerRB, controlTempRB;
    private TextFieldX powerTextField, tempTextField;

    TrimHeaterPanel(int id, CommandSender sender) {
        this.id = id;
        this.sender = sender;
        uiUtils = new UiUtilities(this);
        initComponents();
    }

    public void updatePanel(ThermalState rs) {
        controlState.setEnabled(true);
        TrimHeaterState tState = rs.getTrimHeaterState(id);
        controlState.setText(tState.name());
        Color color = stateColors.get(tState);
        color = color == null ? UiConstants.PURPLE : color;
        controlState.setForeground(color);

        boolean isPower = ThermalConstants.trimPowerStates.contains(tState);
        boolean isTemp = ThermalConstants.trimTempStates.contains(tState);
        JRadioButton selButton = isPower ? controlPowerRB : isTemp ? controlTempRB : controlOffRB;
        selButton.setSelected(true);
        boolean online = tState != TrimHeaterState.OFFLINE;
        controlOffRB.setEnabled(online && selButton != controlOffRB);
        controlPowerRB.setEnabled(online && selButton != controlPowerRB);
        controlTempRB.setEnabled(online && selButton != controlTempRB);

        powerTextField.update(rs.getTrimHeaterPower(id), true);
        tempTextField.update(rs.getPlateTemperature(id), true);
        repaint();
    }

    public void disablePanel() {
        controlState.setEnabled(false);
        controlOffRB.setEnabled(false);
        controlPowerRB.setEnabled(false);
        controlTempRB.setEnabled(false);
        powerTextField.setDisabled();
        tempTextField.setDisabled();
        repaint();
    }

    private void initComponents() {
        controlLabel = UiUtilities.newLabel(plateNames[id] + ":", labelWidth);
        controlState = UiUtilities.newLabel("OFF", stateWidth);
        controlOffRB = uiUtils.newRadioButton("Off", "O");
        controlPowerRB = uiUtils.newRadioButton("Power", "P");
        controlTempRB = uiUtils.newRadioButton("Temp", "T");
        controlBG = new ButtonGroup();
        controlBG.add(controlOffRB);
        controlBG.add(controlPowerRB);
        controlBG.add(controlTempRB);
        powerTextField = uiUtils.newTextFieldX("99999.9", "P", TextFieldXX.TYPE_DOUBLE, true);
        powerUnitsLabel = UiUtilities.newLabel("W", 0);
        tempTextField = uiUtils.newTextFieldX("-99999.9", "T", TextFieldXX.TYPE_DOUBLE, true);
        tempUnitsLabel = UiUtilities.newLabel("°C", 0);

        // Lay out the panel
        setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = new Insets(0, horzInset, 0, horzInset);
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.anchor = GridBagConstraints.WEST;

        // Add heater control to panel
        add(controlLabel, gbc);
        gbc.gridx++;
        add(controlState, gbc);
        gbc.gridx++;
        add(controlOffRB, gbc);
        gbc.gridx++;
        gbc.insets.right = 0;
        add(controlPowerRB, gbc);
        gbc.gridx++;
        add(powerTextField, gbc);
        gbc.gridx++;
        gbc.insets.right = horzInset;
        add(powerUnitsLabel, gbc);
        gbc.gridx++;
        gbc.insets.right = 0;
        add(controlTempRB, gbc);
        gbc.gridx++;
        add(tempTextField, gbc);
        gbc.gridx++;
        gbc.insets.right = horzInset;
        add(tempUnitsLabel, gbc);
    }

    @Override
    public void handleRadioButton(String name) {
        controlOffRB.setEnabled(false);
        controlPowerRB.setEnabled(false);
        controlTempRB.setEnabled(false);
        char btn = name.charAt(0);
        sender.sendCommand(null, "setTrimHeaterState", id, btn == 'P' ? 1 : btn == 'T' ? -1 : 0);
    }

    @Override
    public void handleTextFieldX(String name, Object value) {
        sender.sendCommand(null, name.charAt(0) == 'P' ? "setTrimHeaterPower" : "setPlateTemperature", id, (Double)value);
    }

    private static final long serialVersionUID = 1L;
}
