package org.lsst.ccs.subsystem.refrig.ui;

import java.awt.Color;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.util.HashMap;
import java.util.Map;
import javax.swing.ButtonGroup;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import org.lsst.ccs.subsystem.common.ui.TextFieldX;
import org.lsst.ccs.subsystem.common.ui.UiConstants;
import org.lsst.ccs.subsystem.common.ui.UiUtilities;
import org.lsst.ccs.subsystem.common.ui.jas.CommandSender;
import org.lsst.ccs.subsystem.refrig.constants.AuxHeaterState;
import org.lsst.ccs.subsystem.refrig.constants.ThermalConstants;
import org.lsst.ccs.subsystem.refrig.data.ThermalState;

/**
 *  Implements an auxiliary heater sub-panel.
 *
 *  @author saxton
 */
public class AuxHeaterPanel extends JPanel implements UiUtilities.ActionHandler {

    private static final String[] heaterNames = new String[ThermalState.NUM_AUX_HEATERS];
    static {
        heaterNames[ThermalState.AUX_HEATER_MINUS_Y] = "-Y";
        heaterNames[ThermalState.AUX_HEATER_PLUS_Y] = "+Y";
        heaterNames[ThermalState.AUX_HEATER_CENTER] = "Center";
    }
    private static final int switchLabelWidth = UiUtilities.maxLabelWidth(heaterNames, ":");
    private static final int switchStateWidth = UiUtilities.maxEnumLabelWidth(AuxHeaterState.class);
    private static final Map<AuxHeaterState, Color> stateColors = new HashMap<>();
    static {
        stateColors.put(AuxHeaterState.OFF, Color.BLACK);
        stateColors.put(AuxHeaterState.ON, UiConstants.GREEN);
        stateColors.put(AuxHeaterState.OFFLINE, UiConstants.BLUE);
        stateColors.put(AuxHeaterState.DISABLD, UiConstants.RED);
        stateColors.put(AuxHeaterState.VOLTERR, UiConstants.PURPLE);
        stateColors.put(AuxHeaterState.NOLOAD, UiConstants.PURPLE);
    }
    private static final int horzInset = UiUtilities.getLabelSize("M").width / 2;

    private final int id;
    private final CommandSender sender;
    private final UiUtilities uiUtils;

    private JLabel switchLabel, switchStatus, powerUnitsLabel;
    private ButtonGroup switchBG;
    private JRadioButton switchOffRB, switchOnRB;
    private TextFieldX powerTextField;


    AuxHeaterPanel(int id, CommandSender sender) {
        this.id = id;
        this.sender = sender;
        uiUtils = new UiUtilities(this);
        initComponents();
    }

    public void updatePanel(ThermalState rs) {
        switchStatus.setEnabled(true);
        AuxHeaterState pState = rs.getAuxHeaterState(id);
        switchStatus.setText(pState.name());
        Color color = stateColors.get(pState);
        color = color == null ? UiConstants.PURPLE : color;
        switchStatus.setForeground(color);

        boolean isOn = ThermalConstants.auxOnStates.contains(pState);
        JRadioButton selButton = isOn ? switchOnRB : switchOffRB;
        selButton.setSelected(true);
        switchOffRB.setEnabled(isOn);
        switchOnRB.setEnabled(pState == AuxHeaterState.OFF);

        powerTextField.update(rs.getAuxHeaterPower(id), true);
    }

    public void disablePanel() {
        switchStatus.setEnabled(false);
        switchOffRB.setEnabled(false);
        switchOnRB.setEnabled(false);
        powerTextField.setDisabled();
    }

    private void initComponents() {
        switchLabel = UiUtilities.newLabel(heaterNames[id] + ":", switchLabelWidth);
        switchStatus = UiUtilities.newLabel("OFF", switchStateWidth);
        switchOffRB = uiUtils.newRadioButton("Off", "F");
        switchOnRB = uiUtils.newRadioButton("On", "T");
        switchBG = new ButtonGroup();
        switchBG.add(switchOffRB);
        switchBG.add(switchOnRB);
        powerTextField = uiUtils.newTextFieldX("10000.0", "", TextFieldX.TYPE_DOUBLE, true);
        powerUnitsLabel = UiUtilities.newLabel("W", 0);

        // Lay out the panel
        setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = new Insets(0, horzInset, 0, horzInset);
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.anchor = GridBagConstraints.WEST;
        add(switchLabel, gbc);
        gbc.gridx++;
        add(switchStatus, gbc);
        gbc.gridx++;
        add(switchOffRB, gbc);
        gbc.gridx++;
        gbc.insets.right = 0;
        add(switchOnRB, gbc);
        gbc.gridx++;
        add(powerTextField, gbc);
        gbc.gridx++;
        gbc.insets.right = horzInset;
        add(powerUnitsLabel, gbc);
    }

    @Override
    public void handleRadioButton(String name) {
        switchOffRB.setEnabled(false);
        switchOnRB.setEnabled(false);
        sender.sendCommand(null, "setAuxHeaterState", id, name.charAt(0) == 'T' ? 1 : 0);
    }

    @Override
    public void handleTextFieldX(String name, Object value) {
        sender.sendCommand(null, "setAuxHeaterPower", id, (Double)value);
    }

    private static final long serialVersionUID = 1L;
}
