package org.lsst.ccs.subsystem.refrig.ui;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.util.HashMap;
import java.util.Map;
import javax.swing.ButtonGroup;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JTextField;
import javax.swing.SwingConstants;
import org.lsst.ccs.subsystem.common.ui.UiConstants;
import org.lsst.ccs.subsystem.common.ui.jas.CommandSender;
import org.lsst.ccs.subsystem.refrig.constants.ThermalConstants;
import org.lsst.ccs.subsystem.refrig.constants.TrimHeaterState;
import org.lsst.ccs.subsystem.refrig.data.ThermalState;

/**
 *  Implements a heater sub-panel.
 *
 *  @author saxton
 */
public class TrimHeaterPanel extends JPanel {

    private static final String[] plateNames = new String[ThermalState.NUM_TRIM_HEATERS];
    static {
        plateNames[ThermalState.TRIM_HEATER_COLD] = "Cold";
        plateNames[ThermalState.TRIM_HEATER_CRYO] = "Cryo";
    }
    private static final int labelWidth, stateWidth, powerFieldWidth, tempFieldWidth;
    static {
        JLabel label = new JLabel();
        label.setFont(UiConstants.FONT);
        int width = 0;
        for (String name : plateNames) {
            label.setText(name + ": ");
            width = Math.max(width, label.getPreferredSize().width);
        }
        labelWidth = width;
        width = 0;
        for (TrimHeaterState state : TrimHeaterState.values()) {
            label.setText(state.name());
            width = Math.max(width, label.getPreferredSize().width);
        }
        stateWidth = width;
        JTextField field = new JTextField(" 999.9 ");
        powerFieldWidth = field.getPreferredSize().width;
        field.setText(" -999.9 ");
        tempFieldWidth = field.getPreferredSize().width;
    }
    private static final Map<TrimHeaterState, Color> stateColors = new HashMap<>();
    static {
        stateColors.put(TrimHeaterState.OFF, Color.BLACK);
        stateColors.put(TrimHeaterState.POWER, UiConstants.YELLOW);
        stateColors.put(TrimHeaterState.TEMP, UiConstants.GREEN);
        stateColors.put(TrimHeaterState.OFFLINE, UiConstants.BLUE);
        stateColors.put(TrimHeaterState.DISABLD, UiConstants.RED);
        stateColors.put(TrimHeaterState.VOLTERR, UiConstants.PURPLE);
        stateColors.put(TrimHeaterState.NOLOAD, UiConstants.PURPLE);
        stateColors.put(TrimHeaterState.VOLTERR_, UiConstants.PURPLE);
        stateColors.put(TrimHeaterState.NOLOAD_, UiConstants.PURPLE);
    }

    private final int id;
    private final CommandSender sender;

    private final JLabel controlLabel = new JLabel();
    private final JLabel controlState = new JLabel("OFF");
    private final ButtonGroup controlBG = new ButtonGroup();
    private final JRadioButton controlOffRB = new JRadioButton("Off");
    private final JRadioButton controlPowerRB = new JRadioButton("Power");
    private final JRadioButton controlTempRB = new JRadioButton("Temp");

    private final JTextField powerTextField = new JTextField();
    private final TextFieldX powerTFX = new TextFieldX(powerTextField, TextFieldX.TYPE_DOUBLE);
    private final JLabel powerUnitsLabel = new JLabel("W");

    private final JTextField tempTextField = new JTextField();
    private final TextFieldX tempTFX = new TextFieldX(tempTextField, TextFieldX.TYPE_DOUBLE);
    private final JLabel tempUnitsLabel = new JLabel("°C");

    TrimHeaterPanel(int id, CommandSender sender) {
        this.id = id;
        this.sender = sender;
        initComponents();
    }

    public void updatePanel(ThermalState rs) {
        controlState.setEnabled(true);
        TrimHeaterState tState = rs.getTrimHeaterState(id);
        controlState.setText(tState.name());
        Color color = stateColors.get(tState);
        color = color == null ? UiConstants.PURPLE : color;
        controlState.setForeground(color);

        boolean isPower = ThermalConstants.trimPowerStates.contains(tState);
        boolean isTemp = ThermalConstants.trimTempStates.contains(tState);
        JRadioButton selButton = isPower ? controlPowerRB : isTemp ? controlTempRB : controlOffRB;
        selButton.setSelected(true);
        boolean online = tState != TrimHeaterState.OFFLINE;
        controlOffRB.setEnabled(online && selButton != controlOffRB);
        controlPowerRB.setEnabled(online && selButton != controlPowerRB);
        controlTempRB.setEnabled(online && selButton != controlTempRB);

        powerTFX.update(rs.getTrimHeaterPower(id), true);
        tempTFX.update(rs.getPlateTemperature(id), true);
        repaint();
    }

    public void disablePanel() {
        controlState.setEnabled(false);
        controlOffRB.setEnabled(false);
        controlPowerRB.setEnabled(false);
        controlTempRB.setEnabled(false);
        powerTFX.disable();
        tempTFX.disable();
        repaint();
    }

    private void initComponents() {
        controlLabel.setFont(UiConstants.FONT);
        controlLabel.setText(plateNames[id] + ": ");
        Dimension d = controlLabel.getPreferredSize();
        d.width = labelWidth;
        controlLabel.setPreferredSize(d);
        controlState.setFont(UiConstants.FONT);
        d = controlState.getPreferredSize();
        d.width = stateWidth;
        controlState.setPreferredSize(d);
        controlOffRB.setFont(UiConstants.FONT);
        controlOffRB.setFocusable(false);
        controlOffRB.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                setHeaterState(0);
            }
        });
        controlPowerRB.setFont(UiConstants.FONT);
        controlPowerRB.setFocusable(false);
        controlPowerRB.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                setHeaterState(1);
            }
        });
        controlTempRB.setFont(UiConstants.FONT);
        controlTempRB.setFocusable(false);
        controlTempRB.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                setHeaterState(-1);
            }
        });
        controlBG.add(controlOffRB);
        controlBG.add(controlPowerRB);
        controlBG.add(controlTempRB);

        d = powerTextField.getPreferredSize();
        d.width = powerFieldWidth;
        powerTextField.setPreferredSize(d);
        powerTextField.setHorizontalAlignment(SwingConstants.CENTER);
        powerTextField.setText("...");
        powerTextField.addFocusListener(new FocusAdapter() {
            @Override
            public void focusLost(FocusEvent evt) {
                powerTFX.focusLost();
            }
        });
        powerTextField.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                Double value = (Double)powerTFX.action();
                if (value != null) {
                    sender.sendCommand(null, "setTrimHeaterPower", id, value);
                }
            }
        });
        powerTextField.addKeyListener(new KeyAdapter() {
            @Override
            public void keyTyped(KeyEvent evt) {
                powerTFX.keyTyped(evt.getKeyChar());
            }
        });
        powerUnitsLabel.setFont(UiConstants.FONT);

        d = tempTextField.getPreferredSize();
        d.width = tempFieldWidth;
        tempTextField.setPreferredSize(d);
        tempTextField.setHorizontalAlignment(SwingConstants.CENTER);
        tempTextField.setText("...");
        tempTextField.addFocusListener(new FocusAdapter() {
            @Override
            public void focusLost(FocusEvent evt) {
                tempTFX.focusLost();
            }
        });
        tempTextField.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                Double value = (Double)tempTFX.action();
                if (value != null) {
                    sender.sendCommand(null, "setPlateTemperature", id, value);
                }
            }
        });
        tempTextField.addKeyListener(new KeyAdapter() {
            @Override
            public void keyTyped(KeyEvent evt) {
                tempTFX.keyTyped(evt.getKeyChar());
            }
        });
        tempUnitsLabel.setFont(UiConstants.FONT);

        // Lay out the panel
        setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = new Insets(0, 5, 0, 0);
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.anchor = GridBagConstraints.NORTHWEST;

        // Add heater control to panel
        add(controlLabel, gbc);
        gbc.gridx++;
        add(controlState, gbc);
        gbc.gridx++;
        gbc.insets.top = -4;
        add(controlOffRB, gbc);
        gbc.gridx++;
        add(controlPowerRB, gbc);
        gbc.gridx++;
        gbc.insets.top = 0;
        add(powerTextField, gbc);
        gbc.gridx++;
        add(powerUnitsLabel, gbc);
        gbc.gridx++;
        gbc.insets.top = -4;
        gbc.insets.left = 8;
        add(controlTempRB, gbc);
        gbc.gridx++;
        gbc.insets.top = 0;
        gbc.insets.left = 5;
        add(tempTextField, gbc);
        gbc.gridx++;
        gbc.insets.right = 5;
        add(tempUnitsLabel, gbc);
    }

    private void setHeaterState(int value) {
        controlOffRB.setEnabled(false);
        controlPowerRB.setEnabled(false);
        controlTempRB.setEnabled(false);
        sender.sendCommand(null, "setTrimHeaterState", id, value);
    }

    private static final long serialVersionUID = 1L;
}
