package org.lsst.ccs.subsystem.refrig.ui;

import java.awt.Color;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import javax.swing.JCheckBox;
import javax.swing.JPanel;
import javax.swing.SwingUtilities;
import javax.swing.border.LineBorder;
import javax.swing.border.TitledBorder;
import org.lsst.ccs.subsystem.common.ui.SystemStatusPanel;
import org.lsst.ccs.subsystem.common.ui.UiConstants;
import org.lsst.ccs.subsystem.common.ui.jas.CommandSender;
import org.lsst.ccs.subsystem.refrig.data.ThermalState;

/**
 *  Implements the thermal control panel.
 *
 *  @author Owen Saxton
 */
public class ThermalControlPanel extends JPanel {

    private static final String[] sectionNames = new String[ThermalState.NUM_COLD_SECTIONS];
    static {
        sectionNames[ThermalState.COLD_SECTION_MINUS_Y] = "-Y";
        sectionNames[ThermalState.COLD_SECTION_PLUS_Y] = "+Y";
    }

    private final CommandSender sender;
    private SystemStatusPanel statusPanel;
    private final TrimHeaterPanel[] trimHeaterPanels = new TrimHeaterPanel[ThermalState.NUM_TRIM_HEATERS];
    private final AuxHeaterPanel[] auxHeaterPanels = new AuxHeaterPanel[ThermalState.NUM_AUX_HEATERS];
    private final JPanel coldPanel = new JPanel();
    private final JCheckBox[] coldCbx = new JCheckBox[ThermalState.NUM_COLD_SECTIONS];
    private final JPanel trimPanel = new JPanel();
    private final JPanel auxPanel = new JPanel();
    private final JPanel mainPanel = new JPanel();

    public ThermalControlPanel(CommandSender sender) {
        this.sender = sender;
        initComponents();
        //(new UpdateThermalStatus(new ThermalState())).run();
        (new DisableSystem()).run();
    }

    public void updateControlPanel(ThermalState rs) {
        SwingUtilities.invokeLater(new UpdateThermalStatus(rs));
    }

    public void disableSystem() {
        SwingUtilities.invokeLater(new DisableSystem());
    }

    private void initComponents() {

        // Create status line panel
        statusPanel = new SystemStatusPanel(sender);

        // Create cold section line panel
        for (int sect = 0; sect < ThermalState.NUM_COLD_SECTIONS; sect++) {
            JCheckBox cbx = new JCheckBox("Enable " + sectionNames[sect] + " Cold Htrs");
            cbx.setFont(UiConstants.FONT);
            cbx.setFocusable(false);
            cbx.setName(String.valueOf(sect));
            cbx.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent e) {
                    int sect = Integer.valueOf(((JCheckBox)e.getSource()).getName());
                    sender.sendCommand(null, "enableColdSection", sect, coldCbx[sect].isSelected());
                }
            });
            coldCbx[sect] = cbx;
        }
        coldPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = new Insets(0, 0, 0, 0);
        gbc.anchor = GridBagConstraints.NORTHWEST;
        gbc.gridx = 0;
        gbc.gridy = 0;
        for (JCheckBox cbx : coldCbx) {
            coldPanel.add(cbx, gbc);
            gbc.gridx++;
            gbc.insets.left = 20;
        }
        
        // Create trim heater line panels
        for (int htr = 0; htr < ThermalState.NUM_TRIM_HEATERS; htr++) {
            trimHeaterPanels[htr] = new TrimHeaterPanel(htr, sender);
        }

        // Create trim heater control panel
        TitledBorder tBorder = new TitledBorder(new LineBorder(Color.BLACK), "Trim heater control");
        tBorder.setTitleFont(UiConstants.FONT);
        tBorder.setTitleColor(UiConstants.BLUE);
        tBorder.setTitleJustification(TitledBorder.CENTER);
        trimPanel.setBorder(tBorder);
        trimPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbt = new GridBagConstraints();
        gbt.insets = new Insets(5, 0, 5, 0);
        gbt.gridx = 0;
        gbt.gridy = 0;
        gbt.anchor = GridBagConstraints.NORTH;
        trimPanel.add(coldPanel, gbt);
        gbt.gridy++;
        gbt.insets.top = 0;
        for (JPanel pnl : trimHeaterPanels) {
            gbt.insets.bottom = gbt.gridy == ThermalState.NUM_TRIM_HEATERS ? 5 : 0;
            trimPanel.add(pnl, gbt);
            gbt.gridy++;
        }

        // Create auxiliary heater line control panels
        for (int htr = 0; htr < ThermalState.NUM_AUX_HEATERS; htr++) {
            auxHeaterPanels[htr] = new AuxHeaterPanel(htr, sender);
        }

        // Create auxiliary heater control panel
        TitledBorder aBorder = new TitledBorder(new LineBorder(Color.BLACK), "Auxiliary heater control");
        aBorder.setTitleFont(UiConstants.FONT);
        aBorder.setTitleColor(UiConstants.BLUE);
        aBorder.setTitleJustification(TitledBorder.CENTER);
        auxPanel.setBorder(aBorder);
        auxPanel.setLayout(new GridBagLayout());
        GridBagConstraints gba = new GridBagConstraints();
        gba.insets = new Insets(5, 0, 0, 0);
        gba.gridx = 0;
        gba.gridy = 0;
        gba.anchor = GridBagConstraints.NORTH;
        for (JPanel pnl : auxHeaterPanels) {
            gba.insets.bottom = gba.gridy == ThermalState.NUM_AUX_HEATERS - 1 ? 5 : 0;
            auxPanel.add(pnl, gba);
            gba.gridy++;
            gba.insets.top = 0;
        }
        
        // Add panels to main panel
        mainPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbm = new GridBagConstraints();
        gbm.insets = new Insets(5, 0, 0, 0);
        gbm.gridx = 0;
        gbm.gridy = 0;
        gbm.anchor = GridBagConstraints.NORTH;
        mainPanel.add(statusPanel, gbm);
        gbm.gridy++;
        gbm.insets.bottom = 5;
        mainPanel.add(trimPanel, gbm);
        gbm.gridy++;
        mainPanel.add(auxPanel, gbm);

        // Lay out the complete panel
        add(mainPanel);
    }

    class UpdateThermalStatus implements Runnable {

        private final ThermalState rs;

        UpdateThermalStatus(ThermalState rs) {
            this.rs = rs;
        }

        @Override
        public void run() {
            statusPanel.updateStatus(rs.getTickMillis());
            for (int sect = 0; sect < ThermalState.NUM_COLD_SECTIONS; sect++) {
                coldCbx[sect].setEnabled(true);
                coldCbx[sect].setSelected(rs.isColdSectionEnabled(sect));
            }
            for (TrimHeaterPanel pnl : trimHeaterPanels) {
                pnl.updatePanel(rs);
            }
            for (AuxHeaterPanel pnl : auxHeaterPanels) {
                pnl.updatePanel(rs);
            }
            repaint();
        }

    }

    class DisableSystem implements Runnable {

        @Override
        public void run() {
            statusPanel.disableSystem();
            for (JCheckBox cbx : coldCbx) {
                cbx.setEnabled(false);
            }
            for (TrimHeaterPanel pnl : trimHeaterPanels) {
                pnl.disablePanel();
            }
            for (AuxHeaterPanel pnl : auxHeaterPanels) {
                pnl.disablePanel();
            }
            repaint();
        }
 
    }

    private static final long serialVersionUID = 1L;
}
