package org.lsst.ccs.subsystem.refrig.ui;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.Collection;
import javax.swing.border.TitledBorder;
import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.border.LineBorder;
import org.lsst.ccs.subsystem.common.ui.UiConstants;
import org.lsst.ccs.subsystem.common.ui.jas.CommandSender;
import org.lsst.ccs.subsystem.refrig.constants.CompressorState;
import org.lsst.ccs.subsystem.refrig.constants.ConditionState;
import org.lsst.ccs.subsystem.refrig.constants.LatchState;
import org.lsst.ccs.subsystem.refrig.constants.SwCondState;
import org.lsst.ccs.subsystem.refrig.constants.SwitchState;
import org.lsst.ccs.subsystem.refrig.data.ColdCompState;
import org.lsst.ccs.subsystem.refrig.data.CompState;
import org.lsst.ccs.subsystem.refrig.data.CryoCompState;

/**
 *  Implements the compressor control panel.
 *
 *  @author Owen Saxton
 */
public class CompressorControlPanel extends JPanel {

    private static final int
        SW_TYPE_POWER = 0,
        SW_TYPE_VALVE_NC = 1,
        SW_TYPE_VALVE_NO = 2;

    private static final String[] switchNames = new String[CompState.NUM_SWITCHES];
    static {
        switchNames[CompState.SW_ENABLE]          = "Enable";
        switchNames[CompState.SW_LIGHTS]          = "Panel LEDs";
        switchNames[CompState.SW_HEATER]          = "Comp/Oil-Sep Htrs";
        switchNames[CompState.SW_ORIFICE_VALVE]   = "Orifice Valve (NC)";
        switchNames[CompState.SW_COOLANT_VALVE]   = "Coolant Valve (NO)";
        switchNames[CompState.SW_BYPASS_VALVE]    = "Bypass Valve (NC)";
        switchNames[CompState.SW_SURGE_HEATER]    = "Surge Tank Heater";
    }
    private static final int[] switchTypes = new int[CompState.NUM_SWITCHES];
    static {
        switchTypes[CompState.SW_ENABLE]          = SW_TYPE_POWER;
        switchTypes[CompState.SW_LIGHTS]          = SW_TYPE_POWER;
        switchTypes[CompState.SW_HEATER]          = SW_TYPE_POWER;
        switchTypes[CompState.SW_ORIFICE_VALVE]   = SW_TYPE_VALVE_NC;
        switchTypes[CompState.SW_COOLANT_VALVE]   = SW_TYPE_VALVE_NO;
        switchTypes[CompState.SW_BYPASS_VALVE]    = SW_TYPE_VALVE_NC;
        switchTypes[CompState.SW_SURGE_HEATER]    = SW_TYPE_POWER;
    }
    private static final String[] latchNames = new String[CompState.NUM_LATCHES];
    static {
        latchNames[CompState.LATCH_DISCHARGE_PRESS] = "Discharge Press";
        latchNames[CompState.LATCH_DISCHARGE_TEMP]  = "Discharge Temp";
        latchNames[CompState.LATCH_EXT_PERMIT]      = "External Permit";
        latchNames[CompState.LATCH_LIQUID_TEMP]     = "Liquid Temp";
        latchNames[CompState.LATCH_POWER]           = "Compressr Power";
        latchNames[CompState.LATCH_SENSORS_VALID]   = "Sensor Readings";
        latchNames[CompState.LATCH_SMOKE_DETC]      = "Smoke Detector";
        latchNames[CompState.LATCH_SUCTION_TEMP]    = "Suction Temp";
        latchNames[CompState.LATCH_AFTER_COOLER]    = "After Cooler Temp";
        latchNames[CompState.LATCH_OIL_LEVEL]       = "Oil Level";
    }
    private static final String[] conditionNames = new String[CompState.NUM_CONDITIONS];
    static {
        conditionNames[CompState.COND_CMP_ENABLED]      = "Compr enabled";
        conditionNames[CompState.COND_CMP_WAITING]      = "Compr waiting";
        conditionNames[CompState.COND_CMP_POWERED]      = "Compr powered";
        conditionNames[CompState.COND_CURRENT_VALID]    = "Current valid";
        conditionNames[CompState.COND_DISC_PRESS_VALID] = "Disch press valid";
        conditionNames[CompState.COND_DISC_TEMP_VALID]  = "Disch temp valid";
        conditionNames[CompState.COND_KEYSWITCH_ON]     = "Keyswitch on";
        conditionNames[CompState.COND_LATCHES_CLEAR]    = "Latches clear";
        conditionNames[CompState.COND_LIQD_TEMP_VALID]  = "Liquid temp valid";
        conditionNames[CompState.COND_POWER_LED]        = "Power LED On";
        conditionNames[CompState.COND_SUCT_PRESS_VALID] = "Suctn press valid";
        conditionNames[CompState.COND_SUCT_TEMP_VALID]  = "Suctn temp valid";
        conditionNames[CompState.COND_VOLTAGE_VALID]    = "Voltage valid";
        conditionNames[CompState.COND_CURR_SENSOR_ERR]  = "Current sens err";
        conditionNames[CompState.COND_CMP_ON_8HRS]      = "Compr on 8 hrs";
        conditionNames[CompState.COND_OIL_LEVEL_VALID]  = "Oil level valid";
    }
    private static final String[] swCondNames = new String[CompState.NUM_SW_CONDITIONS];
    static {
        swCondNames[CompState.SWC_CMPR_POWER]     = "Compr power";
        swCondNames[CompState.SWC_DISC_PRESS]     = "Disch press";
        swCondNames[CompState.SWC_DISC_TEMP]      = "Disch temp";
        swCondNames[CompState.SWC_LIQUID_TEMP]    = "Liquid temp";
        swCondNames[CompState.SWC_OIL_LEVEL]      = "Oil Level";
        swCondNames[CompState.SWC_PHASE_SEP_TEMP] = "Phase sep temp";
        swCondNames[CompState.SWC_SUCT_TEMP]      = "Suction temp";
        swCondNames[CompState.SWC_PLATE_TEMP]     = "Plate temp";
    }
    private static final int compStateWidth, switchStateWidth, latchStateWidth, conditionStateWidth, swCondStateWidth;
    static {
        JLabel label = new JLabel();
        label.setFont(UiConstants.FONT);
        int width = 0;
        for (CompressorState state : CompressorState.values()) {
            label.setText(state.name());
            width = Math.max(width, label.getPreferredSize().width);
        }
        compStateWidth = width;
        width = 0;
        for (SwitchState state : SwitchState.values()) {
            label.setText(state.name());
            width = Math.max(width, label.getPreferredSize().width);
            label.setText("OPEN");
            width = Math.max(width, label.getPreferredSize().width);
            label.setText("SHUT");
            width = Math.max(width, label.getPreferredSize().width);
        }
        switchStateWidth = width;
        width = 0;
        for (LatchState state : LatchState.values()) {
            label.setText(state.name());
            width = Math.max(width, label.getPreferredSize().width);
        }
        latchStateWidth = width;
        width = 0;
        for (ConditionState state : ConditionState.values()) {
            label.setText(state.name());
            width = Math.max(width, label.getPreferredSize().width);
        }
        conditionStateWidth = width;
        width = 0;
        for (SwCondState state : SwCondState.values()) {
            label.setText(state.name());
            width = Math.max(width, label.getPreferredSize().width);
        }
        swCondStateWidth = width;
    }

    private String comp;
    private final CommandSender sender;
    private final Collection<Integer> validSwitches, validLatches, validConditions, validSwConds;

    private final JPanel headPanel = new JPanel();
    private final JLabel compLabel = new JLabel("Compressor:  ");
    private final JLabel compName = new JLabel("MM99");
    private final JLabel stateLabel = new JLabel("State:  ");
    private final JLabel compState = new JLabel("STOPPED");

    private final JPanel switchPanel = new JPanel();
    private final JLabel[] switchLabel = new JLabel[CompState.NUM_SWITCHES];
    private final JLabel[] switchStatus = new JLabel[CompState.NUM_SWITCHES];
    private final ButtonGroup[] switchBG = new ButtonGroup[CompState.NUM_SWITCHES];
    private final JRadioButton[] switchOffRB = new JRadioButton[CompState.NUM_SWITCHES];
    private final JRadioButton[] switchOnRB = new JRadioButton[CompState.NUM_SWITCHES];

    private final JPanel latchPanel = new JPanel();
    private final JLabel[] latchLabel = new JLabel[CompState.NUM_LATCHES];
    private final JLabel[] latchStatus = new JLabel[CompState.NUM_LATCHES];
    private final JButton resetButton = new JButton("Reset");

    private final JPanel conditionPanel = new JPanel();
    private final JLabel[] conditionLabel = new JLabel[CompState.NUM_CONDITIONS];
    private final JLabel[] conditionStatus = new JLabel[CompState.NUM_CONDITIONS];

    private final JPanel swCondPanel = new JPanel();
    private final JLabel[] swCondLabel = new JLabel[CompState.NUM_SW_CONDITIONS];
    private final JLabel[] swCondStatus = new JLabel[CompState.NUM_SW_CONDITIONS];

    public CompressorControlPanel(int type, CommandSender cons) {
        sender = cons;
        validSwitches = type == CompState.TYPE_COLD ? ColdCompState.SWITCHES : CryoCompState.SWITCHES;
        validLatches = type == CompState.TYPE_COLD ? ColdCompState.LATCHES : CryoCompState.LATCHES;
        validConditions = type == CompState.TYPE_COLD ? ColdCompState.CONDITIONS : CryoCompState.CONDITIONS;
        validSwConds = type == CompState.TYPE_COLD ? ColdCompState.SW_CONDITIONS : CryoCompState.SW_CONDITIONS;
        initComponents();
        disableSystem();
    }

    public void updateControlPanel(CompState rs) {
        comp = rs.getName();
        compName.setText(comp);
        compName.setEnabled(true);
        CompressorState cState = rs.getCompressorState();
        Color cColor = cState == CompressorState.OFFLINE ? UiConstants.BLUE :
                       cState == CompressorState.STOPPED ? Color.BLACK :
                       cState == CompressorState.RUNNING ? UiConstants.GREEN :
                       cState == CompressorState.WAITING ? UiConstants.YELLOW : UiConstants.RED;
        compState.setText(cState.name());
        compState.setForeground(cColor);
        compState.setEnabled(true);
        for (int sw : rs.getValidSwitches()) {
            SwitchState state = rs.getSwitchState(sw);
            String text = state.name();
            int type = switchTypes[sw];
            if (type != SW_TYPE_POWER) {
                text = state == SwitchState.ON ? type == SW_TYPE_VALVE_NC ? "OPEN" : "SHUT" :
                       state == SwitchState.OFF ? type == SW_TYPE_VALVE_NC ? "SHUT" : "OPEN" : text;
            }
            Color color = state == SwitchState.OFF ? type == SW_TYPE_VALVE_NO ? UiConstants.GREEN : Color.black :
                          state == SwitchState.ON ? type == SW_TYPE_VALVE_NO ? Color.black : UiConstants.GREEN : UiConstants.BLUE;
            switchStatus[sw].setText(text);
            switchStatus[sw].setForeground(color);
            switchStatus[sw].setEnabled(true);
            JRadioButton selButton = state == SwitchState.ON ? switchOnRB[sw] : switchOffRB[sw];
            selButton.setSelected(true);
            switchOffRB[sw].setEnabled(true);
            switchOnRB[sw].setEnabled(true);
        }
        for (int cond : rs.getValidLatches()) {
            LatchState state = rs.getLatchState(cond);
            Color color = state == LatchState.OFFLINE ? UiConstants.BLUE :
                          state == LatchState.CLEAR ? UiConstants.GREEN :
                          state == LatchState.WARNING ? UiConstants.YELLOW :
                          state == LatchState.ACTIVE ? UiConstants.RED : UiConstants.PURPLE;
            latchStatus[cond].setText(state.name());
            latchStatus[cond].setForeground(color);
            latchStatus[cond].setEnabled(true);
        }
        resetButton.setEnabled(true);
        for (int cond : rs.getValidConditions()) {
            ConditionState state = rs.getConditionState(cond);
            Color color = state == ConditionState.OFF ? UiConstants.BLUE :
                          state == ConditionState.NO ? Color.BLACK : UiConstants.GREEN;
            conditionStatus[cond].setText(state.name());
            conditionStatus[cond].setForeground(color);
            conditionStatus[cond].setEnabled(true);
        }
        for (int cond : rs.getValidSwConditions()) {
            SwCondState state = rs.getSwConditionState(cond);
            Color color = state == SwCondState.OFFLINE ? UiConstants.BLUE :
                          state == SwCondState.CLEAR ? UiConstants.GREEN :
                          state == SwCondState.DLYPEND ? UiConstants.PURPLE : UiConstants.RED;
            if (swCondStatus[cond] == null) {
                System.out.println("Can't handle sw condition " + cond + " for compressor " + rs.getName());
                continue;
            }
            swCondStatus[cond].setText(state.name());
            swCondStatus[cond].setForeground(color);
            swCondStatus[cond].setEnabled(true);
        }
        repaint();
    }

    public final void disableSystem() {
        compName.setEnabled(false);
        compState.setEnabled(false);
        for (int sw : validSwitches) {
            switchStatus[sw].setEnabled(false);
            switchOffRB[sw].setEnabled(false);
            switchOnRB[sw].setEnabled(false);
        }
        for (int cond : validLatches) {
            latchStatus[cond].setEnabled(false);
        }
        resetButton.setEnabled(false);
        for (int cond : validConditions) {
            conditionStatus[cond].setEnabled(false);
        }
        for (int cond : validSwConds) {
            swCondStatus[cond].setEnabled(false);
        }
        repaint();
    }

    private void initComponents() {

        // Heading fields
        compLabel.setFont(UiConstants.FONT);
        compName.setFont(UiConstants.FONT);
        stateLabel.setFont(UiConstants.FONT);
        compState.setFont(UiConstants.FONT);
        Dimension di = compState.getPreferredSize();
        di.width = compStateWidth;
        compState.setPreferredSize(di);

        // Switch states
        for (int sw : validSwitches) {
            JLabel label = switchLabel[sw] = new JLabel(switchNames[sw] + ":");
            label.setFont(UiConstants.FONT);
            label = switchStatus[sw] = new JLabel("XXX");
            label.setFont(UiConstants.FONT);
            Dimension d = label.getPreferredSize();
            d.width = switchStateWidth;
            label.setPreferredSize(d);
            int type = switchTypes[sw];
            JRadioButton buttonOff = new JRadioButton(type == SW_TYPE_POWER ? "Off" : type == SW_TYPE_VALVE_NC ? "Shut" : "Open");
            buttonOff.setFont(UiConstants.FONT);
            buttonOff.setFocusable(false);
            buttonOff.setName(Integer.toString(sw));
            buttonOff.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent evt) {
                    String name = ((JRadioButton)evt.getSource()).getName();
                    setSwitchOn(Integer.valueOf(name), false);
                }
            });
            switchOffRB[sw] = buttonOff;
            JRadioButton buttonOn = new JRadioButton(type == SW_TYPE_POWER ? "On" : type == SW_TYPE_VALVE_NC ? "Open" : "Shut");
            buttonOn.setFont(UiConstants.FONT);
            //buttonOn.setText("On");
            buttonOn.setFocusable(false);
            buttonOn.setName(Integer.toString(sw));
            buttonOn.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent evt) {
                    String name = ((JRadioButton)evt.getSource()).getName();
                    setSwitchOn(Integer.valueOf(name), true);
                }
            });
            switchOnRB[sw] = buttonOn;
            ButtonGroup bg = switchBG[sw] = new ButtonGroup();
            bg.add(buttonOff);
            bg.add(buttonOn);
        }

        // Latched condition states
        for (int cond : validLatches) {
            JLabel label = latchLabel[cond] = new JLabel(latchNames[cond] + ":");
            label.setFont(UiConstants.FONT);
            label = latchStatus[cond] = new JLabel("XXX");
            label.setFont(UiConstants.FONT);
            Dimension d = label.getPreferredSize();
            d.width = latchStateWidth;
            label.setPreferredSize(d);
        }

        JButton button = resetButton;
        button.setFont(UiConstants.FONT);
        Insets i = button.getMargin();
        i.left -= 3;
        i.right -= 3;
        button.setMargin(i);
        Dimension dr = button.getPreferredSize();
        dr.height -= 9;
        button.setPreferredSize(dr);
        button.setFocusable(false);
        button.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                String name = ((JButton)evt.getSource()).getName();
                resetLatches();
            }
        });

        // Condition states
        for (int cond : validConditions) {
            JLabel label = conditionLabel[cond] = new JLabel(conditionNames[cond] + ":");
            label.setFont(UiConstants.FONT);
            label = conditionStatus[cond] = new JLabel("XXX");
            label.setFont(UiConstants.FONT);
            Dimension d = label.getPreferredSize();
            d.width = conditionStateWidth;
            label.setPreferredSize(d);
        }

        // Software condition states
        for (int cond : validSwConds) {
            JLabel label = swCondLabel[cond] = new JLabel(swCondNames[cond] + ":");
            label.setFont(UiConstants.FONT);
            label = swCondStatus[cond] = new JLabel("XXX");
            label.setFont(UiConstants.FONT);
            Dimension d = label.getPreferredSize();
            d.width = swCondStateWidth;
            label.setPreferredSize(d);
        }

        // Lay out the head panel
        headPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbh = new GridBagConstraints();
        gbh.anchor = GridBagConstraints.NORTHWEST;
        gbh.gridx = 0;
        gbh.gridy = 0;
        headPanel.add(compLabel, gbh);
        gbh.gridx++;
        headPanel.add(compName, gbh);
        gbh.gridx++;
        gbh.insets.left = 30;
        headPanel.add(stateLabel, gbh);
        gbh.gridx++;
        gbh.insets.left = 0;
        headPanel.add(compState, gbh);
        
        // Lay out the switch panel
        TitledBorder border = new TitledBorder(new LineBorder(Color.BLACK), "Switches");
        border.setTitleFont(UiConstants.FONT);
        border.setTitleJustification(TitledBorder.CENTER);
        border.setTitleColor(UiConstants.BLUE);
        switchPanel.setBorder(border);
        switchPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbs = new GridBagConstraints();
        gbs.anchor = GridBagConstraints.NORTHWEST;
        gbs.gridy = 0;
        //gbs.insets.bottom = 4;
        int ncols = 2, colm = 0;
        for (int sw : validSwitches) {
            int type = switchTypes[sw];
            if (colm == 0) {
                gbs.insets.top = gbs.gridy == 0 ? 4 : 0;
                gbs.gridx = 0;
                gbs.gridy++;
            }
            //gbs.insets.top = 0;
            gbs.insets.left = colm == 0 ? 4 : 40;
            switchPanel.add(switchLabel[sw], gbs);
            gbs.insets.left = 6;
            gbs.gridx++;
            switchPanel.add(switchStatus[sw], gbs);
            gbs.gridx++;
            gbs.insets.top -= 4;
            //gbs.insets.left = 0;
            switchPanel.add(type == SW_TYPE_VALVE_NO ? switchOnRB[sw] : switchOffRB[sw], gbs);
            gbs.gridx++;
            gbs.insets.left = 0;
            gbs.insets.right = 3;
            switchPanel.add(type == SW_TYPE_VALVE_NO ? switchOffRB[sw] : switchOnRB[sw], gbs);
            gbs.insets.right = 0;
            gbs.insets.top += 4;
            gbs.gridx++;
            colm = (colm + 1) % ncols;
        }

        // Lay out the latched conditions panel
        border = new TitledBorder(new LineBorder(Color.BLACK), "PLC Error Conditions");
        border.setTitleFont(UiConstants.FONT);
        border.setTitleJustification(TitledBorder.CENTER);
        border.setTitleColor(UiConstants.BLUE);
        latchPanel.setBorder(border);
        latchPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbl = new GridBagConstraints();
        gbl.anchor = GridBagConstraints.NORTHWEST;
        gbl.gridy = 0;
        gbl.insets.bottom = 4;
        gbl.insets.right = 4;
        ncols = 1;
        colm = 0;
        for (int cond : validLatches) {
            if (colm == 0) {
                gbl.insets.top = gbl.gridy == 0 ? 4 : 0;
                gbl.gridx = 0;
                gbl.gridy++;
            }
            gbl.insets.left = colm == 0 ? 4 : 30;
            latchPanel.add(latchLabel[cond], gbl);
            gbl.insets.left = 6;
            gbl.gridx++;
            latchPanel.add(latchStatus[cond], gbl);
            gbl.gridx++;
            colm = (colm + 1) % ncols;
        }
        gbl.anchor = GridBagConstraints.NORTH;
        gbl.gridwidth = 2 * ncols;
        gbl.gridx = 0;
        gbl.gridy++;
        gbl.insets.top = 4;
        latchPanel.add(resetButton, gbl);

        // Lay out the conditions panel
        border = new TitledBorder(new LineBorder(Color.BLACK), "PLC Running Conditions");
        border.setTitleFont(UiConstants.FONT);
        border.setTitleJustification(TitledBorder.CENTER);
        border.setTitleColor(UiConstants.BLUE);
        conditionPanel.setBorder(border);
        conditionPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.anchor = GridBagConstraints.NORTHWEST;
        gbc.gridy = 0;
        gbc.insets.bottom = 4;
        gbc.insets.right = 4;
        ncols = 2;
        colm = 0;
        for (int cond : validConditions) {
            if (colm == 0) {
                gbc.insets.top = gbc.gridy == 0 ? 4 : 0;
                gbc.gridx = 0;
                gbc.gridy++;
            }
            gbc.insets.left = colm == 0 ? 4 : 30;
            conditionPanel.add(conditionLabel[cond], gbc);
            gbc.insets.left = 6; //0
            gbc.gridx++;
            conditionPanel.add(conditionStatus[cond], gbc);
            gbc.gridx++;
            colm = (colm + 1) % ncols;
        }

        // Lay out the software conditions panel
        border = new TitledBorder(new LineBorder(Color.BLACK), "CCS Error Conditions");
        border.setTitleFont(UiConstants.FONT);
        border.setTitleJustification(TitledBorder.CENTER);
        border.setTitleColor(UiConstants.BLUE);
        swCondPanel.setBorder(border);
        swCondPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbw = new GridBagConstraints();
        gbw.anchor = GridBagConstraints.NORTHWEST;
        gbw.gridy = 0;
        gbw.insets.bottom = 4;
        gbw.insets.right = 4;
        ncols = 2;
        colm = 0;
        for (int cond : validSwConds) {
            if (colm == 0) {
                gbw.insets.top = gbw.gridy == 0 ? 4 : 0;
                gbw.gridx = 0;
                gbw.gridy++;
            }
            gbw.insets.left = colm == 0 ? 4 : 30;
            swCondPanel.add(swCondLabel[cond], gbw);
            gbw.insets.left = 6; //0
            gbw.gridx++;
            swCondPanel.add(swCondStatus[cond], gbw);
            gbw.gridx++;
            colm = (colm + 1) % ncols;
        }

        // Lay out this panel
        setLayout(new GridBagLayout());
        GridBagConstraints gbm = new GridBagConstraints();
        gbm.insets = new Insets(12, 6, 6, 6);
        gbm.anchor = GridBagConstraints.NORTH;
        gbm.gridx = 0;
        gbm.gridy = 0;
        gbm.gridwidth = 2;
        add(headPanel, gbm);
        gbm.gridy++;
        add(switchPanel, gbm);
        gbm.gridy++;
        gbm.gridwidth = 1;
        gbm.gridheight = 2;
        gbm.insets.top = 6;
        //gbm.insets.left = 6;
        //gbm.insets.right = 6;
        add(latchPanel, gbm);
        gbm.gridx++;
        gbm.gridheight = 1;
        add(conditionPanel, gbm);
        gbm.gridy++;
        add(swCondPanel, gbm);
    }

    private void setSwitchOn(int sw, boolean value) {
        sender.sendCommand(comp, "setSwitchOn", sw, value);
    }
    
    private void resetLatches() {
        sender.sendCommand(comp, "resetLatches");
    }

    private static final long serialVersionUID = 1L;
}
