package org.lsst.ccs.subsystem.refrig.ui;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dialog;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.SwingUtilities;
import org.lsst.ccs.subsystem.common.ui.UiConstants;
import org.lsst.ccs.subsystem.common.ui.jas.CommandSender;
import org.lsst.ccs.subsystem.refrig.constants.CompressorState;
import org.lsst.ccs.subsystem.refrig.constants.SwitchState;
import org.lsst.ccs.subsystem.refrig.data.CompState;

/**
 *  Implements the compressor summary line sub-panel.
 *
 *  @author Owen Saxton
 */
public class CompressorSummaryPanel extends JPanel {

    private static final int stateWidth;
    static {
        JLabel label = new JLabel();
        label.setFont(UiConstants.FONT);
        int width = 0;
        for (CompressorState state : CompressorState.values()) {
            label.setText(state.name());
            width = Math.max(width, label.getPreferredSize().width);
        }
        stateWidth = width;
    }

    private String comp;
    private final CommandSender sender;
    private final CompressorControlPanel control;
    private JDialog dialog = null;

    private final JLabel nameLabel = new JLabel("Comp:  ");
    private final JLabel nameValue = new JLabel("Cryo6");
    private final JLabel stateLabel = new JLabel("State:  ");
    private final JLabel stateValue = new JLabel("X");
    private final JLabel switchLabel = new JLabel("Enable:  ");
    private final ButtonGroup switchBG = new ButtonGroup();
    private final JRadioButton switchOffRB = new JRadioButton("Off");
    private final JRadioButton switchOnRB = new JRadioButton("On");
    private final JButton controlButton = new JButton("Full Control...");

    public CompressorSummaryPanel(int type, CommandSender cons) {
        this.sender = cons;
        initComponents();
        control = new CompressorControlPanel(type, cons);
        disableSystem();
    }

    public void updateControlPanel(CompState rs) {
        comp = rs.getName();
        nameValue.setText(comp);
        CompressorState cState = rs.getCompressorState();
        Color cColor = cState == CompressorState.OFFLINE ? UiConstants.BLUE :
                       cState == CompressorState.STOPPED ? Color.BLACK :
                       cState == CompressorState.RUNNING ? UiConstants.GREEN :
                       cState == CompressorState.WAITING ? UiConstants.YELLOW : UiConstants.RED;
        stateValue.setText(cState.name());
        stateValue.setForeground(cColor);
        SwitchState state = rs.getSwitchState(CompState.SW_ENABLE);
        JRadioButton selButton = state == SwitchState.ON ? switchOnRB : switchOffRB;
        selButton.setSelected(true);
        nameValue.setEnabled(true);
        stateValue.setEnabled(true);
        switchOffRB.setEnabled(true);
        switchOnRB.setEnabled(true);
        controlButton.setEnabled(true);
        control.updateControlPanel(rs);
        repaint();
    }

    public final void disableSystem() {
        nameValue.setEnabled(false);
        stateValue.setEnabled(false);
        switchOffRB.setEnabled(false);
        switchOnRB.setEnabled(false);
        controlButton.setEnabled(false);
        control.disableSystem();
        repaint();
    }

    private void initComponents() {

        nameLabel.setFont(UiConstants.FONT);
        nameValue.setFont(UiConstants.FONT);
        nameValue.setPreferredSize(nameValue.getPreferredSize());  // Yes - this does something!

        stateLabel.setFont(UiConstants.FONT);
        stateValue.setFont(UiConstants.FONT);
        Dimension d = stateValue.getPreferredSize();
        d.width = stateWidth;
        stateValue.setPreferredSize(d);

        switchLabel.setFont(UiConstants.FONT);
        switchOffRB.setFont(UiConstants.FONT);
        switchOffRB.setFocusable(false);
        switchOffRB.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                setSwitchOn(false);
            }
        });
        switchOnRB.setFont(UiConstants.FONT);
        switchOnRB.setFocusable(false);
        switchOnRB.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                setSwitchOn(true);
            }
        });
        switchBG.add(switchOffRB);
        switchBG.add(switchOnRB);

        JButton button = controlButton;
        Insets i = button.getMargin();
        i.left -= 3;
        i.right -= 3;
        button.setMargin(i);
        Dimension dr = button.getPreferredSize();
        dr.height -= 5;
        button.setPreferredSize(dr);
        button.setFont(UiConstants.FONT);
        button.setFocusable(false);
        button.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                showHideControl();
            }
        });

        // Lay out this panel
        setLayout(new GridBagLayout());
        GridBagConstraints gbm = new GridBagConstraints();
        gbm.insets = new Insets(0, 0, 0, 0);
        gbm.anchor = GridBagConstraints.WEST;
        gbm.gridx = 0;
        gbm.gridy = 0;
        add(nameLabel, gbm);
        gbm.gridx++;
        add(nameValue, gbm);
        gbm.gridx++;
        gbm.insets.left = 25;
        add(stateLabel, gbm);
        gbm.gridx++;
        gbm.insets.left = 0;
        add(stateValue, gbm);
        gbm.gridx++;
        gbm.insets.left = 25;
        add(switchLabel, gbm);
        gbm.gridx++;
        gbm.insets.left = 0;
        add(switchOffRB, gbm);
        gbm.gridx++;
        gbm.insets.left = 0;
        add(switchOnRB, gbm);
        gbm.gridx++;
        gbm.insets.left = 25;
        add(controlButton, gbm);
    }

    private void setSwitchOn(boolean value) {
        sender.sendCommand(comp, "setSwitchOn", CompState.SW_ENABLE, value);
    }
    
    private void showHideControl() {
        if (dialog != null) {
            dialog.setVisible(false);
            dialog.dispose();
            dialog = null;
            return;
        }
        Window window = SwingUtilities.getWindowAncestor(this);
        dialog = new JDialog(window, comp + " Control", Dialog.ModalityType.MODELESS);
        dialog.add(control, BorderLayout.CENTER);
        dialog.pack();
        dialog.setLocationRelativeTo(window);
        dialog.addWindowListener(new WindowAdapter() {
            @Override
            public void windowClosing(WindowEvent e) {
                dialog.setVisible(false);
                dialog.dispose();
                dialog = null;
            }
        });
        dialog.setVisible(true);
    }

    private static final long serialVersionUID = 1L;
}
