package org.lsst.ccs.subsystem.refrig.ui.jas;

import java.awt.BorderLayout;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import javax.swing.JComponent;
import javax.swing.JPanel;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.plugins.monitor.SectionedTableView;
import org.lsst.ccs.gconsole.plugins.monitor.TabbedView;
import org.lsst.ccs.gconsole.services.optpage.OptionalPage;
import org.lsst.ccs.messaging.AgentMessagingLayer;
import org.lsst.ccs.subsystem.commons.ui.jas.CommandSender;
import org.lsst.ccs.subsystem.commons.ui.jas.ControlPanel;
import org.lsst.ccs.subsystem.commons.ui.jas.PathMonitorFilter;

/**
 * @author tonyj
 */
public class CmprOptionalPage implements OptionalPage {

    private static final String[] exclSections = {"RefrigState", "UpdatePeriod"};
    private final Map<String, ControlPanel> controls = new ConcurrentHashMap<>();
    private final Map<String, String> agentMap = new ConcurrentHashMap<>();
    private final Map<String, String> assocAgentMap = new ConcurrentHashMap<>();
    private final String agentProperty, assocAgentProperty;
    private final Class controlClass;

    public CmprOptionalPage(String agentProperty, Class controlClass, String assocAgentProperty) {
        this.agentProperty = agentProperty;
        this.controlClass = controlClass;
        this.assocAgentProperty = assocAgentProperty;
    }

    @Override
    public boolean isAutoOpen() {
        return false;
    }

    @Override
    public String getPage(AgentInfo agent) {
        //System.out.println("getPage called: agentName = " + (agent == null ? null : agent.getName()));
        if (agent == null) return null;
        if (agent.hasAgentProperty(agentProperty)) {
            //System.out.println("Found main agent: " + agent.getName());
            return getPageName(agent.getName());
        }
        else if (agent.hasAgentProperty(assocAgentProperty)) {
            String group = getGroupProperty(agent);
            String assocAgentName = agent.getName();
            //System.out.println("Found associated agent: " + agent.getName());
            synchronized (this) {
                if (assocAgentMap.put(group, assocAgentName) == null) {
                    String agentName = agentMap.get(group);
                    if (agentName != null) {
                        ControlPanel control = controls.get(getPageName(agentName));
                        if (control != null) {
                            List<String> agents = new ArrayList<>();
                            agents.addAll(Arrays.asList(agentName, assocAgentName));
                            control.getView().setFilter(new PathMonitorFilter(agents, exclSections));
                        }
                    }
                }
            }
        }
        return null;
    }

    @Override
    public boolean agentDisconnected(AgentInfo agent) {
        //System.out.println("agentDisconnected called: agentName = " + agent.getName());
        ControlPanel control = controls.get(getPageName(agent.getName()));
        if (control == null) return true;
        control.disableSystem();
        return false;
    }

    @Override
    public JComponent open(AgentInfo agent, JComponent existingComponent) {
        String agentName = agent.getName();
        //System.out.println("open called: agentName = " + agentName);
        String pageName = getPageName(agentName);
        ControlPanel control = controls.get(pageName);
        if (control != null) {
            control.initControlPanel();
            return existingComponent;
        }

        // Create the control object
        try {
            control = (ControlPanel)controlClass.newInstance();
            controls.put(pageName, control);
        }
        catch (Exception e) {
            return null;
        }

        // Create the monitoring panel
        TabbedView view = new TabbedView();
        control.setView(view);
        view.setViewFactory(name -> new SectionedTableView());
        String group = getGroupProperty(agent);
        synchronized (this) {
            agentMap.put(group, agentName);
            List<String> agents = new ArrayList<>();
            agents.add(agentName);
            String assocAgentName = assocAgentMap.get(group);
            if (assocAgentName != null) {
                agents.add(assocAgentName);
            }
            view.setFilter(new PathMonitorFilter(agents, exclSections));
        }
        view.install();

        // Create the control panel
        JPanel cPanel = control.getControlPanel(new CommandSender(agentName));

        // Create the whole panel
        JPanel panel = new JPanel();
        panel.setLayout(new java.awt.BorderLayout());
        panel.add(view.getPanel(), BorderLayout.CENTER);
        panel.add(cPanel, BorderLayout.SOUTH);

        // Set up the bus listener
        AgentMessagingLayer aml = Console.getConsole().getMessagingAccess();
        aml.addStatusMessageListener(control.getListener(),
                                     (msg) -> msg.getOriginAgentInfo().getName().equals(agentName)
                                                && msg instanceof StatusSubsystemData);

        // Initialize the control panel
        control.initControlPanel();
        return panel;
    }

    @Override
    public void closed(String pageName) {
        //System.out.println("closed called: pageName = " + pageName);
        ControlPanel control = controls.remove(pageName);
        if (control == null) {
            throw new RuntimeException("No page exists with the name " + pageName);
        }
        Console.getConsole().getMessagingAccess().removeStatusMessageListener(control.getListener());
    }

    private String getPageName(String agentName) {
        return "CCS Subsystems/" + agentName;
    }

    private String getGroupProperty(AgentInfo agent) {
        return agent.getAgentProperty("Group", "");
    }

}
