package org.lsst.ccs.subsystem.refrig.ui;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.util.HashMap;
import java.util.Map;
import javax.swing.ButtonGroup;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JTextField;
import javax.swing.SwingConstants;
import org.lsst.ccs.subsystem.commons.ui.UiConstants;
import org.lsst.ccs.subsystem.commons.ui.jas.CommandSender;
import org.lsst.ccs.subsystem.refrig.constants.AuxHeaterState;
import org.lsst.ccs.subsystem.refrig.constants.ThermalConstants;
import org.lsst.ccs.subsystem.refrig.data.ThermalState;

/**
 *  Implements an auxiliary heater sub-panel.
 *
 *  @author saxton
 */
public class AuxHeaterPanel extends JPanel {

    private static final String[] heaterNames = new String[ThermalState.NUM_AUX_HEATERS];
    static {
        heaterNames[ThermalState.AUX_HEATER_MINUS_Y] = "-Y";
        heaterNames[ThermalState.AUX_HEATER_PLUS_Y] = "+Y";
        heaterNames[ThermalState.AUX_HEATER_CENTER] = "Center";
    }
    private static final int switchLabelWidth, switchStateWidth, textFieldWidth;
    static {
        JLabel label = new JLabel();
        label.setFont(UiConstants.FONT);
        int width = 0;
        for (String name : heaterNames) {
            label.setText(name + ":");
            width = Math.max(width, label.getPreferredSize().width);
        }
        switchLabelWidth = width;
        width = 0;
        for (AuxHeaterState state : AuxHeaterState.values()) {
            label.setText(state.name());
            width = Math.max(width, label.getPreferredSize().width);
        }
        switchStateWidth = width;
        JTextField field = new JTextField(" 1000.0 ");
        textFieldWidth = field.getPreferredSize().width;
    }
    private static final Map<AuxHeaterState, Color> stateColors = new HashMap<>();
    static {
        stateColors.put(AuxHeaterState.OFF, Color.BLACK);
        stateColors.put(AuxHeaterState.ON, UiConstants.GREEN);
        stateColors.put(AuxHeaterState.OFFLINE, UiConstants.BLUE);
        stateColors.put(AuxHeaterState.DISABLD, UiConstants.RED);
        stateColors.put(AuxHeaterState.VOLTERR, UiConstants.PURPLE);
        stateColors.put(AuxHeaterState.CANTSET, UiConstants.PURPLE);
        stateColors.put(AuxHeaterState.NOLOAD, UiConstants.PURPLE);
    }

    private final int id;
    private final CommandSender sender;

    private final JLabel switchLabel = new JLabel("");
    private final JLabel switchStatus = new JLabel("OFF");
    private final ButtonGroup switchBG = new ButtonGroup();
    private final JRadioButton switchOffRB = new JRadioButton("Off");
    private final JRadioButton switchOnRB = new JRadioButton("On");
    private final JTextField powerTextField = new JTextField();
    private final TextFieldX powerTFX = new TextFieldX(powerTextField, TextFieldX.TYPE_DOUBLE);
    private final JLabel powerUnitsLabel = new JLabel("W");


    AuxHeaterPanel(int id, CommandSender sender) {
        this.id = id;
        this.sender = sender;
        initComponents();
    }

    public void updatePanel(ThermalState rs) {
        switchStatus.setEnabled(true);
        AuxHeaterState pState = rs.getAuxHeaterState(id);
        switchStatus.setText(pState.name());
        Color color = stateColors.get(pState);
        color = color == null ? UiConstants.PURPLE : color;
        switchStatus.setForeground(color);

        boolean isOn = ThermalConstants.auxOnStates.contains(pState);
        JRadioButton selButton = isOn ? switchOnRB : switchOffRB;
        selButton.setSelected(true);
        switchOffRB.setEnabled(isOn);
        switchOnRB.setEnabled(pState == AuxHeaterState.OFF);

        powerTFX.update(rs.getAuxHeaterPower(id), true);
    }

    public void disablePanel() {
        switchStatus.setEnabled(false);
        switchOffRB.setEnabled(false);
        switchOnRB.setEnabled(false);
        powerTFX.disable();
    }

    private void initComponents() {
        switchLabel.setFont(UiConstants.FONT);
        switchLabel.setText(heaterNames[id] + ":");
        Dimension d = switchLabel.getPreferredSize();
        d.width = switchLabelWidth;
        switchLabel.setPreferredSize(d);
        switchStatus.setFont(UiConstants.FONT);
        d = switchStatus.getPreferredSize();
        d.width = switchStateWidth;
        switchStatus.setPreferredSize(d);
        switchOffRB.setFont(UiConstants.FONT);
        switchOffRB.setFocusable(false);
        switchOffRB.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                setHeaterPowerState(0);
            }
        });
        switchOnRB.setFont(UiConstants.FONT);
        switchOnRB.setFocusable(false);
        switchOnRB.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                setHeaterPowerState(1);
            }
        });
        switchBG.add(switchOffRB);
        switchBG.add(switchOnRB);

        d = powerTextField.getPreferredSize();
        d.width = textFieldWidth;
        powerTextField.setPreferredSize(d);
        powerTextField.setHorizontalAlignment(SwingConstants.CENTER);
        powerTextField.setText("...");
        powerTextField.addFocusListener(new FocusAdapter() {
            @Override
            public void focusLost(FocusEvent evt) {
                powerTFX.focusLost();
            }
        });
        powerTextField.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                Double value = (Double)powerTFX.action();
                if (value != null) {
                    sender.sendCommand(null, "setAuxHeaterPower", id, value);
                }
            }
        });
        powerTextField.addKeyListener(new KeyAdapter() {
            @Override
            public void keyTyped(KeyEvent evt) {
                powerTFX.keyTyped(evt.getKeyChar());
            }
        });
        powerUnitsLabel.setFont(UiConstants.FONT);

        // Lay out the panel
        setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = new Insets(0, 0, 0, 0);
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.anchor = GridBagConstraints.NORTHWEST;
        gbc.insets.left = 6;
        add(switchLabel, gbc);
        gbc.gridx++;
        add(switchStatus, gbc);
        gbc.gridx++;
        gbc.insets.top = -6;
        add(switchOffRB, gbc);
        gbc.gridx++;
        gbc.insets.left = 4;
        add(switchOnRB, gbc);
        gbc.gridx++;
        gbc.insets.left = 6;
        gbc.insets.top = 0;
        add(powerTextField, gbc);
        gbc.gridx++;
        gbc.insets.right = 6;
        add(powerUnitsLabel, gbc);
    }

    private void setHeaterPowerState(int value) {
        switchOffRB.setEnabled(false);
        switchOnRB.setEnabled(false);
        sender.sendCommand(null, "setAuxHeaterState", id, value);
    }

    private static final long serialVersionUID = 1L;
}
