package org.lsst.ccs.subsystem.refrig.ui;

import java.awt.Color;
import java.awt.Insets;
import java.awt.image.BufferedImage;
import java.io.IOException;
import javax.imageio.ImageIO;
import javax.swing.BorderFactory;
import static javax.swing.JComponent.*;
import javax.swing.SwingUtilities;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.gconsole.plugins.monitor.CellTableView;
import org.lsst.ccs.gconsole.plugins.monitor.DisplayChannelSingle;
import org.lsst.ccs.gconsole.plugins.monitor.FormattedValue;
import org.lsst.ccs.gconsole.plugins.monitor.ImageView;
import org.lsst.ccs.gconsole.plugins.monitor.DefaultMonitorCell;
import org.lsst.ccs.gconsole.plugins.monitor.MonitorField;

/**
 * Displays the cold plate schematics.
 *
 * @author onoprien
 */
public class ColdPage {

// -- Fields : -----------------------------------------------------------------
    
    private final AgentInfo agent;
    
    private ImageView view;
    private final MonitorField heaterField;

// -- Life cycle : -------------------------------------------------------------
    
    public ColdPage(AgentInfo agent) {
        
        this.agent = agent;
        
        FormattedValue fv = new FormattedValue();
        fv.horizontalAlignment = SwingUtilities.CENTER;
        fv.format = " %1$6.2f ";
        heaterField = new MonitorField(MonitorField.VALUE.getKey(), null, fv, MonitorField.VALUE.getKeys());        
    }
    
    /**
     * Creates a {@code MonitorView} that displays the vacuum plate schematics.
     * @return {@code MonitorView} that displays the vacuum plate schematics.
     */
    public ImageView getView() {
        
        if (view != null) return view;
        
        try {
            
            BufferedImage back = ImageIO.read(ImageView.class.getResource("/cold.png"));
            view = new ImageView("Cold Plate", back, .5, 2.);
            
            placeRTD(0, 0.5583, 0.7698, LEFT_ALIGNMENT, TOP_ALIGNMENT);
            placeRTD(2, 0.6119, 0.5740, RIGHT_ALIGNMENT, BOTTOM_ALIGNMENT);
            placeRTD(3, 0.6119, 0.4290, RIGHT_ALIGNMENT, BOTTOM_ALIGNMENT, true);
            placeRTD(5, 0.5640, 0.2078, LEFT_ALIGNMENT, TOP_ALIGNMENT, true);
            placeRTD(50, 0.1014, 0.7653, RIGHT_ALIGNMENT, TOP_ALIGNMENT, true);
            placeRTD(52, 0.0443, 0.5889, LEFT_ALIGNMENT, TOP_ALIGNMENT);
            placeRTD(53, 0.0443, 0.4410, LEFT_ALIGNMENT, TOP_ALIGNMENT);
            placeRTD(55, 0.1049, 0.2392, RIGHT_ALIGNMENT, BOTTOM_ALIGNMENT, true);
            
            placeHeaterPower(0, 0.7669, 0.83);
            placeHeaterPower(1, 0.7669, 0.75);
            placeHeaterPower(2, 0.7669, 0.61);
            placeHeaterPower(3, 0.7669, 0.45);
            placeHeaterPower(4, 0.7669, 0.33);
            placeHeaterPower(5, 0.7669, 0.25);
            
            // Total trim power:
            
            String channelPath = agent.getName() +"/main/Trim_Htrs/ColdTotal_P";
            DefaultMonitorCell c = new DefaultMonitorCell(new DisplayChannelSingle(channelPath, null, null), heaterField);
            CellTableView table = new CellTableView(1, 1, c);
            ImageView.Item item = new ImageView.Item(table);
            item.setPosition(0.9, 0.9641);
            item.setAlignment(LEFT_ALIGNMENT, BOTTOM_ALIGNMENT);
//            item.setMaxSize(0.06, 0.029);
            item.setMaxSize(0.08, 0.06);
            view.add(item);
                        
        } catch (IOException e) {
        }
        return view;
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    private void placeRTD(int id, double x, double y, float horAlign, float vertAlign, boolean redBorder) throws IOException {
        String channelPath = String.format("%1$s/main/Cold_Temp/CLP-RTD-%2$02d", agent.getName(), id);
        FormattedValue fv = new FormattedValue();
        fv.horizontalAlignment = SwingUtilities.CENTER;
        fv.format = "<html><center>RTD-"+  String.format("%02d", id) +":<br>%1$6.1f";
        MonitorField field = new MonitorField(MonitorField.VALUE.getKey(), null, fv, MonitorField.VALUE.getKeys());        
        DefaultMonitorCell c = new DefaultMonitorCell(new DisplayChannelSingle(channelPath, null, null), field);
        CellTableView table = new CellTableView(1, 1, c);
        if (redBorder) {
            Insets ins = table.getBorder().getBorderInsets(table);
            table.setBorder(BorderFactory.createMatteBorder(3*ins.top, 3*ins.left, 3*ins.bottom, 3*ins.right, Color.RED));
        }
        ImageView.Item item = new ImageView.Item(table);
        item.setPosition(x, y);
        item.setAlignment(horAlign, vertAlign);
//        item.setMaxSize(0.055, 0.055);
        item.setMaxSize(0.08, 0.08);
        view.add(item);
    }
    
    private void placeRTD(int id, double x, double y, float horAlign, float vertAlign) throws IOException {
        placeRTD(id, x, y, horAlign, vertAlign, false);
    }
    
    private void placeHeaterPower(int id, double x, double y) throws IOException {
        String channelPath = String.format("%1$s/main/Trim_Htrs/ColdHtr%2$1d_P", agent.getName(), id);
        DefaultMonitorCell c = new DefaultMonitorCell(new DisplayChannelSingle(channelPath, null, null), heaterField);
        CellTableView table = new CellTableView(1, 1, c);
        ImageView.Item item = new ImageView.Item(table);
        item.setPosition(x, y);
        item.setAlignment(LEFT_ALIGNMENT, TOP_ALIGNMENT);
//        item.setMaxSize(0.1260, 0.0375);
        item.setMaxSize(0.1260, 0.06);
        view.add(item);
    }
    
}
