package org.lsst.ccs.subsystem.refrig.ui;

import java.awt.Color;
import java.awt.Insets;
import java.awt.image.BufferedImage;
import java.io.IOException;
import javax.imageio.ImageIO;
import javax.swing.BorderFactory;
import static javax.swing.JComponent.*;
import javax.swing.SwingUtilities;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.gconsole.agent.AgentChannel;
import org.lsst.ccs.gconsole.plugins.monitor.CellTableView;
import org.lsst.ccs.gconsole.plugins.monitor.DefaultChannelHandle;
import org.lsst.ccs.gconsole.plugins.monitor.FormattedValue;
import org.lsst.ccs.gconsole.plugins.monitor.ImageView;
import org.lsst.ccs.gconsole.plugins.monitor.MonitorCell;
import org.lsst.ccs.gconsole.plugins.monitor.MonitorField;

/**
 * Displays the cold plate schematics.
 *
 * @author onoprien
 */
public class CryoPage {

// -- Fields : -----------------------------------------------------------------
    
    private final AgentInfo agent;
    
    private ImageView view;
    private final MonitorField rtdField, heaterField, otherField;

// -- Life cycle : -------------------------------------------------------------
    
    public CryoPage(AgentInfo agent) {
        
        this.agent = agent;
        
        FormattedValue fv = new FormattedValue();
        fv.horizontalAlignment = SwingUtilities.CENTER;
        fv.format = " %1$7.2f ";
        rtdField = new MonitorField(MonitorField.VALUE.getKey(), null, fv, MonitorField.VALUE.getKeys());        
        fv.format = " %1$6.2f ";
        heaterField = new MonitorField(MonitorField.VALUE.getKey(), null, fv, MonitorField.VALUE.getKeys());    
        otherField = new MonitorField(MonitorField.VALUE.getKey(), null, fv, MonitorField.VALUE.getKeys()) {
            @Override
            public FormattedValue format(AgentChannel channel) {
                FormattedValue fv = super.format(channel);
                String s = channel.getPath();
                s = s.substring(s.lastIndexOf("/")+1);
                StringBuilder sb = new StringBuilder();
                if (s.startsWith("C")) {
                    sb.append("Cryo Flex ");
                } else if (s.startsWith("GF")) {
                    sb.append("Grid Flex ");
                } else {
                    sb.append("Grid Bay ");
                }
                sb.append(s.charAt(s.length()-1));
                sb.append(": %1$7.2f ");
                fv.format = sb.toString();
                return fv;
            }
        };
    }
    
    /**
     * Creates a {@code MonitorView} that displays the vacuum plate schematics.
     * @return {@code MonitorView} that displays the vacuum plate schematics.
     */
    public ImageView getView() {
        
        if (view != null) return view;
        
        try {
            
            BufferedImage back = ImageIO.read(ImageView.class.getResource("/cryo.png"));
            view = new ImageView("Cold Plate", back, .5, 2.);
            
            placeRTD(2, 0.687, 0.5798, RIGHT_ALIGNMENT, TOP_ALIGNMENT);
            placeRTD(3, 0.687, 0.433, RIGHT_ALIGNMENT, TOP_ALIGNMENT);
            placeRTD(4, 0.687, 0.2846, RIGHT_ALIGNMENT, TOP_ALIGNMENT);
            placeRTD(12, 0.5945, 0.5798, RIGHT_ALIGNMENT, TOP_ALIGNMENT, true);
            placeRTD(13, 0.5945, 0.433, RIGHT_ALIGNMENT, TOP_ALIGNMENT);
            placeRTD(14, 0.5945, 0.2846, RIGHT_ALIGNMENT, TOP_ALIGNMENT, true);
            placeRTD(21, 0.5009, 0.7380, RIGHT_ALIGNMENT, TOP_ALIGNMENT);
            placeRTD(22, 0.5009, 0.5798, RIGHT_ALIGNMENT, TOP_ALIGNMENT);
            placeRTD(24, 0.5009, 0.2846, RIGHT_ALIGNMENT, TOP_ALIGNMENT);
            placeRTD(25, 0.5009, 0.1401, RIGHT_ALIGNMENT, TOP_ALIGNMENT);
            placeRTD(30, 0.4149, 0.8675, LEFT_ALIGNMENT, BOTTOM_ALIGNMENT);
            placeRTD(31, 0.4149, 0.725, LEFT_ALIGNMENT, BOTTOM_ALIGNMENT);
            placeRTD(33, 0.4149, 0.4202, LEFT_ALIGNMENT, BOTTOM_ALIGNMENT);
            placeRTD(34, 0.4149, 0.2726, LEFT_ALIGNMENT, BOTTOM_ALIGNMENT);
            placeRTD(41, 0.3214, 0.725, LEFT_ALIGNMENT, BOTTOM_ALIGNMENT, true);
            placeRTD(42, 0.3214, 0.567, LEFT_ALIGNMENT, BOTTOM_ALIGNMENT);
            placeRTD(43, 0.3214, 0.4202, LEFT_ALIGNMENT, BOTTOM_ALIGNMENT, true);
            placeRTD(51, 0.2259, 0.725, LEFT_ALIGNMENT, BOTTOM_ALIGNMENT);
            placeRTD(52, 0.2259, 0.567, LEFT_ALIGNMENT, BOTTOM_ALIGNMENT);
            placeRTD(53, 0.2259, 0.4202, LEFT_ALIGNMENT, BOTTOM_ALIGNMENT);
            
            placeHeaterPower(0, 0.837, 0.8268);
            placeHeaterPower(1, 0.837, 0.7440);
            placeHeaterPower(2, 0.837, 0.5964);
            placeHeaterPower(3, 0.837, 0.4413);
            placeHeaterPower(4, 0.837, 0.3102);
            placeHeaterPower(5, 0.837, 0.2380);
            
            // Total trim power:
            
            String channelPath = agent.getName() +"/main/Trim_Htrs/CryoTotal_P";
            MonitorCell c = new MonitorCell(new DefaultChannelHandle(channelPath, null, null), heaterField);
            CellTableView table = new CellTableView(1, 1, c);
            ImageView.Item item = new ImageView.Item(table);
            item.setPosition(0.92, 0.96);
            item.setAlignment(LEFT_ALIGNMENT, BOTTOM_ALIGNMENT);
//            item.setMaxSize(0.06, 0.029);
            item.setMaxSize(0.08, 0.06);
            view.add(item);
            
            // Other temps:
            
//            FormattedValue fv = new FormattedValue();
//            fv.horizontalAlignment = SwingUtilities.LEFT;
//            fv.format = "Cryo Flex 1: %1$7.2f ";
//            MonitorField mf = new MonitorField(MonitorField.VALUE.getKey(), null, fv, MonitorField.VALUE.getKeys());

            MonitorCell[] cc = new MonitorCell[12];
            cc[0] = new MonitorCell(new DefaultChannelHandle(agent.getName() +"/main/Grid_Temp/GFX-RTD-01", null, null), otherField);
            cc[1] = new MonitorCell(new DefaultChannelHandle(agent.getName() +"/main/Grid_Temp/GFX-RTD-02", null, null), otherField);
            cc[2] = new MonitorCell(new DefaultChannelHandle(agent.getName() +"/main/Grid_Temp/GFX-RTD-03", null, null), otherField);
            cc[3] = new MonitorCell(new DefaultChannelHandle(agent.getName() +"/main/Grid_Temp/GFX-RTD-04", null, null), otherField);
            cc[4] = new MonitorCell(new DefaultChannelHandle(agent.getName() +"/main/Grid_Temp/CFX-RTD-01", null, null), otherField);
            cc[5] = new MonitorCell(new DefaultChannelHandle(agent.getName() +"/main/Grid_Temp/CFX-RTD-02", null, null), otherField);
            cc[6] = new MonitorCell(new DefaultChannelHandle(agent.getName() +"/main/Grid_Temp/CFX-RTD-03", null, null), otherField);
            cc[7] = new MonitorCell(new DefaultChannelHandle(agent.getName() +"/main/Grid_Temp/CFX-RTD-04", null, null), otherField);
            cc[8] = new MonitorCell(new DefaultChannelHandle(agent.getName() +"/main/Grid_Temp/GRD-RTD-01", null, null), otherField);
            cc[9] = new MonitorCell(new DefaultChannelHandle(agent.getName() +"/main/Grid_Temp/GRD-RTD-02", null, null), otherField);
            cc[10] = new MonitorCell(new DefaultChannelHandle(agent.getName() +"/main/Grid_Temp/GRD-RTD-03", null, null), otherField);
            cc[11] = new MonitorCell(new DefaultChannelHandle(agent.getName() +"/main/Grid_Temp/GRD-RTD-04", null, null), otherField);

            table = new CellTableView(12, 1, cc);
            item = new ImageView.Item(table);
            item.setPosition(0.109, 0.28);
            item.setAlignment(CENTER_ALIGNMENT, TOP_ALIGNMENT);
            item.setMaxSize(0.183, 0.668);
            view.add(item);
            
                        
        } catch (IOException e) {
        }
        return view;
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    private void placeRTD(int id, double x, double y, float horAlign, float vertAlign, boolean redBorder) throws IOException {
        String channelPath = String.format("%1$s/main/Cryo_Temp/CYP-RTD-%2$02d", agent.getName(), id);
        FormattedValue fv = new FormattedValue();
        fv.horizontalAlignment = SwingUtilities.CENTER;
        fv.format = "<html><center>RTD-"+ String.format("%02d", id) +":<br>%1$6.1f";
        MonitorField field = new MonitorField(MonitorField.VALUE.getKey(), null, fv, MonitorField.VALUE.getKeys());        
        MonitorCell c = new MonitorCell(new DefaultChannelHandle(channelPath, null, null), field);
        CellTableView table = new CellTableView(1, 1, c);
        if (redBorder) {
            Insets ins = table.getBorder().getBorderInsets(table);
            table.setBorder(BorderFactory.createMatteBorder(3*ins.top, 3*ins.left, 3*ins.bottom, 3*ins.right, Color.RED));
        }
        ImageView.Item item = new ImageView.Item(table);
        item.setPosition(x, y);
        item.setAlignment(horAlign, vertAlign);
//        item.setMaxSize(0.055, 0.055);
        item.setMaxSize(0.08, 0.064);
        view.add(item);
    }
    
    private void placeRTD(int id, double x, double y, float horAlign, float vertAlign) throws IOException {
        placeRTD(id, x, y, horAlign, vertAlign, false);
    }
    
    private void placeHeaterPower(int id, double x, double y) throws IOException {
        String channelPath = String.format("%1$s/main/Trim_Htrs/CryoHtr%2$1d_P", agent.getName(), id);
        MonitorCell c = new MonitorCell(new DefaultChannelHandle(channelPath, null, null), heaterField);
        CellTableView table = new CellTableView(1, 1, c);
        ImageView.Item item = new ImageView.Item(table);
        item.setPosition(x, y);
        item.setAlignment(LEFT_ALIGNMENT, TOP_ALIGNMENT);
//        item.setMaxSize(0.1260, 0.0375);
        item.setMaxSize(0.1260, 0.06);
        view.add(item);
    }
    
}
