package org.lsst.ccs.subsystem.refrig.ui;

import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import javax.swing.JPanel;
import javax.swing.SwingUtilities;
import org.lsst.ccs.subsystem.refrig.ui.jas.CommandSender;
import org.lsst.ccs.subsystem.refrig.data.CompState;
import org.lsst.ccs.subsystem.refrig.data.RefrigState;

/**
 *  Implements the refrigeration control panel.
 *
 *  @author Owen Saxton
 */
public class RefrigControlPanel extends javax.swing.JPanel {

    private final CommandSender sender;

    private final JPanel mainPanel = new JPanel();
    private SystemStatusPanel statusPanel;
    private int numColdPanels = 0, numCryoPanels = 0;
    private final CompressorSummaryPanel[] coldLines = new CompressorSummaryPanel[RefrigState.MAX_COLD_COMPRESSORS];
    private final CompressorSummaryPanel[] cryoLines = new CompressorSummaryPanel[RefrigState.MAX_CRYO_COMPRESSORS];

    public RefrigControlPanel(CommandSender cons) {
        this.sender = cons;
        initComponents();
        (new UpdateRefrigStatus(new RefrigState())).run();
        (new DisableSystem()).run();
    }

    public void updateControlPanel(RefrigState rs) {
        SwingUtilities.invokeLater(new UpdateRefrigStatus(rs));
    }

    public void disableSystem() {
        SwingUtilities.invokeLater(new DisableSystem());
    }

    private void initComponents() {

        // Status line panel
        statusPanel = new SystemStatusPanel(sender);

        // Set up the compressor summary lines and control panels
        for (int comp = 0; comp < coldLines.length; comp++) {
            coldLines[comp] = new CompressorSummaryPanel(CompState.TYPE_COLD, sender);
        }
        for (int comp = 0; comp < cryoLines.length; comp++) {
            cryoLines[comp] = new CompressorSummaryPanel(CompState.TYPE_CRYO, sender);
        }

        // Layout initial main panel
        mainPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbm = new GridBagConstraints();
        gbm.insets = new Insets(0, 0, 0, 0);
        gbm.anchor = GridBagConstraints.NORTH;
        gbm.gridx = 0;
        gbm.gridy = 0;
        mainPanel.add(statusPanel, gbm);

        // Lay out the complete panel
        add(mainPanel);
    }

    private void layoutMainPanel(int numColdComps, int numCryoComps) {
        if (numColdComps == numColdPanels && numCryoComps == numCryoPanels) return;
        for (int j = 0; j < numColdPanels; j++) {
            mainPanel.remove(coldLines[j]);
        }
        for (int j = 0; j < numCryoPanels; j++) {
            mainPanel.remove(cryoLines[j]);
        }
        numColdPanels = numColdComps;
        numCryoPanels = numCryoComps;
        mainPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbm = new GridBagConstraints();
        gbm.insets = new Insets(0, 0, 0, 0);
        gbm.anchor = GridBagConstraints.NORTH;
        gbm.gridx = 0;
        gbm.gridy = 1;
        for (int j = 0; j < numColdPanels; j++) {
            mainPanel.add(coldLines[j], gbm);
            gbm.gridy++;
        }
        for (int j = 0; j < numCryoPanels; j++) {
            mainPanel.add(cryoLines[j], gbm);
            gbm.gridy++;
        }
    }

    class UpdateRefrigStatus implements Runnable {

        private final RefrigState rs;

        UpdateRefrigStatus(RefrigState rs) {
            this.rs = rs;
        }

        @Override
        public void run() {
            statusPanel.updateStatus(rs.getTickMillis());
            for (CompState cs : rs.getColdStates()) {
                coldLines[cs.getIndex()].updateControlPanel(cs);
            }
            for (CompState cs : rs.getCryoStates()) {
                cryoLines[cs.getIndex()].updateControlPanel(cs);
            }
            if (rs.getNumColdComps() != numColdPanels || rs.getNumCryoComps() != numCryoPanels) {
                layoutMainPanel(rs.getNumColdComps(), rs.getNumCryoComps());
            }
            repaint();
        }

    }

    class DisableSystem implements Runnable {

        @Override
        public void run() {
            statusPanel.disableSystem();
            for (CompressorSummaryPanel comp : coldLines) {
                comp.disableSystem();
            }
            for (CompressorSummaryPanel comp : cryoLines) {
                comp.disableSystem();
            }
            repaint();
        } 
    }

    private static final long serialVersionUID = 1L;
}
