package org.lsst.ccs.subsystem.refrig.ui;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import javax.swing.ButtonGroup;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JTextField;
import javax.swing.SwingConstants;
import org.lsst.ccs.subsystem.refrig.constants.HeaterControlState;
import org.lsst.ccs.subsystem.refrig.constants.HeaterPowerState;
import org.lsst.ccs.subsystem.refrig.data.ThermalState;
import org.lsst.ccs.subsystem.refrig.ui.jas.CommandSender;

/**
 *  Implements a heater sub-panel.
 *
 *  @author saxton
 */
public class TrimHeaterPanel extends JPanel {

    private static final String[] plateNames = new String[ThermalState.NUM_TRIM_HEATERS];
    static {
        plateNames[ThermalState.TRIM_HEATER_COLD] = "Cold";
        plateNames[ThermalState.TRIM_HEATER_CRYO] = "Cryo";
    }
    private static final int powerLabelWidth, powerStateWidth, controlLabelWidth, controlStateWidth, powerFieldWidth, tempFieldWidth;
    static {
        JLabel label = new JLabel();
        label.setFont(UiConstants.FONT);
        int width = 0;
        for (String name : plateNames) {
            label.setText(name + " Power: ");
            width = Math.max(width, label.getPreferredSize().width);
        }
        powerLabelWidth = width;
        width = 0;
        for (HeaterPowerState state : HeaterPowerState.values()) {
            label.setText(state.name());
            width = Math.max(width, label.getPreferredSize().width);
        }
        powerStateWidth = width;
        width = 0;
        for (String name : plateNames) {
            label.setText(name + " Control: ");
            width = Math.max(width, label.getPreferredSize().width);
        }
        controlLabelWidth = width;
        width = 0;
        for (HeaterControlState state : HeaterControlState.values()) {
            label.setText(state.name());
            width = Math.max(width, label.getPreferredSize().width);
        }
        controlStateWidth = width;
        JTextField field = new JTextField(" 999.9 ");
        powerFieldWidth = field.getPreferredSize().width;
        field.setText(" -999.9 ");
        tempFieldWidth = field.getPreferredSize().width;
    }

    private final int id;
    private final CommandSender sender;

    private final JLabel switchLabel = new JLabel();
    private final JLabel switchState = new JLabel("OFF");
    private final ButtonGroup switchBG = new ButtonGroup();
    private final JRadioButton switchOffRB = new JRadioButton("Off");
    private final JRadioButton switchOnRB = new JRadioButton("On");

    private final JLabel controlLabel = new JLabel();
    private final JLabel controlState = new JLabel("OFF");
    private final ButtonGroup controlBG = new ButtonGroup();
    private final JRadioButton controlOffRB = new JRadioButton("Off");
    private final JRadioButton controlPowerRB = new JRadioButton("Power");
    private final JRadioButton controlTempRB = new JRadioButton("Temp");

    private final JTextField powerTextField = new JTextField();
    private final TextFieldX powerTFX = new TextFieldX(powerTextField, TextFieldX.TYPE_DOUBLE);
    private final JLabel powerUnitsLabel = new JLabel("W");

    private final JTextField tempTextField = new JTextField();
    private final TextFieldX tempTFX = new TextFieldX(tempTextField, TextFieldX.TYPE_DOUBLE);
    private final JLabel tempUnitsLabel = new JLabel("°C");

    TrimHeaterPanel(int id, CommandSender sender) {
        this.id = id;
        this.sender = sender;
        initComponents();
    }

    public void updatePanel(ThermalState rs) {
        switchState.setEnabled(true);
        controlState.setEnabled(true);
        HeaterPowerState pState = rs.getTrimHeaterState(id);
        switchState.setText(pState.name());
        switchState.setForeground(pState == HeaterPowerState.ON ? UiConstants.GREEN :
                                   pState == HeaterPowerState.DISABLD ? UiConstants.RED :
                                   pState == HeaterPowerState.OFFLINE ? UiConstants.BLUE : Color.black);
        JRadioButton selButton = pState == HeaterPowerState.ON ? switchOnRB : switchOffRB;
        selButton.setSelected(true);
        switchOffRB.setEnabled(pState == HeaterPowerState.ON);
        switchOnRB.setEnabled(pState == HeaterPowerState.OFF);

        HeaterControlState cState = rs.getTrimHeaterCtrlState(id);
        controlState.setText(cState.name());
        controlState.setForeground(cState == HeaterControlState.POWER ? UiConstants.YELLOW :
                                    cState == HeaterControlState.TEMP ? UiConstants.GREEN :
                                    cState == HeaterControlState.OFFLINE ? UiConstants.BLUE : Color.black);
        selButton = cState == HeaterControlState.POWER ? controlPowerRB :
                    cState == HeaterControlState.TEMP ? controlTempRB : controlOffRB;
        selButton.setSelected(true);
        boolean online = cState != HeaterControlState.OFFLINE;
        controlOffRB.setEnabled(online && selButton != controlOffRB);
        controlPowerRB.setEnabled(online && selButton != controlPowerRB);
        controlTempRB.setEnabled(online && selButton != controlTempRB);

        powerTFX.update(rs.getTrimHeaterPower(id), true);
        tempTFX.update(rs.getPlateTemperature(id), true);
        repaint();
    }

    public void disablePanel() {
        switchState.setEnabled(false);
        switchOffRB.setEnabled(false);
        switchOnRB.setEnabled(false);
        controlState.setEnabled(false);
        controlOffRB.setEnabled(false);
        controlPowerRB.setEnabled(false);
        controlTempRB.setEnabled(false);
        powerTFX.disable();
        tempTFX.disable();
        repaint();
    }

    private void initComponents() {
        switchLabel.setFont(UiConstants.FONT);
        switchLabel.setText(plateNames[id] + " Power: ");
        Dimension d = switchLabel.getPreferredSize();
        d.width = powerLabelWidth;
        switchLabel.setPreferredSize(d);
        switchState.setFont(UiConstants.FONT);
        d = switchState.getPreferredSize();
        d.width = powerStateWidth;
        switchState.setPreferredSize(d);
        switchOffRB.setFont(UiConstants.FONT);
        switchOffRB.setFocusable(false);
        switchOffRB.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                setHeaterPowerState(0);
            }
        });
        switchOnRB.setFont(UiConstants.FONT);
        switchOnRB.setFocusable(false);
        switchOnRB.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                setHeaterPowerState(1);
            }
        });
        switchBG.add(switchOffRB);
        switchBG.add(switchOnRB);

        controlLabel.setFont(UiConstants.FONT);
        controlLabel.setText(plateNames[id] + " Control: ");
        d = controlLabel.getPreferredSize();
        d.width = controlLabelWidth;
        controlLabel.setPreferredSize(d);
        controlState.setFont(UiConstants.FONT);
        d = controlState.getPreferredSize();
        d.width = controlStateWidth;
        controlState.setPreferredSize(d);
        controlOffRB.setFont(UiConstants.FONT);
        controlOffRB.setFocusable(false);
        controlOffRB.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                setHeaterControlState(0);
            }
        });
        controlPowerRB.setFont(UiConstants.FONT);
        controlPowerRB.setFocusable(false);
        controlPowerRB.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                setHeaterControlState(1);
            }
        });
        controlTempRB.setFont(UiConstants.FONT);
        controlTempRB.setFocusable(false);
        controlTempRB.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                setHeaterControlState(-1);
            }
        });
        controlBG.add(controlOffRB);
        controlBG.add(controlPowerRB);
        controlBG.add(controlTempRB);

        d = powerTextField.getPreferredSize();
        d.width = powerFieldWidth;
        powerTextField.setPreferredSize(d);
        powerTextField.setHorizontalAlignment(SwingConstants.CENTER);
        powerTextField.setText("...");
        powerTextField.addFocusListener(new FocusAdapter() {
            @Override
            public void focusLost(FocusEvent evt) {
                powerTFX.focusLost();
            }
        });
        powerTextField.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                Double value = (Double)powerTFX.action();
                if (value != null) {
                    sender.sendCommand(null, "setTrimHeaterPower", id, value);
                }
            }
        });
        powerTextField.addKeyListener(new KeyAdapter() {
            @Override
            public void keyTyped(KeyEvent evt) {
                powerTFX.keyTyped(evt.getKeyChar());
            }
        });
        powerUnitsLabel.setFont(UiConstants.FONT);

        d = tempTextField.getPreferredSize();
        d.width = tempFieldWidth;
        tempTextField.setPreferredSize(d);
        tempTextField.setHorizontalAlignment(SwingConstants.CENTER);
        tempTextField.setText("...");
        tempTextField.addFocusListener(new FocusAdapter() {
            @Override
            public void focusLost(FocusEvent evt) {
                tempTFX.focusLost();
            }
        });
        tempTextField.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                Double value = (Double)tempTFX.action();
                if (value != null) {
                    sender.sendCommand(null, "setPlateTemperature", id, value);
                }
            }
        });
        tempTextField.addKeyListener(new KeyAdapter() {
            @Override
            public void keyTyped(KeyEvent evt) {
                tempTFX.keyTyped(evt.getKeyChar());
            }
        });
        tempUnitsLabel.setFont(UiConstants.FONT);

        // Lay out the panel
        setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = new Insets(0, 5, 0, 0);
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.anchor = GridBagConstraints.NORTHWEST;

        // Add heater power to panel
        gbc.insets.top = 5;
        add(switchLabel, gbc);
        gbc.gridx++;
        add(switchState, gbc);
        gbc.gridx++;
        gbc.insets.top = 1;
        add(switchOffRB, gbc);
        gbc.gridx++;
        add(switchOnRB, gbc);
        gbc.gridx = 0;
        gbc.gridy++;

        // Add heater control to panel
        gbc.insets.top = 0;
        add(controlLabel, gbc);
        gbc.gridx++;
        add(controlState, gbc);
        gbc.gridx++;
        gbc.insets.top = -4;
        add(controlOffRB, gbc);
        gbc.gridx++;
        add(controlPowerRB, gbc);
        gbc.gridx++;
        gbc.insets.top = 0;
        add(powerTextField, gbc);
        gbc.gridx++;
        add(powerUnitsLabel, gbc);
        gbc.gridx++;
        gbc.insets.top = -4;
        gbc.insets.left = 8;
        add(controlTempRB, gbc);
        gbc.gridx++;
        gbc.insets.top = 0;
        gbc.insets.left = 5;
        add(tempTextField, gbc);
        gbc.gridx++;
        gbc.insets.right = 5;
        add(tempUnitsLabel, gbc);
    }

    private void setHeaterPowerState(int value) {
        switchOffRB.setEnabled(false);
        switchOnRB.setEnabled(false);
        sender.sendCommand(null, "setTrimHeaterPowerEnable", id, value);
    }

    private void setHeaterControlState(int value) {
        controlOffRB.setEnabled(false);
        controlPowerRB.setEnabled(false);
        controlTempRB.setEnabled(false);
        sender.sendCommand(null, "setTrimHeaterControl", id, value);
    }

    private static final long serialVersionUID = 1L;
}
