package org.lsst.ccs.subsystem.refrig.ui;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import javax.swing.ButtonGroup;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JTextField;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;
import org.lsst.ccs.subsystem.monitor.ui.CommandSender;
import org.lsst.ccs.subsystem.refrig.data.RefrigState;

/**
 ****************************************************************
 *
 *  Implements the refrigeration cold system test control panel.
 *
 *  @author Owen Saxton
 *
 ****************************************************************
 */
public class ColdControlPanel extends RefrigControlPanel {

    private static final Font FONT = new java.awt.Font("Tahoma", 1, 12);
    private static final Color RED = new Color(150, 0, 0),
                               GREEN = new Color(0, 150, 0);
    private final CommandSender gui;
    private final TextFieldX periodTFX, powerTFX, tempTFX;
    private String subsys;
    private boolean enabled;

    private final JPanel statePanel = new JPanel();
    private final JLabel stateLabel = new JLabel("DAQ State: ");
    private final JLabel stateStatus = new JLabel("STOPPED");

    private final JPanel periodPanel = new JPanel();
    private final JLabel periodLabel = new JLabel("Update Period: ");
    private final JTextField periodTextField = new JTextField();
    private final JLabel periodUnitsLabel = new JLabel("sec");

    private final JPanel mainPanel = new JPanel();

    private JLabel[] cmprPowerLabel;
    private JLabel[] cmprPowerStatus;
    private ButtonGroup[] cmprPowerBG;
    private JRadioButton[] cmprPowerOffRB;
    private JRadioButton[] cmprPowerOnRB;

    private final JLabel heaterPowerLabel = new JLabel("Heater Power: ");
    private final JLabel heaterPowerStatus = new JLabel("OFF");
    private final ButtonGroup heaterPowerBG = new ButtonGroup();
    private final JRadioButton heaterPowerOffRB = new JRadioButton("Off");
    private final JRadioButton heaterPowerOnRB = new JRadioButton("On");

    private final JLabel heaterControlLabel = new JLabel("Heater Control: ");
    private final JLabel heaterControlStatus = new JLabel("OFF");
    private final ButtonGroup heaterControlBG = new ButtonGroup();
    private final JRadioButton heaterControlOffRB = new JRadioButton("Off");
    private final JRadioButton heaterControlManualRB = new JRadioButton("Manual");
    private final JRadioButton heaterControlAutoRB = new JRadioButton("Auto");

    private final JPanel powerPanel = new JPanel();
    private final JTextField powerTextField = new JTextField();
    private final JLabel powerUnitsLabel = new JLabel("W");

    private final JPanel tempPanel = new JPanel();
    private final JTextField tempTextField = new JTextField();
    private final JLabel tempUnitsLabel = new JLabel("°C");

    private final int numCmpr = 1;

    
    public ColdControlPanel(CommandSender gui) {
        this.gui = gui;
        initComponents();
        periodTFX = new TextFieldX(periodTextField, TextFieldX.TYPE_DOUBLE);
        powerTFX = new TextFieldX(powerTextField, TextFieldX.TYPE_DOUBLE);
        tempTFX = new TextFieldX(tempTextField, TextFieldX.TYPE_DOUBLE);
        (new UpdateRefrigStatus(new RefrigState())).run();
        (new DisableSystem()).run();
    }

    private void initComponents() {

        stateLabel.setFont(FONT);
        stateStatus.setFont(FONT);
        Dimension d = stateStatus.getPreferredSize();
        d.width = 100;
        stateStatus.setMinimumSize(d);
        stateStatus.setPreferredSize(d);
        periodLabel.setFont(FONT);
        d = periodTextField.getPreferredSize();
        d.width = 45;
        periodTextField.setMinimumSize(d);
        periodTextField.setPreferredSize(d);
        periodTextField.setHorizontalAlignment(SwingConstants.CENTER);
        periodTextField.setText("...");
        periodTextField.addFocusListener(new FocusAdapter() {
            @Override
            public void focusLost(FocusEvent evt) {
                periodTFX.focusLost();
            }
        });
        periodTextField.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                Double value = (Double)periodTFX.action();
                if (value != null) {
                    gui.sendCommand(subsys, null, "setTickMillis", (int)(1000 * value));
                }
            }
        });
        periodTextField.addKeyListener(new KeyAdapter() {
            @Override
            public void keyTyped(KeyEvent evt) {
                periodTFX.keyTyped(evt.getKeyChar());
            }
        });
        periodUnitsLabel.setFont(FONT);

        cmprPowerLabel = new JLabel[numCmpr];
        cmprPowerStatus = new JLabel[numCmpr];
        cmprPowerBG = new ButtonGroup[numCmpr];
        cmprPowerOffRB = new JRadioButton[numCmpr];
        cmprPowerOnRB = new JRadioButton[numCmpr];
        for (int j = 0; j < numCmpr; j++) {
            String text = "Compressor " + (numCmpr == 1 ? "" : (j + 1) + " ") + "Power: ";
            cmprPowerLabel[j] = new JLabel(text);
            cmprPowerLabel[j].setFont(FONT);
            cmprPowerStatus[j] = new JLabel("OFF");
            cmprPowerStatus[j].setFont(FONT);
            cmprPowerOffRB[j] = new JRadioButton("Off");
            cmprPowerOffRB[j].setFont(FONT);
            cmprPowerOffRB[j].setFocusable(false);
            cmprPowerOffRB[j].addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent evt) {
                    setCmprPowerState(evt);
                }
            });
            cmprPowerOnRB[j] = new JRadioButton("On");
            cmprPowerOnRB[j].setFont(FONT);
            cmprPowerOnRB[j].setFocusable(false);
            cmprPowerOnRB[j].addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent evt) {
                    setCmprPowerState(evt);
                }
            });
            cmprPowerBG[j] = new ButtonGroup();
            cmprPowerBG[j].add(cmprPowerOffRB[j]);
            cmprPowerBG[j].add(cmprPowerOnRB[j]);
        }

        heaterPowerLabel.setFont(FONT);
        heaterPowerStatus.setFont(FONT);
        heaterPowerOffRB.setFont(FONT);
        heaterPowerOffRB.setFocusable(false);
        heaterPowerOffRB.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                setHeaterPowerState(0);
            }
        });
        heaterPowerOnRB.setFont(FONT);
        heaterPowerOnRB.setFocusable(false);
        heaterPowerOnRB.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                setHeaterPowerState(1);
            }
        });
        heaterPowerBG.add(heaterPowerOffRB);
        heaterPowerBG.add(heaterPowerOnRB);

        heaterControlLabel.setFont(FONT);
        heaterControlStatus.setFont(FONT);
        d = heaterControlStatus.getPreferredSize();
        d.width = 60;
        heaterControlStatus.setMinimumSize(d);
        heaterControlStatus.setPreferredSize(d);
        heaterControlOffRB.setFont(FONT);
        heaterControlOffRB.setFocusable(false);
        heaterControlOffRB.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                setHeaterControlState(0);
            }
        });
        heaterControlManualRB.setFont(FONT);
        heaterControlManualRB.setFocusable(false);
        heaterControlManualRB.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                setHeaterControlState(1);
            }
        });
        heaterControlAutoRB.setFont(FONT);
        heaterControlAutoRB.setFocusable(false);
        heaterControlAutoRB.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                setHeaterControlState(-1);
            }
        });
        heaterControlBG.add(heaterControlOffRB);
        heaterControlBG.add(heaterControlManualRB);
        heaterControlBG.add(heaterControlAutoRB);

        d = powerTextField.getPreferredSize();
        d.width = 50;
        powerTextField.setMinimumSize(d);
        powerTextField.setPreferredSize(d);
        powerTextField.setHorizontalAlignment(SwingConstants.CENTER);
        powerTextField.setText("...");
        powerTextField.addFocusListener(new FocusAdapter() {
            @Override
            public void focusLost(FocusEvent evt) {
                powerTFX.focusLost();
            }
        });
        powerTextField.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                Double value = (Double)powerTFX.action();
                if (value != null) {
                    gui.sendCommand(subsys, null, "setHeaterPower", value);
                }
            }
        });
        powerTextField.addKeyListener(new KeyAdapter() {
            @Override
            public void keyTyped(KeyEvent evt) {
                powerTFX.keyTyped(evt.getKeyChar());
            }
        });
        powerUnitsLabel.setFont(FONT);

        d = tempTextField.getPreferredSize();
        d.width = 60;
        tempTextField.setMinimumSize(d);
        tempTextField.setPreferredSize(d);
        tempTextField.setHorizontalAlignment(SwingConstants.CENTER);
        tempTextField.setText("...");
        tempTextField.addFocusListener(new FocusAdapter() {
            @Override
            public void focusLost(FocusEvent evt) {
                tempTFX.focusLost();
            }
        });
        tempTextField.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                Double value = (Double)tempTFX.action();
                if (value != null) {
                    gui.sendCommand(subsys, null, "setLoadTemp", value);
                }
            }
        });
        tempTextField.addKeyListener(new KeyAdapter() {
            @Override
            public void keyTyped(KeyEvent evt) {
                tempTFX.keyTyped(evt.getKeyChar());
            }
        });
        tempUnitsLabel.setFont(FONT);

        // Lay out state & period panels
        statePanel.add(stateLabel);
        statePanel.add(stateStatus);
        periodPanel.add(periodLabel);
        periodPanel.add(periodTextField);
        periodPanel.add(periodUnitsLabel);

        // Add state & period panels to main panel
        mainPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbm = new GridBagConstraints();
        gbm.insets = new Insets(5, 0, 5, 0);
        gbm.gridx = 0;
        gbm.gridy = 0;
        gbm.anchor = GridBagConstraints.WEST;
        gbm.gridwidth = 2;
        mainPanel.add(statePanel, gbm);
        gbm.anchor = GridBagConstraints.EAST;
        gbm.gridwidth = 3;
        gbm.gridx = 2;
        mainPanel.add(periodPanel, gbm);
        gbm.gridwidth = 1;
        gbm.gridx = 0;
        gbm.gridy++;

        // Add compressor power items to main panel
        gbm.anchor = GridBagConstraints.NORTHWEST;
        gbm.insets.bottom = 0;
        gbm.insets.left = 5;
        gbm.insets.right = 5;
        for (int j = 0; j < numCmpr; j++) {
            gbm.insets.top = 0;
            mainPanel.add(cmprPowerLabel[j], gbm);
            gbm.gridx++;
            mainPanel.add(cmprPowerStatus[j], gbm);
            gbm.gridx++;
            gbm.insets.top = -4;
            mainPanel.add(cmprPowerOffRB[j], gbm);
            gbm.gridx++;
            mainPanel.add(cmprPowerOnRB[j], gbm);
            gbm.gridx = 0;
            gbm.gridy++;
        }

        // Add heater power to main panel
        gbm.insets.top = 0;
        gbm.anchor = GridBagConstraints.NORTHEAST;
        mainPanel.add(heaterPowerLabel, gbm);
        gbm.gridx++;
        gbm.anchor = GridBagConstraints.NORTHWEST;
        mainPanel.add(heaterPowerStatus, gbm);
        gbm.gridx++;
        gbm.insets.top = -4;
        mainPanel.add(heaterPowerOffRB, gbm);
        gbm.gridx++;
        mainPanel.add(heaterPowerOnRB, gbm);
        gbm.gridx = 0;
        gbm.gridy++;

        // Add heater control to main panel
        gbm.insets.top = 0;
        gbm.anchor = GridBagConstraints.NORTHEAST;
        mainPanel.add(heaterControlLabel, gbm);
        gbm.gridx++;
        gbm.anchor = GridBagConstraints.NORTHWEST;
        mainPanel.add(heaterControlStatus, gbm);
        gbm.gridx++;
        gbm.insets.top = -4;
        mainPanel.add(heaterControlOffRB, gbm);
        gbm.gridx++;
        mainPanel.add(heaterControlManualRB, gbm);
        gbm.gridx++;
        mainPanel.add(heaterControlAutoRB, gbm);
        gbm.gridx = 0;
        gbm.gridy++;

        // Lay out power & temperature panels and add to main panel
        powerPanel.add(powerTextField);
        powerPanel.add(powerUnitsLabel);
        tempPanel.add(tempTextField);
        tempPanel.add(tempUnitsLabel);
        gbm.gridx = 3;
        mainPanel.add(powerPanel, gbm);
        gbm.gridx++;
        mainPanel.add(tempPanel, gbm);

        // Lay out the complete panel
        add(mainPanel);
    }

    @Override
    public void setSubsystem(String name) {
        subsys = name;
    }

    @Override
    public void updateControlPanel(RefrigState rs) {
        SwingUtilities.invokeLater(new UpdateRefrigStatus(rs));
    }

    @Override
    public void disableSystem() {
        SwingUtilities.invokeLater(new DisableSystem());
    }

    private void setCmprPowerState(ActionEvent evt) {
        for (int j = 0; j < numCmpr; j++) {
            int value = -1;
            if (evt.getSource().equals(cmprPowerOffRB[j])) {
                value = 0;
            }
            else if (evt.getSource().equals(cmprPowerOnRB[j])) {
                value = 1;
            }
            if (value < 0) continue;
            cmprPowerOffRB[j].setEnabled(false);
            cmprPowerOnRB[j].setEnabled(false);
            gui.sendCommand(subsys, null, "setCmprPowerEnable", j, value);
            return;
        }
    }

    private void setHeaterPowerState(int value) {
        heaterPowerOffRB.setEnabled(false);
        heaterPowerOnRB.setEnabled(false);
        gui.sendCommand(subsys, null, "setHeaterPowerEnable", value);
    }
    
    private void setHeaterControlState(int value) {
        heaterControlOffRB.setEnabled(false);
        heaterControlManualRB.setEnabled(false);
        heaterControlAutoRB.setEnabled(false);
        gui.sendCommand(subsys, null, "setHeaterControl", value);
    }
    
    class UpdateRefrigStatus implements Runnable {

        private final RefrigState rs;

        UpdateRefrigStatus(RefrigState rs) {
            this.rs = rs;
        }

        @Override
        public void run() {
            if (!enabled) {
                stateStatus.setText("RUNNING");
                stateStatus.setForeground(GREEN);
                for (int j = 0; j < numCmpr; j++) {
                    cmprPowerStatus[j].setEnabled(true);
                }
                heaterPowerStatus.setEnabled(true);
                heaterControlStatus.setEnabled(true);
                enabled = true;
            }

            for (int j = 0; j < numCmpr; j++) {
                int state = rs.getCmprPowerState(j);
                cmprPowerStatus[j].setText(state == RefrigState.POWER_ON ? "ON" :
                                           state == RefrigState.POWER_TRIPPED ? "TRIPPED" : "OFF");
                cmprPowerStatus[j].setForeground(state == RefrigState.POWER_ON ? GREEN :
                                                 state == RefrigState.POWER_TRIPPED ? RED : Color.black);
                JRadioButton selButton = state == RefrigState.POWER_ON ? cmprPowerOnRB[j] : cmprPowerOffRB[j];
                selButton.setSelected(true);
                cmprPowerOffRB[j].setEnabled(state == RefrigState.POWER_ON);
                cmprPowerOnRB[j].setEnabled(state == RefrigState.POWER_OFF);
            }
            
            int state = rs.getHeaterPowerState();
            heaterPowerStatus.setText(state == RefrigState.POWER_ON ? "ON" :
                                      state == RefrigState.POWER_TRIPPED ? "TRIPPED" : "OFF");
            heaterPowerStatus.setForeground(state == RefrigState.POWER_ON ? GREEN :
                                            state == RefrigState.POWER_TRIPPED ? RED : Color.black);
            JRadioButton selButton = state == RefrigState.POWER_ON ? heaterPowerOnRB : heaterPowerOffRB;
            selButton.setSelected(true);
            heaterPowerOffRB.setEnabled(state == RefrigState.POWER_ON);
            heaterPowerOnRB.setEnabled(state == RefrigState.POWER_OFF);

            state = rs.getHeaterControlState();
            heaterControlStatus.setText(state == RefrigState.CONTROL_ON ? "MANUAL" :
                                        state == RefrigState.CONTROL_AUTO ? "AUTO" : "OFF");
            heaterControlStatus.setForeground(state == RefrigState.CONTROL_ON ? GREEN :
                                              state == RefrigState.CONTROL_AUTO ? RED : Color.black);
            selButton = state == RefrigState.CONTROL_ON ? heaterControlManualRB :
                        state == RefrigState.CONTROL_AUTO ? heaterControlAutoRB : heaterControlOffRB;
            selButton.setSelected(true);
            heaterControlOffRB.setEnabled(selButton != heaterControlOffRB);
            heaterControlManualRB.setEnabled(selButton != heaterControlManualRB);
            heaterControlAutoRB.setEnabled(selButton != heaterControlAutoRB);

            periodTFX.update(rs.getTickMillis() / 1000.0, true);
            powerTFX.update(rs.getHeaterPower(), true);
            tempTFX.update(rs.getLoadTemp(), true);
        }

    }

    class DisableSystem implements Runnable {

        @Override
        public void run() {
            stateStatus.setText("STOPPED");
            stateStatus.setForeground(RED);
            for (int j = 0; j < numCmpr; j++) {
                cmprPowerStatus[j].setEnabled(false);
                cmprPowerOffRB[j].setEnabled(false);
                cmprPowerOnRB[j].setEnabled(false);
            }
            heaterPowerStatus.setEnabled(false);
            heaterPowerOffRB.setEnabled(false);
            heaterPowerOnRB.setEnabled(false);
            heaterControlStatus.setEnabled(false);
            heaterControlOffRB.setEnabled(false);
            heaterControlManualRB.setEnabled(false);
            heaterControlAutoRB.setEnabled(false);
            periodTFX.disable();
            powerTFX.disable();
            tempTFX.disable();
            enabled = false;
        }
 
    }

    private static final long serialVersionUID = 1L;
}
