#!/usr/local/bin/python

import MySQLdb
import datetime
import time
import readline
import os
import re
import sys
import ConfigParser


#****************************************************************************
#
#  Get command line parameters
#
#  dirname   The directory containing the initialization file
#
#  database  The database to use: "prod" or "test"
#
#  subsys    The subsystem to use, e.g. ccs-refrig-subscale
#
#****************************************************************************

if len(sys.argv) < 4:
    print "Too few parameters"
    exit()

dir = sys.argv[1]
database = sys.argv[2]
subsys = sys.argv[3]


#****************************************************************************
#
#  Get database parameters from initialization file
#
#****************************************************************************

cfg = ConfigParser.ConfigParser()
cfg.read(dir + "/refrigPlot.ini")
server = cfg.get(database, "server")
port = cfg.get(database, "port")
dbname = cfg.get(database, "dbname")
user = cfg.get(database, "user")
password = cfg.get(database, "password")


#****************************************************************************
#
#  Global variables initialization
#
#****************************************************************************

points = False
grid = False
namel = []
idl = []
pnl = []
pil = []
opl = []
ftl = []
ltl = []
start = None
end = None
rf = "/tmp/refrig_plot_" + str(os.getpid())
df = rf + ".dat"
cu = None


#****************************************************************************
#
#  Get console input
#
#  @param  prompt  The prompt to display
#
#  @return  The typed input, or None if EOF occurred (ctrl-D typed)
#
#****************************************************************************

def getInput(prompt):
    try:
        return raw_input(prompt)
    except:
        print
        return None


#****************************************************************************
#
#  Get the list of potential items to plot
#
#****************************************************************************

def getItems():
    global namel, idl
    count = cu.execute("select srcName,id from datadesc where srcSubsystem = '"
                       + subsys + "' order by srcName")
    data = cu.fetchall()
    for item in data:
        namel = namel + [item[0]]
        idl = idl + [item[1]]
    return


#****************************************************************************
#
#  Display the list of potential items to plot
#
#****************************************************************************

def showItems():
    print "Available items:"
    n = 0
    posn = 0
    cwidth = 17
    lwidth = 4 * cwidth
    for name in namel:
        item = "%3d: %s" % (n, name)
        size = cwidth * ((len(item) + cwidth) / cwidth)
        if posn + size > lwidth + 1:
            print
            posn = 0
        print "%-*s" % (size - 1, item),
        n = n + 1
        posn += size
    print


#****************************************************************************
#
#  Toggle the value of the points option
#
#****************************************************************************

def togglePoints():
    global points
    points =  not points
    if points:
        value = "enabled"
    else:
        value = "disabled"
    print "Points option", value


#****************************************************************************
#
#  Toggle the value of the grid option
#
#****************************************************************************

def toggleGrid():
    global grid
    grid = not grid
    if grid:
        value = "enabled"
    else:
        value = "disabled"
    print "Grid option", value


#****************************************************************************
#
#  Get a start or end time from the console
#
#  @param  prompt  The prompt to display
#
#  @param  start   The base time to use for time increments, or 0 if the
#                  current time is to be used.
#
#  @return  The entered time, as milliseconds since midnight January 1st,
#           1970 UT, or None if EOF occurred
#
#****************************************************************************

def getTime(prompt, start):
    while True:
        #
        #  Get typed input
        #
        tim = getInput(prompt)
        if tim == None: return None

        #
        #  Get +/- sign value and remove it along with leading/trailing blanks
        #
        tmo = re.match("^ *([-+]?) *([^ ]+) *([^ ]*) *(.*)$", tim)
        if tmo == None: continue
        tm = tmo.group(1, 2, 3, 4)
        if tm[3] != "":
            print "Invalid time"
            continue
        pfx = 0
        if tm[0] != "":
            if tm[0] == "-": pfx = -1
            else: pfx = 1
        tim = tm[1]
        if tm[2] != "": tim = tim + " " + tm[2]

        #
        #  Process value with no sign ([YYYY-[MM-[DD ]]]HH:MM)
        #
        if pfx == 0:
            try:
                ts = time.strptime(tim, "%Y-%m-%d %H:%M")
            except:
                try:
                    ts = time.strptime(time.strftime("%Y-") + tim,
                                       "%Y-%m-%d %H:%M")
                except:
                    try:
                        ts = time.strptime(time.strftime("%Y-%m-") + tim,
                                           "%Y-%m-%d %H:%M")
                    except:
                        try:
                            ts = time.strptime(time.strftime("%Y-%m-%d ") + tim,
                                               "%Y-%m-%d %H:%M")
                        except:
                            print "Invalid time"
                            continue
            return 1000L * int(time.mktime(ts))

        #
        #  Process signed value
        #
        okay = (start != 0 or pfx == -1)
        tmo = re.match("^(\d+) (\d+):(\d+)$", tim)
        if tmo != None:
            td = tmo.group(1, 2, 3)
            tm = 60 * (int(td[2]) + 60 * (int(td[1]) + 24 * int(td[0])))
        else:
            tmo = re.match("^(\d+):(\d+)$", tim)
            if tmo != None:
                td = tmo.group(1, 2)
                tm = 60 * (int(td[1]) + 60 * (int(td[0])))
            else:
                tmo = re.match("^(\d+)$", tim)
                if tmo != None:
                    tm = 60 * int(tmo.group(1))
                else:
                    okay = False
        if okay:
            if pfx == -1: return 1000L * (int(time.time()) - tm)
            return start + 1000L * tm
        print "Invalid time"


#****************************************************************************
#
#  Get the names of the items to plot from the console, along with the
#  start and end times of the interval of interest
#
#  @return  0: if information obtained
#          -1: if EOF occurred
#
#****************************************************************************

def getPlots():
    global pnl, pil, opl, start, end
    start = None
    end = None
    while start == None:
        reply = getInput("Item numbers (or ?, q, p, g): ")
        if reply == None or re.match("^ *q *$", reply) != None:
            return -1
        if re.match("^ *\? *$", reply) != None:
            showItems()
            continue
        if re.match("^ *p *$", reply) != None:
            togglePoints()
            continue
        if re.match("^ *g *$", reply) != None:
            toggleGrid()
            continue
        pls = reply.split()
        if len(pls) == 0: continue
        pnl = []
        pil = []
        opl = []
        icount = 0
        ocount = 0
        for pl in pls:
            op = None
            while pl != "":
                mo = re.match("^([+-]?)([^+-]*)(.*$)", pl)
#                if mo == None: break
                sign = mo.group(1)
                if op == None:
                    op = 0
                else:
                    if sign == "+": op = 1
                    if sign == "-": op = -1
                pi = mo.group(2)
                icount = icount + 1
                try:
                    index = int(pi)
                except:
                    print "Invalid integer"
                    break
                if index < 0 or index >= len(namel):
                    print "Item number out of range"
                    break
                pnl = pnl + [namel[index]]
                pil = pil + [idl[index]]
                opl = opl + [op]
                ocount = ocount + 1
                pl = mo.group(3)
            if ocount != icount: break
        if ocount != icount: continue
        while end == None:
            start = getTime("Start time ([YYYY-[MM-]DD ]]]HH:MM "
                            + "| -[DD [HH:]]MM): ", 0)
            if start == None: break
            end = getTime("End time ([YYYY-[MM-[DD ]]]HH:MM "
                          + "| {+|-}[DD [HH:]]MM): ", start)
    return 0


#****************************************************************************
#
#  Get the action to perform after a plot
#
#  @return  1: redraw the plot
#           0: save the plot in a file
#          -1: no action
#
#****************************************************************************

def getAction():
    reply = getInput("Action (s = save, p = !points, g = !grid, b = !both): ")
    if reply == None:
        return -1
    if re.match("^ *s *$", reply) != None:
        return 0
    if re.match("^ *p *$", reply) != None:
        togglePoints()
        return 1
    if re.match("^ *g *$", reply) != None:
        toggleGrid()
        return 1
    if re.match("^ *b *$", reply) != None:
        togglePoints()
        toggleGrid()
        return 1
    return -1


#****************************************************************************
#
#  Get the specified data
#
#  @param  id     The id of the data item to get
#
#  @param  start  The start time for the interval
#
#  @param  end    The end time for the interval
#
#  @return  The list of data items, or None if the name doesn't exist. or
#           there is no data in the specified time interval.
#
#****************************************************************************

def getData(id, start, end):
    count = cu.execute("select tstampmills,doubleData from rawdata where "
                       + "descr_id = " + str(id) + " and tstampmills >= "
                       + str(start) + " and tstampmills <= " + str(end))
#                       + " order by tstampmills")
    if count == 0:
        print "No matching data found"
        return None
    return cu.fetchall()


#****************************************************************************
#
#  Generate the Gnuplot input data file for a plot
#
#  @param  ix  The index of the item to plot
#
#  @return  0: if file generated successfully
#           1: if no data found
#
#****************************************************************************

def genData(ix):
    global ftl, ltl
    data = getData(pil[ix], start, end)
    if data == None:
        return 1
    print len(data), "values found for", pnl[ix]
    try:
        out = open(df + str(ix), "w")
    except:
        print "Cannot open output data file"
        return 1
    to = -1
    ct = 0
    for item in data:
        ct = item[0] / 1000.0
        if to == -1:
            ftl = ftl + [ct]
            if time.localtime(ct)[8] == 0:
                to = time.timezone
            else:
                to = time.altzone
        out.write(str(ct - to) + " " + str(item[1]) + "\n")
    out.close()
    if ct != 0:
        ltl = ltl + [ct]
    return 0


#****************************************************************************
#
#  Display a plot of one or more (overlaid) items
#
#  @param  ixl  The list of the item indices to plot
#
#  @return  The plotter used to display the plot.  This is to be closed when
#           the plot is no longer needed.
#
#****************************************************************************

def showPlot(ixl):
    gp = os.popen("gnuplot -geometry 1200x900 -bg white", "w")
    sep = 'set title "' + subsys + ': '
    for ix in ixl:
        gp.write(sep + pnl[abs(ix) - 1])
        sep = ' + '
    gp.write('" font "helvetica,25"\n')
    gp.write('set term x11 font "helvetica,20,,medium"\n')
    gp.write('set xdata time\n')
    gp.write('set timefmt "%s"\n')
    lft = time.localtime(ftl[ixl[0] - 1])
    llt = time.localtime(ltl[ixl[0] - 1])
    if lft[0] != llt[0]:
        fmt = '%H:%M\\n%b %d\\n%Y'
    elif lft[2] != llt[2]:
        fmt = '%H:%M\\n%b %d'
    else:
        fmt = '%H:%M'
    gp.write('set xtics format "' + fmt + '"\n')
    if lft[0] == llt[0]:
        if lft[2] == llt[2]:
            fmt = '%b %d, %Y'
        else:
            fmt = '%Y'
        gp.write('set xlabel "' + time.strftime(fmt, lft) + '"\n')
    if points:
        gp.write('set style data points\n')
    else:
        gp.write('set style data lines\n')
    if grid:
        gp.write('set grid xtics ytics\n')
#    gp.write('set ytics format "%.2f"\n')
    y2 = False
    for ix in ixl:
        if ix < 0: y2 = True
    if y2:
        gp.write('set ytics nomirror\n')
#        gp.write('set y2tics nomirror format "%.2f"\n')
        gp.write('set y2tics nomirror\n')
    gp.write('unset mouse\n')
    if len(ixl) == 1:
        gp.write('set key off\n')
    else:
        gp.write('set key on\n')
    sep = 'plot'
    for ix in ixl:
        gp.write(sep + ' "' + df + str(abs(ix) - 1) + '" using 1:2 title "'
                 + pnl[abs(ix) - 1] + '"')
        if ix < 0:
            gp.write(' axes x1y2')
        sep = ','
    gp.write('\n')
    gp.flush()
    return gp


#****************************************************************************
#
#  Save, to a file, a plot of one or more (overlaid) items
#
#  @param  ixl  The list of the item indices to plot
#
#****************************************************************************

def savePlot(ixl):
    y2 = False
    for ix in ixl:
        if ix < 0: y2 = True
    gp = os.popen("GDFONTPATH=/usr/share/fonts/default/Type1/ "
                    + "gnuplot -bg white", "w")
    sep = 'set title "' + subsys + ': '
    for ix in ixl:
        gp.write(sep + pnl[abs(ix) - 1])
        sep = ' + '
    gp.write('" font "b018012l,18"\n')  ## Bookman
    gp.write('set terminal png size 1200,900 font "b018012l" 12\n')
    sep = 'set output "./'
    for ix in ixl:
        gp.write(sep + pnl[abs(ix) - 1])
        sep = '_'
    gp.write('_' + time.strftime("%Y%m%d_%H%M%S") + '.png"\n')
    gp.write('set lmargin 10\n')
    if y2:
        gp.write('set rmargin 10\n')
    else:
        gp.write('set rmargin 4\n')
    gp.write('set tmargin 4\n')
    gp.write('set bmargin 4\n')
    gp.write('set xdata time\n')
    gp.write('set timefmt "%s"\n')
    lft = time.localtime(ftl[ixl[0] - 1])
    llt = time.localtime(ltl[ixl[0] - 1])
    if lft[0] != llt[0]:
        fmt = '%H:%M\\n%b %d\\n%Y'
    elif lft[2] != llt[2]:
        fmt = '%H:%M\\n%b %d'
    else:
        fmt = '%H:%M'
    gp.write('set xtics format "' + fmt + '"\n')
    if lft[0] == llt[0]:
        if lft[2] == llt[2]:
            fmt = '%b %d, %Y'
        else:
            fmt = '%Y'
        gp.write('set xlabel "' + time.strftime(fmt, lft) + '"\n')
    if points:
        gp.write('set style data points\n')
    else:
        gp.write('set style data lines\n')
    if grid:
        gp.write('set grid xtics ytics\n')
#    gp.write('set ytics format "%.2f"\n')
    if y2:
        gp.write('set ytics nomirror\n')
#        gp.write('set y2tics nomirror format "%.2f"\n')
        gp.write('set y2tics nomirror\n')
    if len(ixl) == 1:
        gp.write('set key off\n')
    else:
        gp.write('set key on\n')
    sep = 'plot'
    for ix in ixl:
        gp.write(sep + ' "' + df + str(abs(ix) - 1) + '" using 1:2 title "'
                 + pnl[abs(ix) - 1] + '"')
        if ix < 0:
            gp.write(' axes x1y2')
        sep = ','
    gp.write('\n')
    gp.close()


#****************************************************************************
#
#  Main code
#
#****************************************************************************

first = True

while True:
    db = MySQLdb.connect(server, user, password, dbname, port=int(port))
    cu = db.cursor()

    if first:
        getItems()
        showItems()
        first = False

    if getPlots() != 0: break

    ftl = []
    ltl = []
    gil = []
    for ix in range(len(pnl)):
        if genData(ix) == 0:
            gil = gil + [True]
        else:
            gil = gil + [False]

    action = 1
    while action >= 0:
        gpl = []

        if action > 0:
            ixl = []
            for ix in range(len(pnl)):
                if opl[ix] == 0: ixl = []
                if gil[ix]:
                    if opl[ix] >= 0:
                        ixl = ixl + [ix + 1]
                    else:
                        ixl = ixl + [-(ix + 1)]
                if (ix + 1 >= len(pnl) or opl[ix + 1] == 0) and len(ixl) != 0:
                    gpl = gpl + [showPlot(ixl)]

        action = getAction()
        for gp in gpl: gp.close()

        if action == 0:
            ixl = []
            for ix in range(len(pnl)):
                if opl[ix] == 0: ixl = []
                if gil[ix]:
                    if opl[ix] >= 0:
                        ixl = ixl + [ix + 1]
                    else:
                        ixl = ixl + [-(ix + 1)]
                if (ix + 1>= len(pnl) or opl[ix + 1] == 0) and len(ixl) != 0:
                    savePlot(ixl)

    db.close()
