package org.lsst.ccs.subsystem.refrig.ui;

import java.io.PrintWriter;
import java.util.Timer;
import java.util.TimerTask;
import javax.swing.JComponent;
import javax.swing.JFrame;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.Agent;
import org.lsst.ccs.bus.AgentPresenceListener;
import org.lsst.ccs.bus.BusMessage;
import org.lsst.ccs.bus.Command;
import org.lsst.ccs.bus.DataStatusListener;
import org.lsst.ccs.bus.ModuleInvokerCommand;
import org.lsst.ccs.bus.utils.SynchronousCommandAgent;
import org.lsst.ccs.subsystem.monitor.ui.CommandSender;
import org.lsst.ccs.subsystem.monitor.ui.MonitorAssembly;
import org.lsst.ccs.subsystem.refrig.data.RefrigFullState;
import org.lsst.ccs.subsystem.refrig.data.RefrigState;

/**
 ******************************************************************************
 **
 **  Generalized refrigeration GUI.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class RefrigGUI extends Subsystem
                       implements CommandSender, DataStatusListener,
                                  AgentPresenceListener {

    private final String subsys;
    private final SynchronousCommandAgent ca = new SynchronousCommandAgent(fac);
    private final RefrigAssembly assembly;
    private final MonitorAssembly mon;
    private final RefrigControlPanel panel;
    private PrintWriter out = new PrintWriter(System.out, true);
    private long lastHeartBeat = 0;
    private boolean enabled = false;

    public RefrigGUI(String name) throws Exception {
        super("Refrig" + name + "GUI", Agent.AgentType.CONSOLE);
        assembly = new RefrigAssembly(name, this);
        mon = assembly.getMonitor();
        panel = assembly.getControlPanel();
        subsys = "ccs-refrig-" + name.toLowerCase();
    }

    public void initGui() {
        mon.setSubsystems(subsys);
        setListenToStatus(true);
        setStatusBroadcastPeriod(0);
        getAgentPresenceManager().addAgentPresenceListener(this);
        start();

        new Timer("CheckAlive", true).scheduleAtFixedRate(new TimerTask() {
            @Override
            public void run() {
                if (System.currentTimeMillis() - lastHeartBeat > 10000) {
                    disable();
                }
            }
        }, 0, 5000);
    }

    public JComponent getGuiLayout() {
        return assembly;
    }

    public void resetGui() {
    }

    /**
     * @param args the command line arguments
     * @throws  Exception
     */
    public static void main(String[] args) throws Exception {
        if (args.length == 0) {
            throw new Exception("No Refrigeration control panel name supplied");
        }
        String name = args[0];
        RefrigGUI t = new RefrigGUI(name);
        JFrame frame = new JFrame("Refrigeration " + name + " System");
        frame.setContentPane(t.getGuiLayout());
        frame.pack();
        frame.setVisible(true);
        t.initGui();
        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
    }

    public void setPrintWriter(PrintWriter out) {
        this.out = out;
    }

    @Override
    public void onDataArrival(String source, long time, String key, Object d) {
        if (!source.equals(subsys)) return;
        if (key.equals(RefrigState.KEY) && d instanceof RefrigState) {
            log.debug("RefrigGui received RefrigState message");
            panel.updateControlPanel(((RefrigState)d));
        }
        else {
            mon.handleData(source, key, d);
        }
    }

    @Override
    public void onStatus(BusMessage s) {
        if (!s.getOrigin().equals(subsys)) return;
        lastHeartBeat = s.getTimeStamp();
        enable();
        mon.handleStatus(s);
    }

    @Override
    public void connecting(Agent agent) {
        if (agent.getName().equals(subsys)) {
            enable();
        }
    }

    @Override
    public void disconnecting(Agent agent) {
        if (agent.getName().equals(subsys)) {
            disable();
        }
    }

    @Override
    public String sendCommand(String dest, String target, String cmnd,
                              Object... args) {
        sendCmnd(dest, target, cmnd, args);
        return "";
    }

    private void enable() {
        if (enabled) return;
        enabled = true;
        Object reply = sendCmnd(null, null, "getFullState");
        if (reply instanceof RefrigFullState) {
            log.debug("RefrigGui received RefrigFullState reply");
            RefrigFullState r = (RefrigFullState)reply;
            mon.updateTableModel(subsys, r.getMonitorFullState());
            panel.updateControlPanel(r.getRefrigState());
        }
        else {
            enabled = false;
        }
    }

    private void disable() {
        if (!enabled) return;
        enabled = false;
        panel.disableSystem();
        mon.disableSystem();
    }

    private Object sendCmnd(String dest, String target, String cmnd,
                            Object... args) {
        String dst = (dest == null ? subsys : dest)
                       + (target == null ? "" : "/" + target);
        Command cmd = new ModuleInvokerCommand("", 0, dst, cmnd, args);
        try {
            return ca.invoke(cmd, 5000);
        }
        catch (Exception e) {
            return null;
        }
    }

}
