package org.lsst.ccs.subsystem.refrig.ui;

import java.io.PrintWriter;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;
import javax.swing.JComponent;
import javax.swing.JFrame;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.Agent;
import org.lsst.ccs.bus.BusMessage;
import org.lsst.ccs.bus.Command;
import org.lsst.ccs.bus.CommandReply;
import org.lsst.ccs.bus.DataStatusListener;
import org.lsst.ccs.bus.ModuleInvokerCommand;
import org.lsst.ccs.subsystem.monitor.data.MonitorFullState;
import org.lsst.ccs.subsystem.monitor.ui.CommandSender;
import org.lsst.ccs.subsystem.monitor.ui.MonitorAssembly;
import org.lsst.ccs.subsystem.refrig.data.RefrigFullState;
import org.lsst.ccs.subsystem.refrig.data.RefrigState;

/**
 *
 * @author max
 */
public class MultiGUI extends Subsystem
                      implements CommandSender, DataStatusListener {

    private static final String LL_DEST = "ccs-refrig-longlines",
                                SS_DEST = "ccs-refrig-subscale";
    private final MultiAssembly assembly;
    private final MonitorAssembly mon;
    private final MultiControlPanel panel;
    private final Map<String, SubsysDesc> subsysMap = new HashMap<>();
    private PrintWriter out = new PrintWriter(System.out, true);

    class SubsysDesc {

        final String name;
        long lastHeartBeat = 0, lastStateRequest = 0;
        boolean enabled = false;
        String stateCorrelId;
    
        SubsysDesc(String name) {
            this.name = name;
        }

    }

    public MultiGUI() {
        super("RefrigGUI", Agent.AgentType.CONSOLE);
        assembly = new MultiAssembly(this);
        mon = assembly.getMonitor();
        panel = assembly.getControlPanel();
        subsysMap.put(LL_DEST, new SubsysDesc(LL_DEST));
        subsysMap.put(SS_DEST, new SubsysDesc(SS_DEST));
    }

    public void initGui() {
        mon.setSubsystems(SS_DEST, LL_DEST);
        setListenToStatus(true);
        setStatusBroadcastPeriod(0);
        start();

        Timer t = new Timer();
        t.scheduleAtFixedRate(new TimerTask() {

            @Override
            public void run() {
                for (SubsysDesc desc : subsysMap.values()) {
                    long delta = (System.currentTimeMillis()
                                    - desc.lastHeartBeat) / 1000;
                    if (desc.enabled && delta > 10) {
                        desc.enabled = false;
                        panel.disableSystem(desc.name);
                    }
                }
            }
        }, new Date(), 5000);
    }

    public JComponent getGuiLayout() {
        return assembly;
    }

    public void resetGui() {
    }

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        MultiGUI t = new MultiGUI();
        JFrame frame = new JFrame("Refrigeration Subsystem");
        frame.setContentPane(t.getGuiLayout());
        frame.pack();
        frame.setVisible(true);
        t.initGui();
        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
    }

    public void setPrintWriter(PrintWriter out) {
        this.out = out;
    }

    @Override
    public void onReply(CommandReply s) {
        SubsysDesc desc = subsysMap.get(s.getOrigin());
        if (desc == null) return;
        if (s.getCorrelId().equals(desc.stateCorrelId)) {
            RefrigFullState r = (RefrigFullState)s.getReply();
            log.debug("RefrigGui received RefrigFullState reply");
            MonitorFullState m = r.getMonitorFullState();
            mon.updateTableModel(desc.name, m);
            panel.updateControlPanel(desc.name, r.getRefrigState());
            desc.enabled = true;
        }
    }

    @Override
    public void onDataArrival(String source, long time, String key, Object d) {
        SubsysDesc desc = subsysMap.get(source);
        if (desc == null) return;
        if (key.equals(RefrigState.KEY) && d instanceof RefrigState) {
            log.debug("RefrigGui received RefrigState message from " + source);
            panel.updateControlPanel(source, (RefrigState)d);
        }
        else {
            mon.handleData(source, key, d);
        }
}

    @Override
    public void onStatus(BusMessage s) {
        SubsysDesc desc = subsysMap.get(s.getOrigin());
        if (desc == null) return;

        desc.lastHeartBeat = s.getTimeStamp();

        if (!desc.enabled) {
            long currentTime = System.currentTimeMillis();
            if (currentTime - desc.lastStateRequest > 2000) {
                desc.lastStateRequest = currentTime;
                log.debug("RefrigGui sending getFullState command");
                desc.stateCorrelId = sendCommand(desc.name, null, "getFullState");
            }
            return;
        }

        mon.handleStatus(s);
    }

    @Override
    public String sendCommand(String dest, String target, String cmnd,
                              Object... args) {
        String dst = dest + (target == null ? "" : "/" + target);
        Command cmd = new ModuleInvokerCommand("", 0, dst, cmnd, args);
        fac.sendCommand(cmd);
        return cmd.getCorrelId();
    }

}
