package org.lsst.ccs.subsystem.refrig.ui;

import java.io.PrintWriter;
import java.util.Date;
import java.util.Timer;
import java.util.TimerTask;
import javax.swing.JComponent;
import javax.swing.JFrame;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.BusMessage;
import org.lsst.ccs.bus.CommandReply;
import org.lsst.ccs.bus.MetadataStatus;
import org.lsst.ccs.bus.ModuleInvokerCommand;
import org.lsst.ccs.bus.TrendingStatus;
import org.lsst.ccs.subsystem.monitor.data.MonitorFullState;
import org.lsst.ccs.subsystem.monitor.data.MonitorState;
import org.lsst.ccs.subsystem.monitor.status.MonitorStateStatus;
import org.lsst.ccs.subsystem.monitor.ui.CommandSender;
import org.lsst.ccs.subsystem.monitor.ui.MonitorTrendingTable;
import org.lsst.ccs.subsystem.refrig.data.RefrigFullState;
import org.lsst.ccs.subsystem.refrig.status.RefrigStateStatus;
import org.lsst.ccs.utilities.logging.Logger;
import org.lsst.ccs.utilities.tracers.Names;

/**
 *
 * @author max
 */
public class LongLinesGUI extends Subsystem implements CommandSender {

    private static final String refrig_dest = "ccs-refrig-longlines";
    private PrintWriter out = new PrintWriter(System.out, true);
    private LongLinesAssembly assembly;
    private MonitorTrendingTable table;
    private LongLinesControlPanel panel;
    private long lastHeartBeat = 0, lastStateRequest = 0;
    private boolean enabled = false;
    String stateCorrelId;

    public LongLinesGUI() {
        assembly = new LongLinesAssembly(this);
        table = assembly.getTrendingTable();
        panel = assembly.getControlPanel();
    }

    public void initGui() {
        Logger.configure();     // Should not be necessary
        setName(Names.almostUniqueAgentName("RefrigLongLinesGUI"));
        setListenToStatus(true);
        setStatusBroadcastPeriod(0);
        start();

        Timer t = new Timer();
        t.scheduleAtFixedRate(new TimerTask() {

            @Override
            public void run() {
                long delta = (System.currentTimeMillis() - lastHeartBeat) / 1000;
                if (enabled && delta > 10) {
                    enabled = false;
                    panel.disableSystem();
                }
            }
        }, new Date(), 5000);
    }

    public JComponent getGuiLayout() {
        return assembly;
    }

    public void resetGui() {
    }

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) throws ClassNotFoundException
    {
        LongLinesGUI t = new LongLinesGUI();
        JFrame frame = new JFrame("Refrigeration Long Lines Subsystem");
        frame.setContentPane(t.getGuiLayout());
        frame.pack();
        frame.setVisible(true);
        t.initGui();
        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
    }

    public void setPrintWriter(PrintWriter out) {
        this.out = out;
    }

    @Override
    public void onReply(CommandReply s) {
        //FIXME temporary fix to prevent multiple GUIs from interfering with each other.
        //See https://jira.slac.stanford.edu/browse/LSSTCCS-9
        if (s.getCorrelId().equals(stateCorrelId)) {
            RefrigFullState r = (RefrigFullState)s.getReply();
            log.debug("LongLinesGui received RefrigFullState reply");
            MonitorFullState m = r.getMonitorFullState();
            table.updateTableModel(m);
            panel.updateControlPanel(r.getRefrigState(), m.getMonitorState());
            panel.updateTime();
            enabled = true;
        }
    }

    @Override
    public void onStatus(BusMessage s) {
        if (!s.getOrigin().equals(refrig_dest)) return;

        lastHeartBeat = s.getTimeStamp();

        if (!enabled) {
            long currentTime = System.currentTimeMillis();
            if (currentTime - lastStateRequest > 2000) {
                lastStateRequest = currentTime;
                log.debug("LongLinesGui sending getFullState command");
                ModuleInvokerCommand
                    cmd = new ModuleInvokerCommand("getFullState");
                sendCommand(null, cmd);
                stateCorrelId = cmd.getCorrelId();
            }
            return;
        }

        if (s instanceof TrendingStatus) {
            log.debug("LongLinesGui received TrendingStatus message");
            table.updateTableValue((TrendingStatus)s);
            panel.updateTime();
        }
        else if (s instanceof MetadataStatus) {
            log.debug("LongLinesGui received MetadataStatus message");
            table.updateTableValue((MetadataStatus)s);
        }
        else if (s instanceof MonitorStateStatus) {
            log.debug("LongLinesGui received MonitorStateStatus message");
            MonitorState m = ((MonitorStateStatus)s).getMonitorState();
            table.updateTableValue(m);
            panel.updateControlPanel(null, m);
        }
        else if (s instanceof RefrigStateStatus) {
            log.debug("LongLinesGui received RefrigStateStatus message");
            panel.updateControlPanel(((RefrigStateStatus)s).getRefrigState(), null);
        }
    }

    @Override
    public void sendCommand(String target, ModuleInvokerCommand cmd) {
        String dest = target == null ? refrig_dest : refrig_dest + "/" + target;
        cmd.setDestination(dest);
        updateCurrentSubsystem();
        fac.sendCommand(cmd);
    }

}
